/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*  This file is part of gnome-spell bonobo component

    Copyright (C) 2001 Ximian, Inc.
    Authors:      Radek Doulik <rodo@ximian.com>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
    Boston, MA 02111-1307, USA.
*/

#include <config.h>
#include <glib.h>
#include <gtk/gtkclist.h>
#include <gtk/gtklabel.h>
#include <gtk/gtkwindow.h>
#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-i18n.h>
#include <glade/glade.h>
#include <bonobo/bonobo-control.h>
#include <bonobo/bonobo-exception.h>

#include "Spell.h"
#include "control.h"

typedef struct {
	BonoboControl *control;
	BonoboObjectClient *dict_client;
	GNOME_Spell_Dictionary dict;
	BonoboPropertyBag *pb;

	gchar *language;
	gchar *word;
	gint row;

	GtkWidget *label_word;
	GtkWidget *clist_suggestions;

	GtkWidget *button_replace;
	GtkWidget *button_add;
	GtkWidget *button_ignore;
	GtkWidget *button_skip;

} SpellControlData;

enum {
	PROP_SPELL_WORD,
	PROP_SPELL_REPLACE,
	PROP_SPELL_ADD,
	PROP_SPELL_IGNORE,
	PROP_SPELL_SKIP,
	PROP_SPELL_LANGUAGE,
} SpellControlProps;

static void
control_get_prop (BonoboPropertyBag *bag,
		  BonoboArg         *arg,
		  guint              arg_id,
		  CORBA_Environment *ev,
		  gpointer           user_data)
{
	SpellControlData *cd = (SpellControlData *) user_data;

	switch (arg_id) {
	case PROP_SPELL_LANGUAGE:
		BONOBO_ARG_SET_STRING (arg, cd->language);
		break;
	default:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
		break;
	}
}

static void
clicked_replace (GtkWidget *w, SpellControlData *cd)
{
	BonoboArg *arg;
	gchar *replacement;

	if (gtk_clist_get_text (GTK_CLIST (cd->clist_suggestions), cd->row, 0, &replacement)) {
		/* printf ("set replace\n"); */
		CORBA_Environment ev;

		CORBA_exception_init (&ev);
		GNOME_Spell_Dictionary_setCorrection (cd->dict, cd->word, replacement, &ev);
		CORBA_exception_free (&ev);

		arg = bonobo_arg_new (BONOBO_ARG_STRING);
		BONOBO_ARG_SET_STRING (arg, replacement);
		bonobo_property_bag_notify_listeners (cd->pb, "replace", arg, NULL);
		CORBA_free (arg);
	}
}

static void
row_selected (GtkWidget *w, gint row, gint column, GdkEvent *event, SpellControlData *cd)
{
	cd->row = row;

	if (event && event->type == GDK_2BUTTON_PRESS)
		clicked_replace (cd->button_replace, cd);
}

static void
clicked_add (GtkWidget *w, SpellControlData *cd)
{
	BonoboArg *arg;
	CORBA_Environment ev;

	/* printf ("set add\n"); */
	CORBA_exception_init (&ev);
	GNOME_Spell_Dictionary_addWordToPersonal (cd->dict, cd->word, &ev);
	CORBA_exception_free (&ev);

	arg = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
	BONOBO_ARG_SET_BOOLEAN (arg, TRUE);
	bonobo_property_bag_notify_listeners (cd->pb, "add", arg, NULL);
	CORBA_free (arg);
}

static void
clicked_ignore (GtkWidget *w, SpellControlData *cd)
{
	CORBA_Environment ev;
	BonoboArg *arg;

	/* printf ("set ignore\n"); */
	CORBA_exception_init (&ev);
	GNOME_Spell_Dictionary_addWordToSession (cd->dict, cd->word, &ev);
	CORBA_exception_free (&ev);

	arg = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
	BONOBO_ARG_SET_BOOLEAN (arg, TRUE);
	bonobo_property_bag_notify_listeners (cd->pb, "ignore", arg, NULL);
	CORBA_free (arg);
}

static void
clicked_skip (GtkWidget *w, SpellControlData *cd)
{
	BonoboArg *arg;

	/* printf ("set skip\n"); */
	arg = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
	BONOBO_ARG_SET_BOOLEAN (arg, TRUE);
	bonobo_property_bag_notify_listeners (cd->pb, "skip", arg, NULL);
	CORBA_free (arg);
}

static void
set_word (SpellControlData *cd, gchar *word)
{
	GNOME_Spell_StringSeq *seq;
	CORBA_Environment ev;
	gchar *str, *text [1];
	gint i;

	str = g_strdup_printf ("Suggestions for '%s'", word);
	gtk_label_set_text (GTK_LABEL (cd->label_word), str);
	g_free (str);

	g_free (cd->word);
	cd->word = g_strdup (word);

	/* printf ("set_word %s\n", word); */
	CORBA_exception_init (&ev);
	seq = GNOME_Spell_Dictionary_getSuggestions (cd->dict, word, &ev);
	CORBA_exception_free (&ev);
	if (seq) {
		gtk_clist_freeze (GTK_CLIST (cd->clist_suggestions));
		gtk_clist_clear (GTK_CLIST (cd->clist_suggestions));
		for (i = 0; i < seq->_length; i ++) {
			/* printf ("\t%s\n", seq->_buffer [i]); */
			text [0] = seq->_buffer [i];
			gtk_clist_append (GTK_CLIST (cd->clist_suggestions), text);
		}
		gtk_clist_select_row (GTK_CLIST (cd->clist_suggestions), 0, 0);
		gtk_clist_thaw (GTK_CLIST (cd->clist_suggestions));
		gtk_widget_grab_focus (cd->clist_suggestions);

		CORBA_free (seq);
	}
}

static void
set_language (SpellControlData *cd, gchar *language)
{
	CORBA_Environment ev;

	g_free (cd->language);
	cd->language = g_strdup (language);

	CORBA_exception_init (&ev);
	GNOME_Spell_Dictionary_setTag (cd->dict, "language-tag", cd->language, &ev);
	CORBA_exception_free (&ev);
}

static void
control_set_prop (BonoboPropertyBag *bag,
		  const BonoboArg   *arg,
		  guint              arg_id,
		  CORBA_Environment *ev,
		  gpointer           user_data)
{
	SpellControlData *cd = user_data;
	
	switch (arg_id) {
	case PROP_SPELL_WORD:
		set_word (cd, BONOBO_ARG_GET_STRING (arg));
		break;
	case PROP_SPELL_LANGUAGE:
		set_language (cd, BONOBO_ARG_GET_STRING (arg));
		break;
	default:
		bonobo_exception_set (ev, ex_Bonobo_PropertyBag_NotFound);
		break;
	}
}

static void
spell_control_construct (BonoboControl *control, GtkWidget *table, GladeXML *xml)
{
	BonoboArg          *def;
	SpellControlData   *cd;
	CORBA_Environment ev;

	cd = g_new0 (SpellControlData, 1);
	cd->control = control;

	cd->label_word = glade_xml_get_widget (xml, "label_word");
	cd->clist_suggestions = glade_xml_get_widget (xml, "clist_suggestions");

	gtk_signal_connect (GTK_OBJECT (cd->clist_suggestions), "select_row", row_selected, cd);

	cd->button_replace = glade_xml_get_widget (xml, "button_replace");
	cd->button_add     = glade_xml_get_widget (xml, "button_add");
	cd->button_ignore  = glade_xml_get_widget (xml, "button_ignore");
	cd->button_skip    = glade_xml_get_widget (xml, "button_skip");

	gtk_signal_connect (GTK_OBJECT (cd->button_replace), "clicked", clicked_replace, cd);
	gtk_signal_connect (GTK_OBJECT (cd->button_add), "clicked", clicked_add, cd);
	gtk_signal_connect (GTK_OBJECT (cd->button_ignore), "clicked", clicked_ignore, cd);
	gtk_signal_connect (GTK_OBJECT (cd->button_skip), "clicked", clicked_skip, cd);

	/* PropertyBag */
	cd->pb = bonobo_property_bag_new (control_get_prop, control_set_prop, cd);
	bonobo_control_set_properties (control, cd->pb);

	bonobo_property_bag_add (cd->pb, "word", PROP_SPELL_WORD, BONOBO_ARG_STRING, NULL,
				 "checked word",  BONOBO_PROPERTY_WRITEABLE);
	bonobo_property_bag_add (cd->pb, "language", PROP_SPELL_REPLACE, BONOBO_ARG_STRING, NULL,
				 "dictionary language", BONOBO_PROPERTY_WRITEABLE);

	def = bonobo_arg_new (BONOBO_ARG_STRING);
	BONOBO_ARG_SET_STRING (def, "replace default value");
	bonobo_property_bag_add (cd->pb, "replace", PROP_SPELL_REPLACE, BONOBO_ARG_STRING, def,
				 "replacement to replace word",  BONOBO_PROPERTY_READABLE);
	CORBA_free (def);

	def = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
	BONOBO_ARG_SET_BOOLEAN (def, TRUE);
	bonobo_property_bag_add (cd->pb, "add", PROP_SPELL_REPLACE, BONOBO_ARG_BOOLEAN, def,
				 "add word to dictionary",  BONOBO_PROPERTY_READABLE);
	CORBA_free (def);

	def = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
	BONOBO_ARG_SET_BOOLEAN (def, TRUE);
	bonobo_property_bag_add (cd->pb, "ignore", PROP_SPELL_REPLACE, BONOBO_ARG_BOOLEAN, def,
				 "add word to session dictionary",  BONOBO_PROPERTY_READABLE);
	CORBA_free (def);

	def = bonobo_arg_new (BONOBO_ARG_BOOLEAN);
	BONOBO_ARG_SET_BOOLEAN (def, TRUE);
	bonobo_property_bag_add (cd->pb, "skip", PROP_SPELL_REPLACE, BONOBO_ARG_BOOLEAN, def,
				 "skip this word",  BONOBO_PROPERTY_READABLE);
	CORBA_free (def);

	/* dictionary */
	cd->dict_client = bonobo_object_activate ("OAFIID:GNOME_Spell_Dictionary:" API_VERSION, 0);
	cd->dict = BONOBO_OBJREF (cd->dict_client);

	CORBA_exception_init (&ev);
	GNOME_Spell_Dictionary__set_active (cd->dict, CORBA_TRUE, &ev);
	CORBA_exception_free (&ev);
	set_language (cd, "en");
}

BonoboObject *
gnome_spell_control_new ()
{
	BonoboControl *control;
	GtkWidget *table;
	GladeXML *xml;

	xml = glade_xml_new (GLADE_DATADIR "/spell-checker.glade", "simple_control");
	if (!xml)
		g_error (_("Could not load glade file."));
	table = glade_xml_get_widget (xml, "simple_control");

	control = bonobo_control_new (table);

	if (control){
		spell_control_construct (control, table, xml);
		return BONOBO_OBJECT (control);
	} else {
		gtk_widget_unref (table);
		return NULL;
	}
}
