/* GNOME Scan - Scan as easy as you print
 * Copyright © 2006-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GNOME Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * GNOME Scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GNOME Scan. If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#include <gnome-scan-job.h>
#include <gegl.h>
#include <stdlib.h>
#include <string.h>
#include <gnome-scan-node.h>




struct _GnomeScanJobPrivate {
	GnomeScanStatus _status;
	GnomeScanScanner* _scanner;
	GnomeScanSink* _sink;
	double _progress;
	gboolean cancelled;
	GeglNode* load_buffer;
	GSList* graph;
	GeglNode* gegl_sink;
	GeglProcessor* processor;
	GSList* nodes;
};

#define GNOME_SCAN_JOB_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_SCAN_TYPE_JOB, GnomeScanJobPrivate))
enum  {
	GNOME_SCAN_JOB_DUMMY_PROPERTY,
	GNOME_SCAN_JOB_STATUS,
	GNOME_SCAN_JOB_SCANNER,
	GNOME_SCAN_JOB_SINK,
	GNOME_SCAN_JOB_PROGRESS
};
static void _g_slist_free_g_object_unref (GSList* self);
static void _gnome_scan_job_check_nodes_status_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self);
static void _gnome_scan_job_build_graph_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self);
static void gnome_scan_job_register_node (GnomeScanJob* self, GnomeScanNode* node, GnomeScanNode* old);
static void gnome_scan_job_build_graph (GnomeScanJob* self);
static void gnome_scan_job_check_nodes_status (GnomeScanJob* self);
static GObject * gnome_scan_job_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer gnome_scan_job_parent_class = NULL;
static void gnome_scan_job_finalize (GObject* obj);



static void _g_slist_free_g_object_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) g_object_unref, NULL);
	g_slist_free (self);
}


GnomeScanJob* gnome_scan_job_construct (GType object_type, GnomeScanSink* sink) {
	GnomeScanJob * self;
	g_return_val_if_fail (sink != NULL, NULL);
	self = g_object_newv (object_type, 0, NULL);
	gnome_scan_job_set_sink (self, sink);
	return self;
}


GnomeScanJob* gnome_scan_job_new (GnomeScanSink* sink) {
	return gnome_scan_job_construct (GNOME_SCAN_TYPE_JOB, sink);
}


gboolean gnome_scan_job_is_running (GnomeScanJob* self) {
	g_return_val_if_fail (self != NULL, FALSE);
	return self->priv->_status == GNOME_SCAN_STATUS_PROCESSING;
}


void gnome_scan_job_run (GnomeScanJob* self) {
	gint count;
	g_return_if_fail (self != NULL);
	self->priv->cancelled = FALSE;
	count = 0;
	if (!gnome_scan_job_is_running (self)) {
		{
			GSList* node_collection;
			GSList* node_it;
			node_collection = self->priv->nodes;
			for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
				GnomeScanNode* _tmp0;
				GnomeScanNode* node;
				_tmp0 = NULL;
				node = (_tmp0 = (GnomeScanNode*) node_it->data, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
				{
					gnome_scan_node_start_scan (node);
					(node == NULL) ? NULL : (node = (g_object_unref (node), NULL));
				}
			}
		}
		gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_PROCESSING);
	}
	while (gnome_scan_job_run_once (self)) {
		count++;
	}
	g_debug ("gnome-scan-job.vala:96: %d images acquired", count);
	gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_DONE);
}


void gnome_scan_job_end (GnomeScanJob* self) {
	g_return_if_fail (self != NULL);
	if (!gnome_scan_job_is_running (self)) {
		return;
	}
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->nodes;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GnomeScanNode* _tmp0;
			GnomeScanNode* node;
			_tmp0 = NULL;
			node = (_tmp0 = (GnomeScanNode*) node_it->data, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
			{
				gnome_scan_node_end_scan (node);
				(node == NULL) ? NULL : (node = (g_object_unref (node), NULL));
			}
		}
	}
	gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_READY);
}


void gnome_scan_job_cancel (GnomeScanJob* self) {
	g_return_if_fail (self != NULL);
	self->priv->cancelled = TRUE;
}


gboolean gnome_scan_job_run_once (GnomeScanJob* self) {
	double progress;
	gboolean scanner_is_node;
	GeglProcessor* _tmp3;
	g_return_val_if_fail (self != NULL, FALSE);
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->nodes;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GnomeScanNode* node;
			node = (GnomeScanNode*) node_it->data;
			{
				gboolean _tmp0;
				_tmp0 = FALSE;
				if (!gnome_scan_node_start_image (node)) {
					_tmp0 = GNOME_SCAN_IS_SCANNER (node);
				} else {
					_tmp0 = FALSE;
				}
				if (_tmp0) {
					return FALSE;
				}
			}
		}
	}
	progress = 0.0;
	scanner_is_node = g_slist_length (gnome_scan_node_get_nodes ((GnomeScanNode*) self->priv->_scanner)) == 0;
	if (scanner_is_node) {
		while (TRUE) {
			gboolean _tmp2;
			_tmp2 = FALSE;
			if (!self->priv->cancelled) {
				_tmp2 = gnome_scan_scanner_work (self->priv->_scanner, &progress);
			} else {
				_tmp2 = FALSE;
			}
			if (!_tmp2) {
				break;
			}
			self->priv->_progress = progress / 3.0;
		}
		gegl_node_set (self->priv->load_buffer, "operation", "gegl:load-buffer", "buffer", gnome_scan_scanner_get_buffer (self->priv->_scanner), NULL);
	}
	_tmp3 = NULL;
	self->priv->processor = (_tmp3 = gegl_node_new_processor (self->priv->gegl_sink, NULL), (self->priv->processor == NULL) ? NULL : (self->priv->processor = (g_object_unref (self->priv->processor), NULL)), _tmp3);
	while (TRUE) {
		gboolean _tmp4;
		_tmp4 = FALSE;
		if (!self->priv->cancelled) {
			_tmp4 = gegl_processor_work (self->priv->processor, &progress);
		} else {
			_tmp4 = FALSE;
		}
		if (!_tmp4) {
			break;
		}
		if (scanner_is_node) {
			self->priv->_progress = 0.333 + ((progress * 2) / 3);
		} else {
			self->priv->_progress = progress;
		}
	}
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->nodes;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GnomeScanNode* node;
			node = (GnomeScanNode*) node_it->data;
			{
				gnome_scan_node_end_image (node);
			}
		}
	}
	return TRUE;
}


static void _gnome_scan_job_check_nodes_status_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self) {
	gnome_scan_job_check_nodes_status (self);
}


static void _gnome_scan_job_build_graph_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self) {
	gnome_scan_job_build_graph (self);
}


static void gnome_scan_job_register_node (GnomeScanJob* self, GnomeScanNode* node, GnomeScanNode* old) {
	g_return_if_fail (self != NULL);
	if (old != NULL) {
		GQuark _tmp1;
		guint _tmp0;
		GQuark _tmp3;
		guint _tmp2;
		g_signal_handlers_disconnect_matched ((GObject*) old, G_SIGNAL_MATCH_ID | G_SIGNAL_MATCH_DETAIL | G_SIGNAL_MATCH_FUNC | G_SIGNAL_MATCH_DATA, (g_signal_parse_name ("notify::status", G_TYPE_OBJECT, &_tmp0, &_tmp1, FALSE), _tmp0), 0, NULL, (GCallback) _gnome_scan_job_check_nodes_status_g_object_notify, self);
		g_signal_handlers_disconnect_matched ((GObject*) old, G_SIGNAL_MATCH_ID | G_SIGNAL_MATCH_DETAIL | G_SIGNAL_MATCH_FUNC | G_SIGNAL_MATCH_DATA, (g_signal_parse_name ("notify::graph", G_TYPE_OBJECT, &_tmp2, &_tmp3, FALSE), _tmp2), 0, NULL, (GCallback) _gnome_scan_job_build_graph_g_object_notify, self);
		self->priv->nodes = g_slist_remove (self->priv->nodes, old);
	}
	if (node != NULL) {
		g_signal_connect_object ((GObject*) node, "notify::status", (GCallback) _gnome_scan_job_check_nodes_status_g_object_notify, self, 0);
		g_signal_connect_object ((GObject*) node, "notify::graph", (GCallback) _gnome_scan_job_build_graph_g_object_notify, self, 0);
		if (GNOME_SCAN_IS_SCANNER (node)) {
			GnomeScanNode* _tmp4;
			_tmp4 = NULL;
			self->priv->nodes = g_slist_prepend (self->priv->nodes, (_tmp4 = node, (_tmp4 == NULL) ? NULL : g_object_ref (_tmp4)));
		} else {
			if (GNOME_SCAN_IS_SINK (node)) {
				GnomeScanNode* _tmp5;
				_tmp5 = NULL;
				self->priv->nodes = g_slist_append (self->priv->nodes, (_tmp5 = node, (_tmp5 == NULL) ? NULL : g_object_ref (_tmp5)));
			} else {
				GSList* last;
				last = g_slist_last (self->priv->nodes);
				if (GNOME_SCAN_IS_SINK (last->data)) {
					GnomeScanNode* _tmp6;
					_tmp6 = NULL;
					self->priv->nodes = g_slist_insert_before (self->priv->nodes, last, (_tmp6 = node, (_tmp6 == NULL) ? NULL : g_object_ref (_tmp6)));
				} else {
					GnomeScanNode* _tmp7;
					_tmp7 = NULL;
					self->priv->nodes = g_slist_append (self->priv->nodes, (_tmp7 = node, (_tmp7 == NULL) ? NULL : g_object_ref (_tmp7)));
				}
			}
		}
	}
	gnome_scan_job_check_nodes_status (self);
	gnome_scan_job_build_graph (self);
}


static void gnome_scan_job_build_graph (GnomeScanJob* self) {
	GeglNode* prev;
	GSList* _tmp1;
	GeglNode* _tmp6;
	GeglNode* _tmp5;
	g_return_if_fail (self != NULL);
	if (((gint) self->priv->_status) < ((gint) GNOME_SCAN_STATUS_READY)) {
		return;
	}
	prev = NULL;
	/* remove all children from current graph*/
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->graph;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GeglNode* _tmp0;
			GeglNode* node;
			_tmp0 = NULL;
			node = (_tmp0 = (GeglNode*) node_it->data, (_tmp0 == NULL) ? NULL : g_object_ref (_tmp0));
			{
				if (prev != NULL) {
					gegl_node_disconnect (node, "input");
				}
				prev = node;
				(node == NULL) ? NULL : (node = (g_object_unref (node), NULL));
			}
		}
	}
	_tmp1 = NULL;
	self->priv->graph = (_tmp1 = NULL, (self->priv->graph == NULL) ? NULL : (self->priv->graph = (_g_slist_free_g_object_unref (self->priv->graph), NULL)), _tmp1);
	/* list all nodes*/
	prev = NULL;
	if (g_slist_length (gnome_scan_node_get_nodes ((GnomeScanNode*) self->priv->_scanner)) == 0) {
		GeglNode* _tmp2;
		_tmp2 = NULL;
		self->priv->load_buffer = (_tmp2 = gegl_node_new (), (self->priv->load_buffer == NULL) ? NULL : (self->priv->load_buffer = (g_object_unref (self->priv->load_buffer), NULL)), _tmp2);
		gegl_node_set (self->priv->load_buffer, "operation", "gegl:load-buffer", NULL);
		prev = self->priv->load_buffer;
	} else {
		GeglNode* _tmp3;
		_tmp3 = NULL;
		self->priv->load_buffer = (_tmp3 = NULL, (self->priv->load_buffer == NULL) ? NULL : (self->priv->load_buffer = (g_object_unref (self->priv->load_buffer), NULL)), _tmp3);
	}
	{
		GSList* gsnode_collection;
		GSList* gsnode_it;
		gsnode_collection = self->priv->nodes;
		for (gsnode_it = gsnode_collection; gsnode_it != NULL; gsnode_it = gsnode_it->next) {
			GnomeScanNode* gsnode;
			gsnode = (GnomeScanNode*) gsnode_it->data;
			{
				{
					GSList* gnode_collection;
					GSList* gnode_it;
					gnode_collection = gnome_scan_node_get_nodes (gsnode);
					for (gnode_it = gnode_collection; gnode_it != NULL; gnode_it = gnode_it->next) {
						GeglNode* gnode;
						gnode = (GeglNode*) gnode_it->data;
						{
							GeglNode* _tmp4;
							_tmp4 = NULL;
							self->priv->graph = g_slist_append (self->priv->graph, (_tmp4 = gnode, (_tmp4 == NULL) ? NULL : g_object_ref (_tmp4)));
							if (prev != NULL) {
								gegl_node_link (prev, gnode);
							}
							prev = gnode;
						}
					}
				}
			}
		}
	}
	_tmp6 = NULL;
	_tmp5 = NULL;
	self->priv->gegl_sink = (_tmp6 = (_tmp5 = prev, (_tmp5 == NULL) ? NULL : g_object_ref (_tmp5)), (self->priv->gegl_sink == NULL) ? NULL : (self->priv->gegl_sink = (g_object_unref (self->priv->gegl_sink), NULL)), _tmp6);
}


static void gnome_scan_job_check_nodes_status (GnomeScanJob* self) {
	GnomeScanStatus status;
	gboolean _tmp0;
	gboolean _tmp1;
	g_return_if_fail (self != NULL);
	status = 0;
	_tmp0 = FALSE;
	_tmp1 = FALSE;
	if (self->priv->_scanner == NULL) {
		_tmp1 = TRUE;
	} else {
		_tmp1 = self->priv->_sink == NULL;
	}
	if (_tmp1) {
		_tmp0 = TRUE;
	} else {
		_tmp0 = g_slist_length (gnome_scan_node_get_nodes ((GnomeScanNode*) self->priv->_sink)) == 0;
	}
	if (_tmp0) {
		gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_UNCONFIGURED);
	} else {
		gboolean _tmp2;
		gboolean _tmp3;
		gboolean _tmp4;
		gboolean redo_graph;
		status = GNOME_SCAN_STATUS_READY;
		_tmp2 = FALSE;
		if (self->priv->_scanner != NULL) {
			_tmp2 = ((gint) gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_scanner)) < ((gint) status);
		} else {
			_tmp2 = FALSE;
		}
		if (_tmp2) {
			status = gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_scanner);
		}
		_tmp3 = FALSE;
		if (self->priv->_sink != NULL) {
			_tmp3 = ((gint) gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_sink)) < ((gint) status);
		} else {
			_tmp3 = FALSE;
		}
		if (_tmp3) {
			status = gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_sink);
		}
		_tmp4 = FALSE;
		if (((gint) status) > ((gint) self->priv->_status)) {
			_tmp4 = status == GNOME_SCAN_STATUS_READY;
		} else {
			_tmp4 = FALSE;
		}
		redo_graph = _tmp4;
		gnome_scan_job_set_status (self, status);
		/* auto cancel job is status is not ready (failure,
		 etc.)*/
		if (((gint) status) < ((gint) GNOME_SCAN_STATUS_READY)) {
			self->priv->cancelled = TRUE;
		}
		if (redo_graph) {
			gnome_scan_job_build_graph (self);
		}
	}
}


GnomeScanStatus gnome_scan_job_get_status (GnomeScanJob* self) {
	g_return_val_if_fail (self != NULL, 0);
	return self->priv->_status;
}


void gnome_scan_job_set_status (GnomeScanJob* self, GnomeScanStatus value) {
	g_return_if_fail (self != NULL);
	self->priv->_status = value;
	g_object_notify ((GObject *) self, "status");
}


GnomeScanScanner* gnome_scan_job_get_scanner (GnomeScanJob* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_scanner;
}


void gnome_scan_job_set_scanner (GnomeScanJob* self, GnomeScanScanner* value) {
	GnomeScanScanner* _tmp1;
	GnomeScanScanner* old;
	GnomeScanScanner* _tmp3;
	GnomeScanScanner* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	old = (_tmp1 = self->priv->_scanner, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_scanner = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (self->priv->_scanner == NULL) ? NULL : (self->priv->_scanner = (g_object_unref (self->priv->_scanner), NULL)), _tmp3);
	gnome_scan_job_register_node (self, (GnomeScanNode*) value, (GnomeScanNode*) old);
	(old == NULL) ? NULL : (old = (g_object_unref (old), NULL));
	g_object_notify ((GObject *) self, "scanner");
}


GnomeScanSink* gnome_scan_job_get_sink (GnomeScanJob* self) {
	g_return_val_if_fail (self != NULL, NULL);
	return self->priv->_sink;
}


void gnome_scan_job_set_sink (GnomeScanJob* self, GnomeScanSink* value) {
	GnomeScanSink* _tmp1;
	GnomeScanSink* old;
	GnomeScanSink* _tmp3;
	GnomeScanSink* _tmp2;
	g_return_if_fail (self != NULL);
	_tmp1 = NULL;
	old = (_tmp1 = self->priv->_sink, (_tmp1 == NULL) ? NULL : g_object_ref (_tmp1));
	_tmp3 = NULL;
	_tmp2 = NULL;
	self->priv->_sink = (_tmp3 = (_tmp2 = value, (_tmp2 == NULL) ? NULL : g_object_ref (_tmp2)), (self->priv->_sink == NULL) ? NULL : (self->priv->_sink = (g_object_unref (self->priv->_sink), NULL)), _tmp3);
	gnome_scan_job_register_node (self, (GnomeScanNode*) value, (GnomeScanNode*) old);
	(old == NULL) ? NULL : (old = (g_object_unref (old), NULL));
	g_object_notify ((GObject *) self, "sink");
}


double gnome_scan_job_get_progress (GnomeScanJob* self) {
	g_return_val_if_fail (self != NULL, 0.0);
	return self->priv->_progress;
}


static GObject * gnome_scan_job_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties) {
	GObject * obj;
	GnomeScanJobClass * klass;
	GObjectClass * parent_class;
	GnomeScanJob * self;
	klass = GNOME_SCAN_JOB_CLASS (g_type_class_peek (GNOME_SCAN_TYPE_JOB));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = GNOME_SCAN_JOB (obj);
	{
		GSList* _tmp0;
		_tmp0 = NULL;
		self->priv->graph = (_tmp0 = NULL, (self->priv->graph == NULL) ? NULL : (self->priv->graph = (_g_slist_free_g_object_unref (self->priv->graph), NULL)), _tmp0);
	}
	return obj;
}


static void gnome_scan_job_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GnomeScanJob * self;
	gpointer boxed;
	self = GNOME_SCAN_JOB (object);
	switch (property_id) {
		case GNOME_SCAN_JOB_STATUS:
		g_value_set_enum (value, gnome_scan_job_get_status (self));
		break;
		case GNOME_SCAN_JOB_SCANNER:
		g_value_set_object (value, gnome_scan_job_get_scanner (self));
		break;
		case GNOME_SCAN_JOB_SINK:
		g_value_set_object (value, gnome_scan_job_get_sink (self));
		break;
		case GNOME_SCAN_JOB_PROGRESS:
		g_value_set_double (value, gnome_scan_job_get_progress (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gnome_scan_job_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GnomeScanJob * self;
	self = GNOME_SCAN_JOB (object);
	switch (property_id) {
		case GNOME_SCAN_JOB_STATUS:
		gnome_scan_job_set_status (self, g_value_get_enum (value));
		break;
		case GNOME_SCAN_JOB_SCANNER:
		gnome_scan_job_set_scanner (self, g_value_get_object (value));
		break;
		case GNOME_SCAN_JOB_SINK:
		gnome_scan_job_set_sink (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gnome_scan_job_class_init (GnomeScanJobClass * klass) {
	gnome_scan_job_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GnomeScanJobPrivate));
	G_OBJECT_CLASS (klass)->get_property = gnome_scan_job_get_property;
	G_OBJECT_CLASS (klass)->set_property = gnome_scan_job_set_property;
	G_OBJECT_CLASS (klass)->constructor = gnome_scan_job_constructor;
	G_OBJECT_CLASS (klass)->finalize = gnome_scan_job_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_STATUS, g_param_spec_enum ("status", "status", "status", GNOME_SCAN_TYPE_STATUS, GNOME_SCAN_STATUS_INITIALIZING, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_SCANNER, g_param_spec_object ("scanner", "scanner", "scanner", GNOME_SCAN_TYPE_SCANNER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_SINK, g_param_spec_object ("sink", "sink", "sink", GNOME_SCAN_TYPE_SINK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_PROGRESS, g_param_spec_double ("progress", "progress", "progress", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void gnome_scan_job_instance_init (GnomeScanJob * self) {
	self->priv = GNOME_SCAN_JOB_GET_PRIVATE (self);
	self->priv->_status = GNOME_SCAN_STATUS_INITIALIZING;
	self->priv->_progress = (double) 0;
	self->priv->cancelled = FALSE;
	self->priv->processor = NULL;
	self->priv->nodes = NULL;
}


static void gnome_scan_job_finalize (GObject* obj) {
	GnomeScanJob * self;
	self = GNOME_SCAN_JOB (obj);
	(self->priv->_scanner == NULL) ? NULL : (self->priv->_scanner = (g_object_unref (self->priv->_scanner), NULL));
	(self->priv->_sink == NULL) ? NULL : (self->priv->_sink = (g_object_unref (self->priv->_sink), NULL));
	(self->priv->load_buffer == NULL) ? NULL : (self->priv->load_buffer = (g_object_unref (self->priv->load_buffer), NULL));
	(self->priv->graph == NULL) ? NULL : (self->priv->graph = (_g_slist_free_g_object_unref (self->priv->graph), NULL));
	(self->priv->gegl_sink == NULL) ? NULL : (self->priv->gegl_sink = (g_object_unref (self->priv->gegl_sink), NULL));
	(self->priv->processor == NULL) ? NULL : (self->priv->processor = (g_object_unref (self->priv->processor), NULL));
	(self->priv->nodes == NULL) ? NULL : (self->priv->nodes = (_g_slist_free_g_object_unref (self->priv->nodes), NULL));
	G_OBJECT_CLASS (gnome_scan_job_parent_class)->finalize (obj);
}


GType gnome_scan_job_get_type (void) {
	static GType gnome_scan_job_type_id = 0;
	if (gnome_scan_job_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GnomeScanJobClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gnome_scan_job_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GnomeScanJob), 0, (GInstanceInitFunc) gnome_scan_job_instance_init, NULL };
		gnome_scan_job_type_id = g_type_register_static (G_TYPE_OBJECT, "GnomeScanJob", &g_define_type_info, 0);
	}
	return gnome_scan_job_type_id;
}




