/* GNOME Scan - Scan as easy as you print
 * Copyright © 2006-2008  Étienne Bersac <bersace@gnome.org>
 *
 * GNOME Scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * GNOME Scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with GNOME Scan. If not, write to:
 *
 *	the Free Software Foundation, Inc.
 *	51 Franklin Street, Fifth Floor
 *	Boston, MA 02110-1301, USA
 */

#include <glib.h>
#include <glib-object.h>
#include <float.h>
#include <math.h>
#include <gegl.h>
#include <stdlib.h>
#include <string.h>


#define GNOME_SCAN_TYPE_JOB (gnome_scan_job_get_type ())
#define GNOME_SCAN_JOB(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_JOB, GnomeScanJob))
#define GNOME_SCAN_JOB_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_JOB, GnomeScanJobClass))
#define GNOME_SCAN_IS_JOB(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_JOB))
#define GNOME_SCAN_IS_JOB_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_JOB))
#define GNOME_SCAN_JOB_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_JOB, GnomeScanJobClass))

typedef struct _GnomeScanJob GnomeScanJob;
typedef struct _GnomeScanJobClass GnomeScanJobClass;
typedef struct _GnomeScanJobPrivate GnomeScanJobPrivate;

#define GNOME_SCAN_TYPE_STATUS (gnome_scan_status_get_type ())

#define GNOME_SCAN_TYPE_NODE (gnome_scan_node_get_type ())
#define GNOME_SCAN_NODE(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_NODE, GnomeScanNode))
#define GNOME_SCAN_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_NODE, GnomeScanNodeClass))
#define GNOME_SCAN_IS_NODE(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_NODE))
#define GNOME_SCAN_IS_NODE_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_NODE))
#define GNOME_SCAN_NODE_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_NODE, GnomeScanNodeClass))

typedef struct _GnomeScanNode GnomeScanNode;
typedef struct _GnomeScanNodeClass GnomeScanNodeClass;

#define GNOME_SCAN_TYPE_SCANNER (gnome_scan_scanner_get_type ())
#define GNOME_SCAN_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_SCANNER, GnomeScanScanner))
#define GNOME_SCAN_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_SCANNER, GnomeScanScannerClass))
#define GNOME_SCAN_IS_SCANNER(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_SCANNER))
#define GNOME_SCAN_IS_SCANNER_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_SCANNER))
#define GNOME_SCAN_SCANNER_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_SCANNER, GnomeScanScannerClass))

typedef struct _GnomeScanScanner GnomeScanScanner;
typedef struct _GnomeScanScannerClass GnomeScanScannerClass;

#define GNOME_SCAN_TYPE_SINK (gnome_scan_sink_get_type ())
#define GNOME_SCAN_SINK(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_SINK, GnomeScanSink))
#define GNOME_SCAN_SINK_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_SINK, GnomeScanSinkClass))
#define GNOME_SCAN_IS_SINK(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_SINK))
#define GNOME_SCAN_IS_SINK_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_SINK))
#define GNOME_SCAN_SINK_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_SINK, GnomeScanSinkClass))

typedef struct _GnomeScanSink GnomeScanSink;
typedef struct _GnomeScanSinkClass GnomeScanSinkClass;

#define GNOME_SCAN_TYPE_PRESELECTION (gnome_scan_preselection_get_type ())
#define GNOME_SCAN_PRESELECTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_PRESELECTION, GnomeScanPreselection))
#define GNOME_SCAN_PRESELECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_PRESELECTION, GnomeScanPreselectionClass))
#define GNOME_SCAN_IS_PRESELECTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_PRESELECTION))
#define GNOME_SCAN_IS_PRESELECTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_PRESELECTION))
#define GNOME_SCAN_PRESELECTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_PRESELECTION, GnomeScanPreselectionClass))

typedef struct _GnomeScanPreselection GnomeScanPreselection;
typedef struct _GnomeScanPreselectionClass GnomeScanPreselectionClass;
#define _g_object_unref0(var) ((var == NULL) ? NULL : (var = (g_object_unref (var), NULL)))
#define __g_slist_free_g_object_unref0(var) ((var == NULL) ? NULL : (var = (_g_slist_free_g_object_unref (var), NULL)))

#define GNOME_SCAN_TYPE_OPTION (gnome_scan_option_get_type ())
#define GNOME_SCAN_OPTION(obj) (G_TYPE_CHECK_INSTANCE_CAST ((obj), GNOME_SCAN_TYPE_OPTION, GnomeScanOption))
#define GNOME_SCAN_OPTION_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), GNOME_SCAN_TYPE_OPTION, GnomeScanOptionClass))
#define GNOME_SCAN_IS_OPTION(obj) (G_TYPE_CHECK_INSTANCE_TYPE ((obj), GNOME_SCAN_TYPE_OPTION))
#define GNOME_SCAN_IS_OPTION_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), GNOME_SCAN_TYPE_OPTION))
#define GNOME_SCAN_OPTION_GET_CLASS(obj) (G_TYPE_INSTANCE_GET_CLASS ((obj), GNOME_SCAN_TYPE_OPTION, GnomeScanOptionClass))

typedef struct _GnomeScanOption GnomeScanOption;
typedef struct _GnomeScanOptionClass GnomeScanOptionClass;

struct _GnomeScanJob {
	GObject parent_instance;
	GnomeScanJobPrivate * priv;
};

struct _GnomeScanJobClass {
	GObjectClass parent_class;
};

typedef enum  {
	GNOME_SCAN_STATUS_UNKNOWN = 0,
	GNOME_SCAN_STATUS_FAILED = 100,
	GNOME_SCAN_STATUS_INITIALIZING = 200,
	GNOME_SCAN_STATUS_BUSY = 300,
	GNOME_SCAN_STATUS_UNCONFIGURED = 400,
	GNOME_SCAN_STATUS_READY = 500,
	GNOME_SCAN_STATUS_PROCESSING = 600,
	GNOME_SCAN_STATUS_DONE = 700
} GnomeScanStatus;

struct _GnomeScanJobPrivate {
	GnomeScanStatus _status;
	GnomeScanScanner* _scanner;
	GnomeScanSink* _sink;
	double _progress;
	GSList* preselections;
	gboolean cancelled;
	GeglNode* load_buffer;
	GSList* graph;
	GeglNode* gegl_sink;
	GeglProcessor* processor;
	GSList* nodes;
};


static gpointer gnome_scan_job_parent_class = NULL;

GType gnome_scan_job_get_type (void);
GType gnome_scan_status_get_type (void);
GType gnome_scan_node_get_type (void);
GType gnome_scan_scanner_get_type (void);
GType gnome_scan_sink_get_type (void);
GType gnome_scan_preselection_get_type (void);
#define GNOME_SCAN_JOB_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), GNOME_SCAN_TYPE_JOB, GnomeScanJobPrivate))
enum  {
	GNOME_SCAN_JOB_DUMMY_PROPERTY,
	GNOME_SCAN_JOB_STATUS,
	GNOME_SCAN_JOB_SCANNER,
	GNOME_SCAN_JOB_SINK,
	GNOME_SCAN_JOB_PROGRESS
};
static void _g_slist_free_g_object_unref (GSList* self);
void gnome_scan_job_set_sink (GnomeScanJob* self, GnomeScanSink* value);
GnomeScanJob* gnome_scan_job_new (GnomeScanSink* sink);
GnomeScanJob* gnome_scan_job_construct (GType object_type, GnomeScanSink* sink);
void gnome_scan_job_register_preselection (GnomeScanJob* self, GnomeScanPreselection* preselection);
gboolean gnome_scan_job_is_running (GnomeScanJob* self);
void gnome_scan_node_start_scan (GnomeScanNode* self);
void gnome_scan_job_set_status (GnomeScanJob* self, GnomeScanStatus value);
gboolean gnome_scan_job_run_once (GnomeScanJob* self);
void gnome_scan_job_run (GnomeScanJob* self);
void gnome_scan_node_end_scan (GnomeScanNode* self);
void gnome_scan_job_end (GnomeScanJob* self);
void gnome_scan_job_cancel (GnomeScanJob* self);
gboolean gnome_scan_node_start_image (GnomeScanNode* self);
GSList* gnome_scan_node_get_nodes (GnomeScanNode* self);
gboolean gnome_scan_scanner_work (GnomeScanScanner* self, double* progress);
GeglBuffer* gnome_scan_scanner_get_buffer (GnomeScanScanner* self);
void gnome_scan_node_end_image (GnomeScanNode* self);
static void gnome_scan_job_check_nodes_status (GnomeScanJob* self);
static void _gnome_scan_job_check_nodes_status_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self);
static void gnome_scan_job_build_graph (GnomeScanJob* self);
static void _gnome_scan_job_build_graph_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self);
static void gnome_scan_job_apply_preselections (GnomeScanJob* self, GnomeScanNode* node);
static void gnome_scan_job_register_node (GnomeScanJob* self, GnomeScanNode* node, GnomeScanNode* old);
GType gnome_scan_option_get_type (void);
GnomeScanOption* gnome_scan_node_lookup_option (GnomeScanNode* self, const char* name);
const char* gnome_scan_preselection_get_option_name (GnomeScanPreselection* self);
void gnome_scan_preselection_apply (GnomeScanPreselection* self, GnomeScanOption* option);
GnomeScanStatus gnome_scan_node_get_status (GnomeScanNode* self);
GnomeScanStatus gnome_scan_job_get_status (GnomeScanJob* self);
GnomeScanScanner* gnome_scan_job_get_scanner (GnomeScanJob* self);
void gnome_scan_job_set_scanner (GnomeScanJob* self, GnomeScanScanner* value);
GnomeScanSink* gnome_scan_job_get_sink (GnomeScanJob* self);
double gnome_scan_job_get_progress (GnomeScanJob* self);
static void gnome_scan_job_finalize (GObject* obj);
static void gnome_scan_job_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec);
static void gnome_scan_job_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec);



static void _g_slist_free_g_object_unref (GSList* self) {
	g_slist_foreach (self, (GFunc) g_object_unref, NULL);
	g_slist_free (self);
}


GnomeScanJob* gnome_scan_job_construct (GType object_type, GnomeScanSink* sink) {
	GnomeScanJob * self;
	g_return_val_if_fail (sink != NULL, NULL);
	self = (GnomeScanJob*) g_object_new (object_type, NULL);
	gnome_scan_job_set_sink (self, sink);
	return self;
}


GnomeScanJob* gnome_scan_job_new (GnomeScanSink* sink) {
	return gnome_scan_job_construct (GNOME_SCAN_TYPE_JOB, sink);
}


static gpointer _g_object_ref0 (gpointer self) {
	return self ? g_object_ref (self) : NULL;
}


void gnome_scan_job_register_preselection (GnomeScanJob* self, GnomeScanPreselection* preselection) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (preselection != NULL);
	self->priv->preselections = g_slist_append (self->priv->preselections, _g_object_ref0 (preselection));
}


gboolean gnome_scan_job_is_running (GnomeScanJob* self) {
	gboolean result;
	g_return_val_if_fail (self != NULL, FALSE);
	result = self->priv->_status == GNOME_SCAN_STATUS_PROCESSING;
	return result;
}


void gnome_scan_job_run (GnomeScanJob* self) {
	gint count;
	g_return_if_fail (self != NULL);
	self->priv->cancelled = FALSE;
	count = 0;
	if (!gnome_scan_job_is_running (self)) {
		{
			GSList* node_collection;
			GSList* node_it;
			node_collection = self->priv->nodes;
			for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
				GnomeScanNode* node;
				node = _g_object_ref0 ((GnomeScanNode*) node_it->data);
				{
					gnome_scan_node_start_scan (node);
					_g_object_unref0 (node);
				}
			}
		}
		gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_PROCESSING);
	}
	while (TRUE) {
		if (!gnome_scan_job_run_once (self)) {
			break;
		}
		count++;
	}
	g_debug ("gnome-scan-job.vala:98: %d images acquired", count);
	gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_DONE);
}


void gnome_scan_job_end (GnomeScanJob* self) {
	g_return_if_fail (self != NULL);
	if (!gnome_scan_job_is_running (self)) {
		return;
	}
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->nodes;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GnomeScanNode* node;
			node = _g_object_ref0 ((GnomeScanNode*) node_it->data);
			{
				gnome_scan_node_end_scan (node);
				_g_object_unref0 (node);
			}
		}
	}
	gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_READY);
}


void gnome_scan_job_cancel (GnomeScanJob* self) {
	g_return_if_fail (self != NULL);
	self->priv->cancelled = TRUE;
}


gboolean gnome_scan_job_run_once (GnomeScanJob* self) {
	gboolean result;
	double progress;
	gboolean scanner_is_node;
	GeglProcessor* _tmp2_;
	g_return_val_if_fail (self != NULL, FALSE);
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->nodes;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GnomeScanNode* node;
			node = (GnomeScanNode*) node_it->data;
			{
				gboolean _tmp0_;
				_tmp0_ = FALSE;
				if (!gnome_scan_node_start_image (node)) {
					_tmp0_ = GNOME_SCAN_IS_SCANNER (node);
				} else {
					_tmp0_ = FALSE;
				}
				if (_tmp0_) {
					result = FALSE;
					return result;
				}
			}
		}
	}
	progress = 0.0;
	scanner_is_node = g_slist_length (gnome_scan_node_get_nodes ((GnomeScanNode*) self->priv->_scanner)) == 0;
	if (scanner_is_node) {
		while (TRUE) {
			gboolean _tmp1_;
			_tmp1_ = FALSE;
			if (!self->priv->cancelled) {
				_tmp1_ = gnome_scan_scanner_work (self->priv->_scanner, &progress);
			} else {
				_tmp1_ = FALSE;
			}
			if (!_tmp1_) {
				break;
			}
			self->priv->_progress = progress / 3.0;
		}
		gegl_node_set (self->priv->load_buffer, "operation", "gegl:load-buffer", "buffer", gnome_scan_scanner_get_buffer (self->priv->_scanner), NULL);
	}
	self->priv->processor = (_tmp2_ = gegl_node_new_processor (self->priv->gegl_sink, NULL), _g_object_unref0 (self->priv->processor), _tmp2_);
	while (TRUE) {
		gboolean _tmp3_;
		_tmp3_ = FALSE;
		if (!self->priv->cancelled) {
			_tmp3_ = gegl_processor_work (self->priv->processor, &progress);
		} else {
			_tmp3_ = FALSE;
		}
		if (!_tmp3_) {
			break;
		}
		if (scanner_is_node) {
			self->priv->_progress = 0.333 + ((progress * 2) / 3);
		} else {
			self->priv->_progress = progress;
		}
	}
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->nodes;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GnomeScanNode* node;
			node = (GnomeScanNode*) node_it->data;
			{
				gnome_scan_node_end_image (node);
			}
		}
	}
	result = TRUE;
	return result;
}


static void _gnome_scan_job_check_nodes_status_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self) {
	gnome_scan_job_check_nodes_status (self);
}


static void _gnome_scan_job_build_graph_g_object_notify (GnomeScanNode* _sender, GParamSpec* pspec, gpointer self) {
	gnome_scan_job_build_graph (self);
}


static void gnome_scan_job_register_node (GnomeScanJob* self, GnomeScanNode* node, GnomeScanNode* old) {
	g_return_if_fail (self != NULL);
	if (old != NULL) {
		GQuark _tmp1_;
		guint _tmp0_;
		GQuark _tmp3_;
		guint _tmp2_;
		g_signal_handlers_disconnect_matched ((GObject*) old, G_SIGNAL_MATCH_ID | G_SIGNAL_MATCH_DETAIL | G_SIGNAL_MATCH_FUNC | G_SIGNAL_MATCH_DATA, (g_signal_parse_name ("notify::status", G_TYPE_OBJECT, &_tmp0_, &_tmp1_, FALSE), _tmp0_), 0, NULL, (GCallback) _gnome_scan_job_check_nodes_status_g_object_notify, self);
		g_signal_handlers_disconnect_matched ((GObject*) old, G_SIGNAL_MATCH_ID | G_SIGNAL_MATCH_DETAIL | G_SIGNAL_MATCH_FUNC | G_SIGNAL_MATCH_DATA, (g_signal_parse_name ("notify::graph", G_TYPE_OBJECT, &_tmp2_, &_tmp3_, FALSE), _tmp2_), 0, NULL, (GCallback) _gnome_scan_job_build_graph_g_object_notify, self);
		self->priv->nodes = g_slist_remove (self->priv->nodes, old);
	}
	if (node != NULL) {
		gnome_scan_job_apply_preselections (self, node);
		g_signal_connect_object ((GObject*) node, "notify::status", (GCallback) _gnome_scan_job_check_nodes_status_g_object_notify, self, 0);
		g_signal_connect_object ((GObject*) node, "notify::graph", (GCallback) _gnome_scan_job_build_graph_g_object_notify, self, 0);
		if (GNOME_SCAN_IS_SCANNER (node)) {
			self->priv->nodes = g_slist_prepend (self->priv->nodes, _g_object_ref0 (node));
		} else {
			if (GNOME_SCAN_IS_SINK (node)) {
				self->priv->nodes = g_slist_append (self->priv->nodes, _g_object_ref0 (node));
			} else {
				GSList* last;
				last = g_slist_last (self->priv->nodes);
				if (GNOME_SCAN_IS_SINK (last->data)) {
					self->priv->nodes = g_slist_insert_before (self->priv->nodes, last, _g_object_ref0 (node));
				} else {
					self->priv->nodes = g_slist_append (self->priv->nodes, _g_object_ref0 (node));
				}
			}
		}
	}
	gnome_scan_job_check_nodes_status (self);
	gnome_scan_job_build_graph (self);
}


static void gnome_scan_job_apply_preselections (GnomeScanJob* self, GnomeScanNode* node) {
	g_return_if_fail (self != NULL);
	g_return_if_fail (node != NULL);
	{
		GSList* presel_collection;
		GSList* presel_it;
		presel_collection = self->priv->preselections;
		for (presel_it = presel_collection; presel_it != NULL; presel_it = presel_it->next) {
			GnomeScanPreselection* presel;
			presel = _g_object_ref0 ((GnomeScanPreselection*) presel_it->data);
			{
				GnomeScanOption* option;
				option = gnome_scan_node_lookup_option (node, gnome_scan_preselection_get_option_name (presel));
				if (option != NULL) {
					gnome_scan_preselection_apply (presel, option);
				}
				_g_object_unref0 (presel);
				_g_object_unref0 (option);
			}
		}
	}
}


static void gnome_scan_job_build_graph (GnomeScanJob* self) {
	GeglNode* prev;
	GSList* _tmp0_;
	GeglNode* _tmp3_;
	g_return_if_fail (self != NULL);
	if (((gint) self->priv->_status) < ((gint) GNOME_SCAN_STATUS_READY)) {
		return;
	}
	prev = NULL;
	{
		GSList* node_collection;
		GSList* node_it;
		node_collection = self->priv->graph;
		for (node_it = node_collection; node_it != NULL; node_it = node_it->next) {
			GeglNode* node;
			node = _g_object_ref0 ((GeglNode*) node_it->data);
			{
				if (prev != NULL) {
					gegl_node_disconnect (node, "input");
				}
				prev = node;
				_g_object_unref0 (node);
			}
		}
	}
	self->priv->graph = (_tmp0_ = NULL, __g_slist_free_g_object_unref0 (self->priv->graph), _tmp0_);
	prev = NULL;
	if (g_slist_length (gnome_scan_node_get_nodes ((GnomeScanNode*) self->priv->_scanner)) == 0) {
		GeglNode* _tmp1_;
		self->priv->load_buffer = (_tmp1_ = gegl_node_new (), _g_object_unref0 (self->priv->load_buffer), _tmp1_);
		gegl_node_set (self->priv->load_buffer, "operation", "gegl:load-buffer", NULL);
		prev = self->priv->load_buffer;
	} else {
		GeglNode* _tmp2_;
		self->priv->load_buffer = (_tmp2_ = NULL, _g_object_unref0 (self->priv->load_buffer), _tmp2_);
	}
	{
		GSList* gsnode_collection;
		GSList* gsnode_it;
		gsnode_collection = self->priv->nodes;
		for (gsnode_it = gsnode_collection; gsnode_it != NULL; gsnode_it = gsnode_it->next) {
			GnomeScanNode* gsnode;
			gsnode = (GnomeScanNode*) gsnode_it->data;
			{
				{
					GSList* gnode_collection;
					GSList* gnode_it;
					gnode_collection = gnome_scan_node_get_nodes (gsnode);
					for (gnode_it = gnode_collection; gnode_it != NULL; gnode_it = gnode_it->next) {
						GeglNode* gnode;
						gnode = (GeglNode*) gnode_it->data;
						{
							self->priv->graph = g_slist_append (self->priv->graph, _g_object_ref0 (gnode));
							if (prev != NULL) {
								gegl_node_link (prev, gnode);
							}
							prev = gnode;
						}
					}
				}
			}
		}
	}
	self->priv->gegl_sink = (_tmp3_ = _g_object_ref0 (prev), _g_object_unref0 (self->priv->gegl_sink), _tmp3_);
}


static void gnome_scan_job_check_nodes_status (GnomeScanJob* self) {
	GnomeScanStatus status;
	gboolean _tmp0_;
	gboolean _tmp1_;
	g_return_if_fail (self != NULL);
	status = 0;
	_tmp0_ = FALSE;
	_tmp1_ = FALSE;
	if (self->priv->_scanner == NULL) {
		_tmp1_ = TRUE;
	} else {
		_tmp1_ = self->priv->_sink == NULL;
	}
	if (_tmp1_) {
		_tmp0_ = TRUE;
	} else {
		_tmp0_ = g_slist_length (gnome_scan_node_get_nodes ((GnomeScanNode*) self->priv->_sink)) == 0;
	}
	if (_tmp0_) {
		gnome_scan_job_set_status (self, GNOME_SCAN_STATUS_UNCONFIGURED);
	} else {
		gboolean _tmp2_;
		gboolean _tmp3_;
		gboolean _tmp4_;
		gboolean redo_graph;
		status = GNOME_SCAN_STATUS_READY;
		_tmp2_ = FALSE;
		if (self->priv->_scanner != NULL) {
			_tmp2_ = ((gint) gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_scanner)) < ((gint) status);
		} else {
			_tmp2_ = FALSE;
		}
		if (_tmp2_) {
			status = gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_scanner);
		}
		_tmp3_ = FALSE;
		if (self->priv->_sink != NULL) {
			_tmp3_ = ((gint) gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_sink)) < ((gint) status);
		} else {
			_tmp3_ = FALSE;
		}
		if (_tmp3_) {
			status = gnome_scan_node_get_status ((GnomeScanNode*) self->priv->_sink);
		}
		_tmp4_ = FALSE;
		if (((gint) status) > ((gint) self->priv->_status)) {
			_tmp4_ = status == GNOME_SCAN_STATUS_READY;
		} else {
			_tmp4_ = FALSE;
		}
		redo_graph = _tmp4_;
		gnome_scan_job_set_status (self, status);
		if (((gint) status) < ((gint) GNOME_SCAN_STATUS_READY)) {
			self->priv->cancelled = TRUE;
		}
		if (redo_graph) {
			gnome_scan_job_build_graph (self);
		}
	}
}


GnomeScanStatus gnome_scan_job_get_status (GnomeScanJob* self) {
	GnomeScanStatus result;
	g_return_val_if_fail (self != NULL, 0);
	result = self->priv->_status;
	return result;
}


void gnome_scan_job_set_status (GnomeScanJob* self, GnomeScanStatus value) {
	g_return_if_fail (self != NULL);
	self->priv->_status = value;
	g_object_notify ((GObject *) self, "status");
}


GnomeScanScanner* gnome_scan_job_get_scanner (GnomeScanJob* self) {
	GnomeScanScanner* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_scanner;
	return result;
}


void gnome_scan_job_set_scanner (GnomeScanJob* self, GnomeScanScanner* value) {
	GnomeScanScanner* old;
	GnomeScanScanner* _tmp0_;
	g_return_if_fail (self != NULL);
	old = _g_object_ref0 (self->priv->_scanner);
	self->priv->_scanner = (_tmp0_ = _g_object_ref0 (value), _g_object_unref0 (self->priv->_scanner), _tmp0_);
	gnome_scan_job_register_node (self, (GnomeScanNode*) value, (GnomeScanNode*) old);
	_g_object_unref0 (old);
	g_object_notify ((GObject *) self, "scanner");
}


GnomeScanSink* gnome_scan_job_get_sink (GnomeScanJob* self) {
	GnomeScanSink* result;
	g_return_val_if_fail (self != NULL, NULL);
	result = self->priv->_sink;
	return result;
}


void gnome_scan_job_set_sink (GnomeScanJob* self, GnomeScanSink* value) {
	GnomeScanSink* old;
	GnomeScanSink* _tmp0_;
	g_return_if_fail (self != NULL);
	old = _g_object_ref0 (self->priv->_sink);
	self->priv->_sink = (_tmp0_ = _g_object_ref0 (value), _g_object_unref0 (self->priv->_sink), _tmp0_);
	gnome_scan_job_register_node (self, (GnomeScanNode*) value, (GnomeScanNode*) old);
	_g_object_unref0 (old);
	g_object_notify ((GObject *) self, "sink");
}


double gnome_scan_job_get_progress (GnomeScanJob* self) {
	double result;
	g_return_val_if_fail (self != NULL, 0.0);
	result = self->priv->_progress;
	return result;
}


static void gnome_scan_job_class_init (GnomeScanJobClass * klass) {
	gnome_scan_job_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (GnomeScanJobPrivate));
	G_OBJECT_CLASS (klass)->get_property = gnome_scan_job_get_property;
	G_OBJECT_CLASS (klass)->set_property = gnome_scan_job_set_property;
	G_OBJECT_CLASS (klass)->finalize = gnome_scan_job_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_STATUS, g_param_spec_enum ("status", "status", "status", GNOME_SCAN_TYPE_STATUS, GNOME_SCAN_STATUS_INITIALIZING, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_SCANNER, g_param_spec_object ("scanner", "scanner", "scanner", GNOME_SCAN_TYPE_SCANNER, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE | G_PARAM_CONSTRUCT));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_SINK, g_param_spec_object ("sink", "sink", "sink", GNOME_SCAN_TYPE_SINK, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), GNOME_SCAN_JOB_PROGRESS, g_param_spec_double ("progress", "progress", "progress", -G_MAXDOUBLE, G_MAXDOUBLE, 0.0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void gnome_scan_job_instance_init (GnomeScanJob * self) {
	self->priv = GNOME_SCAN_JOB_GET_PRIVATE (self);
	self->priv->_status = GNOME_SCAN_STATUS_INITIALIZING;
	self->priv->_progress = (double) 0;
	self->priv->cancelled = FALSE;
	self->priv->processor = NULL;
	self->priv->nodes = NULL;
}


static void gnome_scan_job_finalize (GObject* obj) {
	GnomeScanJob * self;
	self = GNOME_SCAN_JOB (obj);
	_g_object_unref0 (self->priv->_scanner);
	_g_object_unref0 (self->priv->_sink);
	__g_slist_free_g_object_unref0 (self->priv->preselections);
	_g_object_unref0 (self->priv->load_buffer);
	__g_slist_free_g_object_unref0 (self->priv->graph);
	_g_object_unref0 (self->priv->gegl_sink);
	_g_object_unref0 (self->priv->processor);
	__g_slist_free_g_object_unref0 (self->priv->nodes);
	G_OBJECT_CLASS (gnome_scan_job_parent_class)->finalize (obj);
}


GType gnome_scan_job_get_type (void) {
	static GType gnome_scan_job_type_id = 0;
	if (gnome_scan_job_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (GnomeScanJobClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) gnome_scan_job_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (GnomeScanJob), 0, (GInstanceInitFunc) gnome_scan_job_instance_init, NULL };
		gnome_scan_job_type_id = g_type_register_static (G_TYPE_OBJECT, "GnomeScanJob", &g_define_type_info, 0);
	}
	return gnome_scan_job_type_id;
}


static void gnome_scan_job_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	GnomeScanJob * self;
	self = GNOME_SCAN_JOB (object);
	switch (property_id) {
		case GNOME_SCAN_JOB_STATUS:
		g_value_set_enum (value, gnome_scan_job_get_status (self));
		break;
		case GNOME_SCAN_JOB_SCANNER:
		g_value_set_object (value, gnome_scan_job_get_scanner (self));
		break;
		case GNOME_SCAN_JOB_SINK:
		g_value_set_object (value, gnome_scan_job_get_sink (self));
		break;
		case GNOME_SCAN_JOB_PROGRESS:
		g_value_set_double (value, gnome_scan_job_get_progress (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void gnome_scan_job_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	GnomeScanJob * self;
	self = GNOME_SCAN_JOB (object);
	switch (property_id) {
		case GNOME_SCAN_JOB_STATUS:
		gnome_scan_job_set_status (self, g_value_get_enum (value));
		break;
		case GNOME_SCAN_JOB_SCANNER:
		gnome_scan_job_set_scanner (self, g_value_get_object (value));
		break;
		case GNOME_SCAN_JOB_SINK:
		gnome_scan_job_set_sink (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}




