/* -*- Mode: C; indent-tabs-mode: t; c-basic-offset: 4; tab-width: 4 -*- */
/*
 * gnome-scan
 * Copyright (C) Étienne Bersac 2007 <bersace03@laposte.net>
 * 
 * gnome-scan is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * gnome-scan is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with gnome-scan.  If not, write to:
 * 	The Free Software Foundation, Inc.,
 * 	51 Franklin Street, Fifth Floor
 * 	Boston, MA  02110-1301, USA.
 */

#include <glib/gi18n.h>
#include <gtk/gtk.h>
#include <gnome-scan-plugin.h>
#include <gnome-scan-preview-plugin-area.h>
#include "gsane-meta-param.h"

#define	GSANE_DEFINE_META_PARAM(Name, name, opts, params)				\
	static void	meta_param_##name##_class_init (MetaParamClass *klass);	\
	static void	meta_param_##name##_init (MetaParam *mp);				\
	static void	meta_param_##name##_finalize	(MetaParam *param);		\
	static void	meta_param_##name##_add_param (MetaParam *param, GParamSpec *spec);	\
	static gboolean	meta_param_##name##_get_params (MetaParam *param);	\
	static GValue*	meta_param_##name##_get_value (MetaParam *param, GParamSpec *spec);	\
	static SANE_Int	meta_param_##name##_set_value (MetaParam *param, GParamSpec *spec, GValue *value); \
	GType meta_param_##name##_get_type () {								\
		static GType 				type;								\
		if (type == 0) {												\
			static MetaParamTypeInfo info = {							\
				sizeof (MetaParam##Name),								\
				(GClassInitFunc) meta_param_##name##_class_init,		\
				(GInstanceInitFunc) meta_param_##name##_init			\
			};															\
			type = meta_param_type_register_static ("MetaParam" #Name, &info); \
		}																\
		return type;													\
	}																	\
	static void meta_param_##name##_class_init (MetaParamClass *klass) { \
		MetaParamClass *parent_class = GSANE_META_PARAM_CLASS (klass);	\
		parent_class->finalize	= meta_param_##name##_finalize;			\
		parent_class->add_param		= meta_param_##name##_add_param;	\
		parent_class->get_params	= meta_param_##name##_get_params;	\
		parent_class->set_value		= meta_param_##name##_set_value;	\
		parent_class->get_value		= meta_param_##name##_get_value;	\
		parent_class->options_names	= g_strsplit (opts, ",", 8);		\
		parent_class->params_names = g_strsplit (params, ",", 8);		\
	}																	\
	MetaParam* meta_param_##name (GSaneScanner *gss) {					\
		return meta_param_internal (meta_param_##name##_get_type(), #name, gss); }

typedef struct _MetaParamTypeInfo MetaParamTypeInfo;
struct _MetaParamTypeInfo
{
	guint					instance_size;
	GClassInitFunc			class_init;
	GInstanceInitFunc		init;
};

/* META PARAMS */
static void	meta_param_class_init	(MetaParamClass *klass);
static void	meta_param_init			(MetaParam *param);

GType
meta_param_get_type ()
{
	static GType type = 0;
	static GTypeInfo	tinfo = {
		sizeof (MetaParamClass),
		NULL, NULL, /* base */
		(GClassInitFunc) meta_param_class_init, NULL, NULL, /* class */
		sizeof (MetaParam), 0,
		(GInstanceInitFunc) meta_param_init, NULL
	};
	static GTypeFundamentalInfo	finfo = {
		G_TYPE_FLAG_CLASSED | G_TYPE_FLAG_INSTANTIATABLE | G_TYPE_FLAG_DERIVABLE
	};
	
	if (type == 0) {
		type = g_type_fundamental_next();
		g_type_register_fundamental (type,
									 "MetaParam",
									 &tinfo,
									 &finfo,
									 G_TYPE_FLAG_ABSTRACT);
	}
	return type;
}

static void
meta_param_class_init (MetaParamClass *klass)
{
	
}

static void
meta_param_init (MetaParam *param)
{
}

static GType
meta_param_type_register_static (const gchar *name, const MetaParamTypeInfo *mp_info)
{
	GTypeInfo info = {
		sizeof (MetaParamClass),
		NULL, NULL, /* base */
		(GClassInitFunc) mp_info->class_init, NULL, mp_info,
		mp_info->instance_size, 0,
		mp_info->init, NULL,
	};
	
	return g_type_register_static (GSANE_TYPE_META_PARAM,
								   name, &info, 0);
}

MetaParam*
meta_param_internal		(GType type, gchar *name, GSaneScanner *gss)
{
	MetaParam* param = GSANE_META_PARAM (g_type_create_instance (type));
	param->name	= name;
	param->gss	= gss;
	return param;
}

gchar*
meta_param_get_name (MetaParam *mp)
{
	return g_strdup (mp->name);
}

void
meta_param_add_params (MetaParam *mp, GParamSpec *spec)
{
	GSANE_META_PARAM_GET_CLASS (mp)->add_param (mp, spec);
}

gboolean
meta_param_get_params (MetaParam *mp)
{
	return GSANE_META_PARAM_GET_CLASS (mp)->get_params (mp);
}

SANE_Int
meta_param_set_value (MetaParam *mp, GParamSpec *spec, GnomeScanSettings *settings)
{
	mp->settings = g_object_ref (settings);
	GValue *value = gnome_scan_settings_get (settings, g_param_spec_get_name (spec));
	SANE_Int ret = GSANE_META_PARAM_GET_CLASS (mp)->set_value (mp, spec, value);
	g_object_unref (mp->settings);
	mp->settings = NULL;
	return ret;
}

GValue*
meta_param_get_value (MetaParam *mp, GParamSpec *spec)
{
	return GSANE_META_PARAM_GET_CLASS (mp)->get_value (mp, spec);
}

void meta_param_destroy (MetaParam* mp)
{
	MetaParamClass *klass = GSANE_META_PARAM_GET_CLASS (mp);
	klass->finalize (mp);
	g_type_free_instance (&mp->instance);
}





/* AREA */
/* TODO : handle paper-width,paper-height */
GSANE_DEFINE_META_PARAM(PaperSize, paper_size, "tl-x,tl-y,br-x,br-y,page-format,paper-size", "paper-size,orig,orientation");

static void
meta_param_paper_size_init (MetaParam *mp)
{
}

static void
meta_param_paper_size_finalize (MetaParam *param)
{
}

static void
meta_param_paper_size_add_param (MetaParam *mp, GParamSpec *spec)
{
	MetaParamPaperSize *mpps = GSANE_META_PARAM_PAPER_SIZE (mp);
	const gchar *name = g_param_spec_get_name (spec);
	
	/*g_debug ("%s:%i %s, %s", __FUNCTION__, __LINE__, mp->name, name);*/
	
	if (g_str_equal (name, "tl-x")) {
		mpps->tl_x = spec;
	}
	else if (g_str_equal (name, "tl-y")) {
		mpps->tl_y = spec;
	}
	else if (g_str_equal (name, "br-x")) {
		mpps->br_x = spec;
	}
	else if (g_str_equal (name, "br-y")) {
		mpps->br_y = spec;
	}
	else if (g_str_equal (name, "paper-size") ||
			 g_str_equal (name, "paper-format")) {
		g_debug (G_STRLOC ": Warning : ignoring %s option",
				 name);
	}
}

static gboolean
meta_param_paper_size_get_params			(MetaParam *mp)
{
	GParamSpec *pspec;
	GValue *integer = g_new0 (GValue, 1);
	GSParamSpecRange* spec;
	MetaParamPaperSize *mpps;
	GeglRectangle rect, deft;
	gint i;
	mpps = GSANE_META_PARAM_PAPER_SIZE (mp);

	g_value_init (integer, G_TYPE_INT);
#define set_int(val,var)	g_value_transform (val, integer); var = g_value_get_int (integer)
	spec = GS_PARAM_SPEC_RANGE (mpps->tl_x);
	/* 	set_int (spec->minimum, rect.x); /\* usualy 0 *\/ */
	set_int (spec->maximum, rect.width);
	set_int (spec->default_value, deft.x);
	
	spec = GS_PARAM_SPEC_RANGE (mpps->tl_y);
	/* 	set_int (spec->minimum, rect.y); /\* usualy 0 *\/ */
	set_int (spec->maximum, rect.height);
	set_int (spec->default_value, deft.y);
	
	spec = GS_PARAM_SPEC_RANGE (mpps->br_x);
	set_int (spec->default_value, deft.width);
	deft.width-= deft.x;
	
	spec = GS_PARAM_SPEC_RANGE (mpps->br_y);
	set_int (spec->default_value, deft.height);
	deft.height-= deft.y;
#undef set_int

	/* note: decomposing area in three options orientation, origin and paper-size
	 except these optoin to appear in this order. */
	
	/* ORIENTATION */
	pspec = gs_param_spec_page_orientation ("page-orientation", N_("Page Orientation"), N_("Page orientation"),
											GS_PARAM_GROUP_FORMAT,
											GTK_PAGE_ORIENTATION_PORTRAIT,
											G_PARAM_SPEC (mpps->tl_x)->flags);
	gs_param_spec_set_index (pspec,
							 gs_param_spec_get_index (G_PARAM_SPEC (mpps->tl_x))+2);
	g_param_spec_set_qdata (pspec, GSANE_META_PARAM_QUARK, mp);
	gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (mp->gss), pspec);
	
	/* ORIG */
	pspec = gs_param_spec_pointer ("origin", N_("Origin"), N_("Origin of scan window"),
								   GS_PARAM_GROUP_PREVIEW,
								   GNOME_TYPE_SCAN_PREVIEW_PLUGIN_AREA,
								   G_PARAM_SPEC (mpps->tl_x)->flags);
	gs_param_spec_set_index (pspec,
							 gs_param_spec_get_index (G_PARAM_SPEC (mpps->tl_x))+1);
	g_param_spec_set_qdata (pspec, GSANE_META_PARAM_QUARK, mp);
	gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (mp->gss), pspec);

	/* PAPER SIZE */
	static const gchar*names[] = {
		GTK_PAPER_NAME_A5,
		GTK_PAPER_NAME_A4,
		GTK_PAPER_NAME_B5,
		GTK_PAPER_NAME_LETTER,
		GTK_PAPER_NAME_EXECUTIVE,
		GTK_PAPER_NAME_LEGAL,
		"iso_dl",
		"om_small-photo",
		NULL
	};
	GSList *enumeration = NULL;

	GValue *v;
#define get_opt(opt,var)	v=gsane_scanner_option_get_value (mp->gss, mpps->opt); \
	g_value_transform(v,integer);										\
	g_free(v);															\
	var = g_value_get_int(integer);

	gint x,y,w,h;
	get_opt(tl_x, x);
	get_opt(tl_y, y);
	get_opt(br_x, w);
	get_opt(br_y, h);
	w-=x;
	h-=y;
	
#undef get_opt

	g_value_unset (integer);
	g_free (integer);

	enumeration = g_slist_append (enumeration,
								  gtk_paper_size_new_custom ("manual",
															 _("Manual"),
															 w,
															 h,
															 GTK_UNIT_MM)); /* what if w and h are not mm, but e.g. pixel ? */
	enumeration = g_slist_append (enumeration,
								  gtk_paper_size_new_custom ("maximal",
															 _("Maximal"),
															 (gdouble) rect.width,
															 (gdouble) rect.height,
															 gs_param_spec_get_unit (mpps->tl_x)));
	for (i = 0 ; names[i] ; i++) {
		enumeration = g_slist_append (enumeration,
									  gtk_paper_size_new (names[i]));
	}
	
	pspec = gs_param_spec_paper_size ("paper-size", N_("Paper Size"),
									  N_("Document paper size."),
									  GS_PARAM_GROUP_FORMAT,
									  enumeration->data,
									  enumeration,
									  G_PARAM_SPEC (mpps->tl_x)->flags);
	gs_param_spec_set_index (pspec,
							 gs_param_spec_get_index (G_PARAM_SPEC (mpps->tl_x)));
			
	g_param_spec_set_qdata (pspec, GSANE_META_PARAM_QUARK, mp);
	gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (mp->gss), pspec);


	return TRUE;
}

static SANE_Int
meta_param_paper_size_set_value			(MetaParam *mp, GParamSpec *spec, GValue *value)
{
	MetaParamPaperSize *mpps = GSANE_META_PARAM_PAPER_SIZE (mp);
	SANE_Int i = 0;
	gint x, y, w, h;
	
	/* setup translator for integer/double handling */
	GValue *integer = g_new0 (GValue, 1);
	GValue *trans = g_new0 (GValue, 1);
	GValue *v;
	g_value_init (integer, G_TYPE_INT);
	g_value_init (trans, G_PARAM_SPEC_VALUE_TYPE (mpps->tl_x));
	g_param_value_set_default (mpps->tl_x, trans);
	
	const gchar*name = g_param_spec_get_name (spec);
	
#define set_opt(opt,val)	g_value_set_int(integer,(val));			\
	g_value_transform(integer, trans);								\
	i=i|gsane_scanner_option_set_value (mp->gss, mpps->opt, trans)

#define get_opt(opt,var)	v=gsane_scanner_option_get_value (mp->gss, mpps->opt); \
	g_value_transform(v,integer);										\
	g_free(v);															\
	var = g_value_get_int(integer);

	/* paper-size */
	if (g_str_equal (name, "paper-size")) {
		mpps->ps = g_value_get_boxed (value);
		guint unit;
		
		switch (gs_param_spec_get_unit (mpps->tl_x)) {
		case GS_UNIT_MM:
			unit = GTK_UNIT_MM;
			break;
		case GS_UNIT_PIXEL:
		default:
			/* warning: gtk uses hardcoded 72dpi resolution */
			unit = GTK_UNIT_PIXEL;
			break;
		}

		/* take offset in account */
		get_opt(tl_x, x);
		get_opt(tl_y, y);

		switch (mpps->po) {
		case GTK_PAGE_ORIENTATION_LANDSCAPE:
		case GTK_PAGE_ORIENTATION_REVERSE_LANDSCAPE:
			h = gtk_paper_size_get_width (mpps->ps, unit);
			w = gtk_paper_size_get_height (mpps->ps, unit);
			break;
		case GTK_PAGE_ORIENTATION_PORTRAIT:
		case GTK_PAGE_ORIENTATION_REVERSE_PORTRAIT:
		default:
			w = gtk_paper_size_get_width (mpps->ps, unit);
			h = gtk_paper_size_get_height (mpps->ps, unit);
			break;
		}
			
		set_opt(br_x, x+w);
		set_opt(br_y, y+h);
	}
	else if (g_str_equal (name, "origin")) {
		mpps->origin = g_value_get_pointer (value);

		/* offset the current area */

		get_opt(tl_x, x);
		get_opt(tl_y, y);
		get_opt(br_x, w);
		get_opt(br_y, h);
		w-=x;
		h-=y;

		set_opt(tl_x, mpps->origin->x);
		set_opt(tl_y, mpps->origin->y);
		set_opt(br_x, mpps->origin->x+w);
		set_opt(br_y, mpps->origin->y+h);
	}
	else if (g_str_equal (name, "page-orientation")) {
		mpps->po = g_value_get_enum (value);
		gint t;

		get_opt(tl_x, x);
		get_opt(tl_y, y);
		get_opt(br_x, w);
		w-=x;
		get_opt(br_y, h);
		h-=y;
		switch (mpps->po) {
		case GTK_PAGE_ORIENTATION_PORTRAIT:
		case GTK_PAGE_ORIENTATION_REVERSE_PORTRAIT:
			if (w > h) {
				t = h;
				h = w;
				w = t;
			}		
			break;
		case GTK_PAGE_ORIENTATION_LANDSCAPE:
		case GTK_PAGE_ORIENTATION_REVERSE_LANDSCAPE:
			if (w < h) {
				t = h;
				h = w;
				w = t;
			}		
			break;
		}

		set_opt (br_x, x+w);
		set_opt (br_y, y+h);
	}
#undef set_opt
#undef get_opt

	g_value_unset (trans);
	g_free (trans);
	
	return i;
}

static GValue*
meta_param_paper_size_get_value			(MetaParam *mp, GParamSpec *spec)
{
	g_warning ("%s not yet implemented", __FUNCTION__);
	return NULL;
}



/* SOURCE */
GSANE_DEFINE_META_PARAM(Source, source, "doc-source,source,adf,duplex", "source");

/* SANE 2 well known source option value */
#define	GSANE_SOURCE_FLATBED	N_("Flatbed")
#define	GSANE_SOURCE_ADF		N_("Automatic Document Feeder")
/* device source option allowing to scan film and transparency */
#define	GSANE_SOURCE_TRANS		N_("Transparency Adapter")

static void
meta_param_source_init (MetaParam *mp)
{
	MetaParamSource *mps = GSANE_META_PARAM_SOURCE (mp);
	mps->dic = g_hash_table_new (g_str_hash, g_str_equal);
}

static void
meta_param_source_finalize (MetaParam *mp)
{
	MetaParamSource *mps = GSANE_META_PARAM_SOURCE (mp);
	g_hash_table_destroy (mps->dic);
}

/* workaround to make all backend source option consistent */
static const gchar*
meta_param_source_get_src (const gchar *sane_val)
{
	static const gchar* flatbed_src[] = {
		"FB",
		"Normal",
		NULL
	};
	static const gchar* adf_src[] = {
		"ADF",
		"ADF Front",
		NULL
	};
	static const gchar* trans_src[] = {
		"Transparency",
		"Film",
		NULL
	};
	
	if (gsane_str_matches_strv (sane_val, flatbed_src)) {
		return GSANE_SOURCE_FLATBED;
	}
	else if (gsane_str_matches_strv (sane_val, adf_src)) {
		return GSANE_SOURCE_ADF;
	}
	else if (gsane_str_matches_strv (sane_val, trans_src)) {
		return GSANE_SOURCE_TRANS;
	}
	return sane_val;
}

static void
meta_param_source_add_param (MetaParam *mp, GParamSpec *spec)
{
	MetaParamSource *mps = GSANE_META_PARAM_SOURCE (mp);
	static const gchar* src_names[] = {
		"source",
		"doc-source", /* samsung backend use doc-source x( */
		NULL
	};
	
	if (gsane_str_matches_strv (g_param_spec_get_name (spec), src_names)) {
		mps->source = spec;
		return;
	}
	
#define	set_opt_if_matches(param, oname, opt)	if (g_str_equal (#oname, g_param_spec_get_name (opt))) { \
		param->oname = opt; return; }
	
	set_opt_if_matches(mps, duplex, spec);
	set_opt_if_matches(mps, adf, spec);
	
#undef set_opt_if_matches
}

static gboolean
meta_param_source_get_params (MetaParam *mp)
{
	MetaParamSource *mps = GSANE_META_PARAM_SOURCE (mp);
	GSParamSpecEnum *psenum;
	GParamSpec *spec;
	GValueArray* sources;
	GValue *value, *default_value;
	const gchar *sane_val, *src;
	guint i;
	
	sources	= g_value_array_new (3);
	if (G_IS_PARAM_SPEC (mps->source)) {
		psenum = GS_PARAM_SPEC_ENUM (mps->source);
		
		for (i = 0; i < psenum->values->n_values; i++) {
			value = g_value_array_get_nth (psenum->values, i);
			sane_val = g_value_get_string (value);
			src = meta_param_source_get_src (sane_val);
			if (src) {
				g_hash_table_insert (mps->dic,
									 g_strdup (src),
									 g_strdup (sane_val));
				value = g_new0 (GValue, 1);
				g_value_init (value, G_TYPE_STRING);
				g_value_set_string (value, src);
				g_value_array_append (sources, value);
			}
		}
		
		/* get default value and convert it */
		default_value = g_new0 (GValue, 1);
		g_value_init (default_value,
					  G_PARAM_SPEC_VALUE_TYPE (mps->source));
		g_param_value_set_default (mps->source, default_value);
		value = g_new0 (GValue, 1);
		g_value_init (value, G_TYPE_STRING);
		g_value_copy (default_value, value);
		g_value_unset (default_value);
		g_free (default_value);
		
		spec = gs_param_spec_enum ("source",
								   g_param_spec_get_nick (mps->source),
								   g_param_spec_get_blurb (mps->source),
								   gs_param_spec_get_group (mps->source),
								   sources, value,
								   mps->source->flags);
		gs_param_spec_set_group (spec, GS_PARAM_GROUP_SCANNER_FRONT);
		gs_param_spec_set_domain (spec, gs_param_spec_get_domain (mps->source));
		gs_param_spec_set_index (spec, gs_param_spec_get_index (mps->source));
		g_param_spec_set_qdata (spec, GSANE_META_PARAM_QUARK, mp);
		
		gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (mp->gss), spec);
	}
	
	/* duplex, not tested too */
	if (G_IS_PARAM_SPEC_BOOLEAN (mps->duplex)) {
		gs_param_spec_set_group (mps->duplex,
								 GS_PARAM_GROUP_FORMAT);
		g_param_spec_set_qdata (mps->duplex, GSANE_META_PARAM_QUARK, mp);
		gnome_scan_plugin_params_add (GNOME_SCAN_PLUGIN (mp->gss),
									  mps->duplex);
	}
	return TRUE;
}

static SANE_Int
meta_param_source_set_value (MetaParam *mp, GParamSpec *spec, GValue *value)
{
	MetaParamSource *mps = GSANE_META_PARAM_SOURCE (mp);
	GValue *val;
	gchar *src_val, *sane_val;
	SANE_Int i = 0;
	
	if (g_str_equal (g_param_spec_get_name (spec), "duplex")) {
		gsane_scanner_option_set_value (mp->gss, spec, value);
	}
	else if (g_str_equal (g_param_spec_get_name (spec), "source")) {
		val = g_new0 (GValue, 1);
		src_val = g_value_dup_string (value);
		
		/* Handle device using a boolean ADF option
		 * i'm not able to test this … */
		if (mps->adf) {
			g_value_init (val, G_TYPE_BOOLEAN);
			g_value_set_boolean (val, g_str_equal (src_val, GSANE_SOURCE_ADF));
			i = gsane_scanner_option_set_value (mp->gss, mps->adf, val);
			g_value_unset (val);
		}
		
		/* set backend value */
		sane_val = g_hash_table_lookup (mps->dic, src_val);
		if (sane_val) {
			g_value_init (val, G_TYPE_STRING);
			g_value_set_string (val, sane_val);
			i = i|gsane_scanner_option_set_value (mp->gss, mps->source, val);
			g_value_unset (val);
		}
		
		mp->gss->adf = g_str_equal (src_val, GSANE_SOURCE_ADF);
		
		g_free (val);
		g_free (src_val);
	}
	
	return i;
}

static GValue*
meta_param_source_get_value (MetaParam *mp, GParamSpec *spec)
{
	return NULL;
}
