# Infrared Remote Control Properties for GNOME
# Copyright (C) 2008 Fluendo Embedded S.L. (www.fluendo.com)
#
# This program is free software; you can redistribute it and/or modify it under
# the terms of the GNU General Public License as published by the Free Software
# Foundation; either version 2 of the License, or (at your option) any later
# version.
#
# This program is distributed in the hope that it will be useful, but WITHOUT
# ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
# FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more
# details.
#
# You should have received a copy of the GNU General Public License along with
# this program; if not, write to the Free Software Foundation, Inc.,
# 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
'''
The main window of the application.
'''

import dbus, gobject, gtk, gtk.glade, logging

from gettext               import gettext as _
from gnome_lirc_properties import config, lirc, model, hardware, policykit

from gnome_lirc_properties.ui.common                import show_message
from gnome_lirc_properties.ui.CustomConfiguration   import CustomConfiguration
from gnome_lirc_properties.ui.ReceiverChooserDialog import ReceiverChooserDialog

class RemoteControlProperties(object):
    '''
    The main window.
    '''

    def __init__(self, glade_xml):
        # Prevent UI changes to be written back to configuration files:
        self.__custom_configuration = None
        self.__configuration_level = -1

        # Initialize models and views.
        self.__ui = glade_xml
        self.__ui.signal_autoconnect(self)

        self.__dialog = self.__ui.get_widget('lirc_properties_dialog')
        self.__receiver_chooser = ReceiverChooserDialog(self.__ui)

        self.__setup_models()
        self.__setup_key_listener()
        self.__setup_device_area()
        self.__setup_product_lists()
        self.__setup_authorization()

        # Give auto-detect-button and progress bar equal size, to ensure that
        # switching between them doesn't make the dialog change its size:

        size_group = gtk.SizeGroup(gtk.SIZE_GROUP_VERTICAL)
        size_group.add_widget(self.__ui.get_widget('auto-detect-alignment'))
        size_group.add_widget(self.__ui.get_widget('progress-box'))

        # Look in the configuration to show previously-chosen details:

        self.__restore_hardware_settings()

        # Allow UI changes to be written back to configuration files:
        self.__configuration_level = 0

    def __setup_models(self):
        '''
        Initialize model objects of the dialog.
        '''

        # pylint: disable-msg=W0201,E1101

        receivers_db = hardware.HardwareDatabase(self.__ui.relative_file('receivers.conf'))

        self.__remotes_db = lirc.RemotesDatabase()
        self.__remotes_db.load(self.__ui.relative_file('linux-input-layer-lircd.conf'))
        self.__remotes_db.load_folder(config.LIRC_REMOTES_DATABASE)

        self.__hardware_manager = hardware.HardwareManager(receivers_db)
        self.__hardware_manager.connect('search-progress', self._on_search_progress)
        self.__hardware_manager.connect('search-finished', self._on_search_finished)
        self.__hardware_manager.connect('receiver-found',  self._on_receiver_found)

        self.__receiver_vendors = model.ReceiverVendorList()
        self.__receiver_vendors.load(receivers_db)

        self.__remote_vendors = model.RemoteVendorList()
        self.__remote_vendors.load(self.__remotes_db)

    def __setup_key_listener(self):
        '''
        Initialize the key-listener and related widgets.
        '''

        # pylint: disable-msg=W0201,E1101

        self.__image_preview_status = self.__ui.get_widget('image_preview_status')
        self.__label_preview_status = self.__ui.get_widget('label_preview_status')
        self.__label_preview_result = self.__ui.get_widget('label_preview_result')

        size_group = gtk.SizeGroup(gtk.SIZE_GROUP_VERTICAL)
        size_group.add_widget(self.__image_preview_status)
        size_group.add_widget(self.__label_preview_status)
        size_group.add_widget(self.__label_preview_result)

        self.__key_listener = lirc.KeyListener()
        self.__key_listener.connect('changed',     self.__on_lirc_changed)
        self.__key_listener.connect('key-pressed', self.__on_lirc_key_pressed)

    def __setup_device_area(self):
        '''
        Initialize widgets of the device selection area.
        '''

        # pylint: disable-msg=W0201

        self.__spinbutton_device = self.__ui.get_widget('spinbutton_device')
        self.__label_device      = self.__ui.get_widget('label_device')
        self.__combo_device      = self.__ui.get_widget('combo_device')
        self.__entry_device      = self.__combo_device.child

        size_group = gtk.SizeGroup(gtk.SIZE_GROUP_VERTICAL)
        size_group.add_widget(self.__spinbutton_device.get_parent())
        size_group.add_widget(self.__combo_device)

    def __setup_product_lists(self):
        '''
        Initialize widgets with product listings.
        '''

        # pylint: disable-msg=W0201

        self.__combo_receiver_product_list = self.__ui.get_widget('product-list')
        self.__combo_receiver_vendor_list = self.__ui.get_widget('vendor-list')
        self.__combo_receiver_vendor_list.set_model(self.__receiver_vendors)
        self.__combo_receiver_vendor_list.set_active(0)

        self.__combo_remote_product_list = self.__ui.get_widget('remote-product-list')
        self.__combo_remote_vendor_list = self.__ui.get_widget('remote-vendor-list')
        self.__combo_remote_vendor_list.set_model(self.__remote_vendors)
        self.__combo_remote_vendor_list.set_active(0)

        self.__radiobutton_supplied_remote = self.__ui.get_widget('radiobutton_supplied_remote')
        self.__radiobutton_other_remote = self.__ui.get_widget('radiobutton_other_remote')

    def __setup_authorization(self):
        '''
        Initialize authorization facilities.
        '''

        # pylint: disable-msg=W0201

        self.__auth = policykit.PolicyKitAuthentication()

        # Discover whether PolicyKit has already given us authorization
        # (without asking the user) so we can unlock the UI at startup if
        # necessary:

        granted = self.__auth.is_authorized()
        self._set_widgets_locked(not granted)

    def __restore_hardware_settings(self):
        '''
        Restore hardware settings from configuration files.
        '''

        assert 0 != self.__configuration_level

        settings = lirc.HardwareConfParser(config.LIRC_HARDWARE_CONF)

        receiver_vendor = settings.get('RECEIVER_VENDOR')
        receiver_model  = settings.get('RECEIVER_MODEL')
        receiver_device = settings.get('RECEIVER_DEVICE')

        remote_vendor = settings.get('REMOTE_VENDOR')
        remote_model  = settings.get('REMOTE_MODEL')

        # Try to select configured receiver vendor:

        model_iter = self.__receiver_vendors.find_iter(receiver_vendor)

        if model_iter:
            self.__combo_receiver_vendor_list.set_active_iter(model_iter)

        receiver_products = self.__combo_receiver_product_list.get_model()

        # Try to select configured receiver model:

        model_iter = receiver_products.find_iter(receiver_model)

        if model_iter:
            self.__combo_receiver_product_list.set_active_iter(model_iter)

        # Try to select configured device node

        self.selected_device = receiver_device or ''

        # Try to select configured remote vendor:

        model_iter = self.__remote_vendors.find_iter(remote_vendor)

        if model_iter:
            self.__combo_remote_vendor_list.set_active_iter(model_iter)

        remote_products = self.__combo_remote_product_list.get_model()

        # Try to select configured remote model:

        model_iter = remote_products.find_iter(remote_model)

        if model_iter:
            self.__combo_remote_product_list.set_active_iter(model_iter)

        # Toggle radio buttons to show, if restored remote is supplied remote.

        supplied_remote = (self.selected_receiver and
                           self.selected_receiver.find_supplied_remote(self.__remotes_db))

        if self.selected_remote is supplied_remote:
            self.__radiobutton_supplied_remote.set_active(True)

        else:
            self.__radiobutton_other_remote.set_active(True)

    def __on_lirc_changed(self, listener):
        '''
        Handle state changes of the LIRC key listener.
        '''

        if listener.connected:
            self.__label_preview_result.show()
            self.__label_preview_result.set_text(_('<none>'))
            self.__label_preview_status.set_text(_('Press remote control buttons to test:'))
            self.__image_preview_status.hide()

        else:
            self.__label_preview_result.hide()
            self.__label_preview_status.set_text(_('Remote control daemon not running. ' +
                                                   'Cannot test buttons.'))
            self.__image_preview_status.show()


    # pylint: disable-msg=W0613,R0913
    def __on_lirc_key_pressed(self, listener, remote, repeat, name, code):
        '''
        Handle key presses reported by the LIRC key listener.
        '''

        display_name = lirc.KeyCodes.get_display_name(name)
        display_name = gobject.markup_escape_text(display_name)
        self.__label_preview_result.set_markup('<b>%s</b>' % display_name)

    # pylint: disable-msg=C0103
    def _on_radiobutton_supplied_remote_toggled(self, radio_button = None):
        '''
        Gray-out the custom IR remote control widgets if the user wants to use
        the remote control supplied with the IR receiver:
        '''

        # Gray out the widgets if necessary:
        use_supplied = self.__radiobutton_supplied_remote.get_active()
        alignment = self.__ui.get_widget('remote-property-alignment')
        alignment.set_sensitive(not use_supplied)

    # pylint: disable-msg=C0103
    def _on_radiobutton_other_remote_size_allocate(self, widget, alloc):
        '''
        Keep padding of remote-property-alignment consistent
         with the padding implied by the radio buttons.
        '''

        xpad = widget.get_child().allocation.x - widget.allocation.x
        align = self.__ui.get_widget('remote-property-alignment')
        align.set_padding(0, 0, xpad, 0)

    def _on_vendor_list_changed(self, vendor_list):
        '''
        Change the combobox to show the list of models for the selected 
        manufacturer:
        '''

        tree_iter = vendor_list.get_active_iter()
        vendors = vendor_list.get_model()
        products, = vendors.get(tree_iter, 1)

        self.__combo_receiver_product_list.set_model(products)
        self.__combo_receiver_product_list.set_active(0)

    def __setup_devices_model(self, device_nodes):
        '''
        Populate the combo box for device with device nodes.
        '''

        self.__combo_device.show()
        self.__spinbutton_device.hide()
        self.__label_device.set_text_with_mnemonic(_('_Device:'))
        self.__label_device.set_mnemonic_widget(self.__combo_device)

        if device_nodes:
            # populate device combo's item list
            device_node_model = self.__combo_device.get_model()

            if not device_node_model:
                device_node_model = gtk.ListStore(gobject.TYPE_STRING)
                self.__combo_device.set_model(device_node_model)
                self.__combo_device.set_text_column(0)

            else:
                device_node_model.clear()

            for node in device_nodes:
                tree_iter = device_node_model.append()
                device_node_model.set(tree_iter, 0, node)

            # activate first device node
            self.__combo_device.set_sensitive(True)
            self.__combo_device.set_active(0)

        else:
            # deactivate combo box, if device is not configurable
            self.__combo_device.set_sensitive(False)
            self.selected_device = ''


    def __setup_numeric_device(self, device_node):
        '''
        Initialize the spin-button for numeric device selection.
        '''

        device_node = self.__hardware_manager.parse_numeric_device_node(device_node)
        label, value, minimum, maximum = device_node

        label = label and '%s:' % label or _('_Device:')

        self.__combo_device.hide()
        self.__spinbutton_device.show()

        self.__label_device.set_text_with_mnemonic(label)
        self.__label_device.set_mnemonic_widget(self.__spinbutton_device)

        self.__spinbutton_device.set_range(minimum, maximum)
        self.__spinbutton_device.set_value(value)

        self._on_spinbutton_device_value_changed()

    # pylint: disable-msg=C0103,W0613
    def _on_spinbutton_device_value_changed(self, spinbutton=None):
        '''
        Handle changes to the spin-button for numeric device selection.
        '''

        self.selected_device = ('%d' % self.__spinbutton_device.get_value())

    def _on_product_list_changed(self, product_list):
        '''
        Handle selection changes to receiver product list.
        '''

        # lookup selection:
        tree_iter = product_list.get_active_iter()
        receiver, = product_list.get_model().get(tree_iter, 1)

        if receiver:
            self._begin_update_configuration()

        try:
            # resolve device nodes:
            device_nodes = receiver and receiver.device_nodes or None

            if device_nodes:
                device_nodes = [node.strip() for node in device_nodes.split(',')]
                device_nodes = self.__hardware_manager.resolve_device_nodes(device_nodes)

            if 1 == len(device_nodes or []) and device_nodes[0].startswith('numeric:'):
                # show some spinbutton, when the device node requires numeric input
                self.__setup_numeric_device(device_nodes[0])

            else:
                # show the combobox, when device nodes can be choosen from list
                self.__setup_devices_model(device_nodes)

            # highlight supplied IR remote:
            supplied_remote = receiver and receiver.find_supplied_remote(self.__remotes_db)

            if supplied_remote:
                self.select_remote(supplied_remote)

        finally:
            # TODO: Also update configuration file on device-entry changes
            # TODO: Should we do anything with the configuration
            # if the remote or device is changed to ""/None?

            if receiver:
                self._end_update_configuration(receiver, self.selected_device)

    def _on_remote_vendor_list_changed(self, vendor_list):
        '''
        Handle selection changes to remote vendor list.
        '''

        tree_iter = vendor_list.get_active_iter()
        vendors = vendor_list.get_model()
        products, = vendors.get(tree_iter, 1)

        self.__combo_remote_product_list.set_model(products)
        self.__combo_remote_product_list.set_active(0)

    def _on_remote_product_list_changed(self, product_list):
        '''
        Handle selection changes to remote product list.
        '''

        tree_iter = product_list.get_active_iter()
        remote, = product_list.get_model().get(tree_iter, 1)

        if remote:
            self._begin_update_configuration()
            self._end_update_configuration(remote)

    def _on_search_progress(self, manager, fraction, message):
        '''
        Handle progress reports from the auto-detection process.
        '''

        self.__ui.get_widget('property-table').set_sensitive(False)
        self.__ui.get_widget('auto-detect-alignment').hide()
        self.__ui.get_widget('progress-box').show()

        auto_detect_progress = self.__ui.get_widget('auto-detect-progress')

        if fraction >= 0:
            auto_detect_progress.set_fraction(fraction)

        # gtk.ProgressBar doesn't support any text padding.
        # Work arround that visual glitch with spaces.
        auto_detect_progress.set_text(' %s ' % message)

    def _on_search_finished(self, manager = None):
        '''
        Handle end of the auto-detection process.
        '''

        self.__ui.get_widget('property-table').set_sensitive(True)
        self.__ui.get_widget('auto-detect-alignment').show()
        self.__ui.get_widget('progress-box').hide()

        n_receivers = self.__receiver_chooser.n_receivers

        if 1 == n_receivers:
            self._select_chosen_receiver()

        elif 0 == n_receivers:
            tree_iter = self.__receiver_vendors.get_iter_first()
            self.__combo_receiver_vendor_list.set_active_iter(tree_iter)

            responses = (
                (gtk.RESPONSE_CANCEL, gtk.STOCK_CANCEL),
                (gtk.RESPONSE_REJECT, _('_Search Again'), gtk.STOCK_REFRESH),
            )

            if (gtk.RESPONSE_REJECT ==
                show_message(self.__dialog,
                             _('No IR Receivers Found'),
                             _('Could not find any IR receiver. Is your device attached?\n\n' +
                             'Also notice that some devices, like homebrew serial port ' +
                             'receivers, must be selected manually, since there is no ' +
                             'way to detect them automatically.'),
                             buttons=responses)):
                self._on_auto_detect_button_clicked()

    def _on_receiver_found(self, manager, receiver, udi, device):
        '''
        Update the list of auto-detected receivers, when a new one was found.
        '''

        if self.__receiver_chooser.append(receiver, udi, device) > 1:
            self.__receiver_chooser.show()

    def _select_chosen_receiver(self):
        '''
        Choose the receiver selected in the auto-detection list.
        '''

        receiver, device = self.__receiver_chooser.selected_receiver

        self._begin_update_configuration()

        try:
            # highlight the selected receiver's vendor name:
            tree_iter = self.__receiver_vendors.find_iter(receiver.vendor)
            products = None

            if tree_iter:
                self.__combo_receiver_vendor_list.set_active_iter(tree_iter)
                products, = self.__receiver_vendors.get(tree_iter, 1)

            # highlight the selected receiver's product name:
            tree_iter = products and products.find_iter(receiver.product)

            if tree_iter:
                self.__combo_receiver_product_list.set_active_iter(tree_iter)

            # enter the selected receiver's device node:
            self.selected_device = (device or '')

        finally:
            self._end_update_configuration(receiver, device)

        # TODO: choose supplied remote after auto-detection

    def select_remote(self, remote):
        '''
        Select the specified remote control.
        '''

        vendor_name = remote.vendor or _('Unknown')
        product_name = remote.product or remote.name
        products = None

        tree_iter = self.__remote_vendors.find_iter(vendor_name)

        if tree_iter:
            self.__combo_remote_vendor_list.set_active_iter(tree_iter)
            products, = self.__remote_vendors.get(tree_iter, 1)

        tree_iter = products and products.find_iter(product_name)

        if tree_iter:
            self.__combo_remote_product_list.set_active_iter(tree_iter)

    def _begin_update_configuration(self):
        '''
        Temporarly prevent UI changes to be written back to configuration
        files. Must be paired with _end_update_configuration() call.
        '''

        if self.__configuration_level >= 0:
            self.__configuration_level += 1

    def _end_update_configuration(self, device, *args):
        '''
        Allow UI changes to be written back to configuration files again.
        Must be paired with _begin_update_configuration() call.
        '''

        if self.__configuration_level < 0:
            return

        self.__configuration_level -= 1

        try:
            mechanism = dbus.SystemBus().get_object('org.gnome.LircProperties.Mechanism', '/')

            while True:
                try:
                    if device:
                        # This can throw an AccessDeniedException exeption:
                        device.update_configuration(mechanism, *args)
                        device = None

                    if 0 == self.__configuration_level:
                        # This can throw an AccessDeniedException exeption:
                        mechanism.ManageLircDaemon('restart')

                        if self.__key_listener:
                            self.__key_listener.start()

                    break

                except dbus.DBusException, ex:
                    name = ex.get_dbus_name()

                    # The PolicyKit mechanism might complain that we have not yet
                    # requested authorization:
                    if name == 'org.gnome.LircProperties.AccessDeniedException':
                        #Request authorization from PolicyKit so we can try again.
                        granted = self._unlock()

                        if not granted:
                            # _unlock() already shows a dialog.
                            # (Though it is maybe not the right place to do that.)
                            show_message(self.__dialog,
                                         _('Cannot Update Configuration'),
                                         _('The System has refused access to this feature.'))
                            break # Stop trying.

                    elif name.startswith('org.gnome.LircProperties.'):
                        show_message(self.__dialog,
                                     _('Cannot Update Configuration'),
                                     _('Configuration mechanism reported %s.') % ex.message)
                        break # Stop trying.

                    else:
                        logging.error(ex)
                        break # Stop trying.

        except dbus.DBusException, ex:
            logging.error(ex)

    def _on_receiver_chooser_dialog_response(self, dialog, response):
        '''
        Handle confirmed selections in the chooser for auto-detected receivers.
        '''

        if gtk.RESPONSE_ACCEPT == response:
            self._select_chosen_receiver()

        dialog.hide()

    def _on_auto_detect_button_clicked(self, button=None):
        '''
        Handle clicks on the auto-detection button.
        '''

        # bring user interface to initial state:
        self._on_search_progress(None, 0, _('Searching remote controls...'))
        self.__receiver_chooser.reset()

        # start searching for supported remote controls:
        self.__hardware_manager.search_receivers()

    def _on_auto_detect_stop_button_clicked(self, button):
        '''
        Handle clicks on the auto-detection's "Cancel" button.
        '''

        self.__hardware_manager.cancel()

    def _on_custom_configuration_button_clicked(self, button):
        '''
        Handle clicks on the auto-detection's "Custom Configuration" button.
        '''

        if not self.__custom_configuration:
            self.__custom_configuration = CustomConfiguration(self.__ui)

        self.__custom_configuration.run(self.selected_receiver,
                                        self.selected_device,
                                        self.selected_remote)

    def _on_button_close_clicked(self, button):
        '''
        Handle clicks on the "Close" button.
        '''

        self.__dialog.hide()

    def _on_button_unlock_clicked(self, button):
        '''
        Handle clicks on the "Unlock" button.
        '''

        granted = self._unlock()

        if granted:
            self.__combo_receiver_vendor_list.grab_focus()

    def _unlock(self):
        '''
        Ask PolicyKit to allow the user to use our PolicyKit mechanism.
        We must ask PolicyKit again later before actually using the mechanism,
        but PolicyKit should only show the dialog the first time. Actually, use
        __auth.is_authorized() to check if we are already authorized, because
        ObtainAuthorization returns 0 if we are already authorized (which is
        probably a temporary bug)
        '''

        if self.__dialog == None:
            logging.warning('_unlock() called before the dialog ' +
                            'was instantiated, but we need an xid')
            return False

        if not self.__dialog.window:
            logging.warning('_unlock() called before the dialog ' +
                            'was realized, but we need an xid')
            return False

        granted = self.__auth.obtain_authorization(self.__dialog)

        # Warn the user (because PolicyKit does not seem to)
        # Note that PolicyKit can fail silently (just returning 0) when
        # something is wrong with the mechanism. And it fails (!) when
        # the user is _already_ authenticated.
        if not granted:
            # TODO: Improve this text
            # (PolicyKit should really show this instead of failing silently,
            # or at least something should be recommended by PolicyKit.):
            show_message(self.__dialog,
                         _('Could Not Unlock.'),
                         _('The system will not allow you to access ' +
                           'these features. Please contact your system ' +
                           'administrator for assistance.'))

        self._set_widgets_locked(not granted)

        return granted

    def _set_widgets_locked(self, locked):
        '''
        Gray (or ungray) widgets which require PolicyKit authorization.
        '''
        self.__ui.get_widget('vbox').set_sensitive(not locked)

        # Gray out the Unlock button if we are now already unlocked:
        button = self.__ui.get_widget('unlockbutton')
        button.set_sensitive(locked)

    # pylint: disable-msg=R0201
    def _on_button_help_clicked(self, button):
        '''
        Handle "Help" button clicks.
        '''

        # TODO: Implement help
        show_message(self.__dialog, _('Sorry, no help available yet.'))

    # pylint: disable-msg=R0201
    def _on_window_hide(self, window):
        '''
        React on the dialog getting hidden.
        '''

        # Stop the main loop so that run() returns:
        gtk.main_quit()

    def _on_close_keep_alive(self, dialog):
        '''
        Prevent that pygtk destroys the dialog on ESC.
        '''

        dialog.hide()
        return True

    def _on_lirc_properties_dialog_realize(self, dialog):
        '''
        Start service which need the dialog to be realized.
        '''

        self.__key_listener.start()

    def run(self):
        '''
        Show the dialog and return when the window is hidden.
        '''

        self.__dialog.connect('hide', self._on_window_hide)
        self.__dialog.show()

        gtk.main()

    def __get_selected_receiver(self):
        '''
        Retreive the currently selected receiver.
        '''

        tree_iter = self.__combo_receiver_product_list.get_active_iter()
        receiver, = self.__combo_receiver_product_list.get_model().get(tree_iter, 1)

        return receiver

    def __get_selected_remote(self):
        '''
        Retreive the currently selected remote.
        '''

        tree_iter = self.__combo_remote_product_list.get_active_iter()
        remote = None

        if tree_iter:
            remote, = self.__combo_remote_product_list.get_model().get(tree_iter, 1)

        return remote

    def __get_selected_device(self):
        '''
        Retreive the currently selected device.
        '''

        return self.__entry_device.get_text().strip()

    def __set_selected_device(self, device_node):
        '''
        Change the currently selected device.
        '''

        self.__entry_device.set_text(device_node)

    selected_receiver = property(__get_selected_receiver)
    selected_remote   = property(__get_selected_remote)
    selected_device   = property(__get_selected_device, __set_selected_device)

