/* GNOME DB libary
 * Copyright (C), 1999-2001 The Free Software Foundation
 *
 * AUTHORS:
 *	Rodrigo Moya <rodrigo@gnome-db.org>
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "gnome-db-dataset.h"
#include "gnome-db-util.h"
#include <libgnomeui/gnome-stock.h>

static void gnome_db_dataset_init (GnomeDbDataset *dset);
static void gnome_db_database_class_init (GnomeDbDatasetClass *klass);

static void gnome_db_dataset_destroy (GtkObject *object);

/*
 * GnomeDbDataset widget signals
 */
enum {
	GNOME_DB_DATASET_LAST_SIGNAL
};

static gint gnome_db_dataset_signals[GNOME_DB_DATASET_LAST_SIGNAL] = { 0 };

/*
 * Private functions
 */
static void
update_fields (GnomeDbDataset *dset)
{
	gint cnt;
	
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));
	g_return_if_fail(GDA_IS_RECORDSET(dset->recset));
	
	for (cnt = 0; cnt < gda_recordset_rowsize(dset->recset); cnt ++) {
		GtkWidget* entry;
		GdaField* field;
		GList*     node;
		
		field = gda_recordset_field_idx(dset->recset, cnt);
		node = g_list_nth(dset->entries, cnt);
		if (node) {
			entry = (GtkWidget *) node->data;
			if (GTK_IS_ENTRY(entry)) {
				gchar* str = gda_stringify_value(0, 0, field);
				gtk_entry_set_text(GTK_ENTRY(entry), str);
			}
		}
	}
}

/*
 * Callbacks
 */
static void
go_first_cb (GtkButton *button, GnomeDbDataset *dset)
{
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));
	g_return_if_fail(GDA_IS_RECORDSET(dset->recset));
	
	gda_recordset_move_first(dset->recset);
	update_fields(dset);
}

static void
go_previous_cb (GtkButton *button, GnomeDbDataset *dset)
{
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));
	g_return_if_fail(GDA_IS_RECORDSET(dset->recset));
	
	gda_recordset_move_prev(dset->recset);
	update_fields(dset);
}

static void
go_next_cb (GtkButton *button, GnomeDbDataset *dset)
{
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));
	g_return_if_fail(GDA_IS_RECORDSET(dset->recset));
	
	gda_recordset_move_next(dset->recset);
	update_fields(dset);
}

static void
go_last_cb (GtkButton *button, GnomeDbDataset *dset)
{
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));
	g_return_if_fail(GDA_IS_RECORDSET(dset->recset));
	
	gda_recordset_move_last(dset->recset);
	update_fields(dset);
}

/*
 * GnomeDbDataset widget interface
 */
static void
gnome_db_dataset_class_init (GnomeDbDatasetClass *klass)
{
	GtkObjectClass *object_class = GTK_OBJECT_CLASS(klass);
	
	object_class->destroy = gnome_db_dataset_destroy;
}

static void
gnome_db_dataset_init (GnomeDbDataset *dset)
{
	GtkWidget* frame;
	GtkWidget* table;
	
	frame = gnome_db_new_frame_widget(NULL);
	gtk_box_pack_start(GTK_BOX(dset), frame, 1, 1, 0);

	table = gnome_db_new_table_widget(4, 2, FALSE);
	gtk_container_add(GTK_CONTAINER(frame), table);
	
	/* create top toolbar */
	dset->first_btn = gnome_db_new_button_widget_with_pixmap("", GNOME_STOCK_MENU_FIRST);
	gtk_signal_connect(GTK_OBJECT(dset->first_btn),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(go_first_cb),
	                   (gpointer) dset);
	gtk_table_attach(GTK_TABLE(table), dset->first_btn, 0, 1, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	dset->previous_btn = gnome_db_new_button_widget_with_pixmap("", GNOME_STOCK_MENU_BACK);
	gtk_signal_connect(GTK_OBJECT(dset->previous_btn),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(go_previous_cb),
	                   (gpointer) dset);
	gtk_table_attach(GTK_TABLE(table), dset->previous_btn, 1, 2, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	dset->next_btn = gnome_db_new_button_widget_with_pixmap("", GNOME_STOCK_MENU_FORWARD);
	gtk_signal_connect(GTK_OBJECT(dset->next_btn),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(go_next_cb),
	                   (gpointer) dset);
	gtk_table_attach(GTK_TABLE(table), dset->next_btn, 2, 3, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	dset->last_btn = gnome_db_new_button_widget_with_pixmap("", GNOME_STOCK_MENU_LAST);
	gtk_signal_connect(GTK_OBJECT(dset->last_btn),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(go_last_cb),
	                   (gpointer) dset);
	gtk_table_attach(GTK_TABLE(table), dset->last_btn, 3, 4, 0, 1, GTK_FILL, GTK_FILL, 3, 3);

	dset->frame = gnome_db_new_frame_widget(NULL);
	gtk_table_attach(GTK_TABLE(table), dset->frame, 0, 4, 1, 2,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 GTK_FILL | GTK_EXPAND | GTK_SHRINK,
	                 3, 3);

	/* initialize all fields */
	dset->recset = NULL;
	dset->entries = NULL;
}

GtkType
gnome_db_dataset_get_type (void)
{
	static GtkType db_dataset_type = 0;
	
	if (!db_dataset_type) {
		GtkTypeInfo db_dataset_info = {
			"GnomeDbDataset",
			sizeof (GnomeDbDataset),
			sizeof (GnomeDbDatasetClass),
			(GtkClassInitFunc) gnome_db_dataset_class_init,
			(GtkObjectInitFunc) gnome_db_dataset_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_dataset_type = gtk_type_unique(gtk_vbox_get_type(), &db_dataset_info);
	}
	return db_dataset_type;
}

/**
 * gnome_db_dataset_new
 * @recset: recordset to be shown
 *
 * Creates a new GnomeDbDataset widget, which provides a visual way of
 * traversing a #GdaRecordset object record-by-record
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_dataset_new (GdaRecordset *recset)
{
	GnomeDbDataset *dset;
	
	dset = GNOME_DB_DATASET(gtk_type_new(GNOME_DB_TYPE_DATASET));
	if (GDA_IS_RECORDSET(recset))
		gnome_db_dataset_set_recordset(dset, recset);
	return GTK_WIDGET(dset);
}

static void
gnome_db_dataset_destroy (GtkObject *object)
{
	GnomeDbDataset *dset;
	GtkObjectClass *parent_class;
	
	g_return_if_fail(GNOME_DB_IS_DATASET(object));
	
	dset = GNOME_DB_DATASET(object);

	if (GDA_IS_RECORDSET (dset->recset))
		gda_recordset_free (dset->recset);
	g_list_free(dset->entries);

	parent_class = gtk_type_class (gtk_vbox_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_dataset_get_recordset
 * @dset: a GnomeDbDataset widget
 *
 * Returns the #GdaRecordset associated with the given dataset
 */
GdaRecordset *
gnome_db_dataset_get_recordset (GnomeDbDataset *dset)
{
	g_return_val_if_fail(GNOME_DB_IS_DATASET(dset), NULL);
	return dset->recset;
}

/**
 * gnome_db_dataset_set_recordset
 *
 * Associates the given #GdaRecordset object to a dataset widget. The dataset
 * widget does not destroy the old recordset being shown, you are responsible
 * for doing so
 */
void
gnome_db_dataset_set_recordset (GnomeDbDataset *dset, GdaRecordset *recset)
{
	gint cnt;
	
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));

	if (GDA_IS_RECORDSET (dset->recset))
		gda_recordset_free (dset->recset);
	gtk_object_ref (GTK_OBJECT (recset));
	dset->recset = recset;
	
	/* create all fields */
	if (GTK_IS_WIDGET(dset->container))
		gtk_widget_destroy(dset->container);
	if (GDA_IS_RECORDSET(dset->recset)) {
		dset->container = gnome_db_new_table_widget(2, gda_recordset_rowsize(dset->recset), FALSE);
		gtk_container_add(GTK_CONTAINER(dset->frame), dset->container);
		
		/* create a field and a entry for each field */
		for (cnt = 0; cnt < gda_recordset_rowsize(dset->recset); cnt++) {
			GtkWidget* label;
			GtkWidget* entry;
			GdaField* field = gda_recordset_field_idx(dset->recset, cnt);
			
			label = gnome_db_new_label_widget(gda_field_get_name(field));
			gtk_table_attach(GTK_TABLE(dset->container), label, 0, 1, cnt, cnt + 1,
			                 GTK_FILL, GTK_FILL, 3, 3);

			entry = gnome_db_new_entry_widget(0, FALSE);
			gtk_table_attach(GTK_TABLE(dset->container), entry, 1, 2, cnt, cnt + 1,
			                 GTK_FILL, GTK_FILL, 3, 3);
			dset->entries = g_list_append(dset->entries, (gpointer) entry);
		}
		
		/* show first record */
		gda_recordset_move_first(dset->recset);
		update_fields(dset);
	}
}

/**
 * gnome_db_dataset_set_show_buttons
 */
void
gnome_db_dataset_set_show_buttons (GnomeDbDataset *dset, gboolean show)
{
	g_return_if_fail(GNOME_DB_IS_DATASET(dset));

	if (show) {
		gtk_widget_show(dset->first_btn);
		gtk_widget_show(dset->previous_btn);
		gtk_widget_show(dset->next_btn);
		gtk_widget_show(dset->last_btn);
	}
	else {
		gtk_widget_hide(dset->first_btn);
		gtk_widget_hide(dset->previous_btn);
		gtk_widget_hide(dset->next_btn);
		gtk_widget_hide(dset->last_btn);
	}
}
