/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "config.h"
#include "gnome-db-list.h"
#include "gnome-db-util.h"

static void gnome_db_list_class_init (GnomeDbListClass *klass);
static void gnome_db_list_init (GnomeDbList *dblist);

static void gnome_db_list_destroy (GtkObject *object, gpointer user_data);

/*
 * GnomeDbList widget signals
 */
enum {
	GNOME_DB_LIST_LOADED,
	GNOME_DB_LIST_SELECT_ROW,
	GNOME_DB_LIST_ADD_ROW,
	GNOME_DB_LIST_LAST_SIGNAL
};

static gint gnome_db_list_signals[GNOME_DB_LIST_LAST_SIGNAL] = { 0, };

/*
 * Callbacks
 */
static void
recordset_destroyed_cb (GtkObject *object, gpointer user_data)
{
	GdaRecordset *recset = (GdaRecordset *) object;
	GnomeDbList *dblist = (GnomeDbList *) user_data;

	g_return_if_fail(GDA_IS_RECORDSET(recset));
	g_return_if_fail(GNOME_DB_IS_LIST(dblist));

	if (recset == dblist->recset) {
		/* stop timeout if active */
		if (dblist->timeout_handle != -1) {
			gtk_timeout_remove(dblist->timeout_handle);
			dblist->timeout_handle = -1;
		}

		dblist->recset = NULL;
		dblist->total_rows = 0;
		gnome_db_clear_clist(GTK_CLIST(dblist->list));
    }
}

static void
row_selected_cb (GtkCList *clist,
                 gint row,
                 gint col,
                 GdkEvent *event,
                 GnomeDbList *dblist)
{
	gchar *str = 0;

	g_return_if_fail(GNOME_DB_IS_LIST(dblist));
	g_return_if_fail(GTK_IS_CLIST(dblist->list));

	gtk_clist_get_text(clist, row, col, &str);
	if (str != 0) {
		gtk_signal_emit_by_name(GTK_OBJECT(dblist), "select_row");
    }
}

static gint
timeout_callback (GnomeDbList *dblist)
{
	gint position, cnt;
  
	g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
	g_return_val_if_fail(GTK_IS_CLIST(dblist->list), 0);
	g_return_val_if_fail(GDA_IS_RECORDSET(dblist->recset), 0);

	position = gda_recordset_move(dblist->recset, 1, 0);
	if (position != GDA_RECORDSET_INVALID_POSITION && !gda_recordset_eof(dblist->recset)) {
		gchar*    row[2] = { "", 0 };
		GdaField* field;
      
		field = gda_recordset_field_idx(dblist->recset, dblist->col);
		if (field) {
			row[1] = gda_stringify_value(0, 0, field);
			gtk_clist_freeze(GTK_CLIST(dblist->list));
			gtk_clist_append(GTK_CLIST(dblist->list), (gchar **) row);
			gtk_signal_emit(GTK_OBJECT(dblist),
							gnome_db_list_signals[GNOME_DB_LIST_ADD_ROW],
							dblist->total_rows);
			gtk_clist_thaw(GTK_CLIST(dblist->list));

			dblist->total_rows++;
		}
    }
	else {
		gtk_signal_emit_by_name(GTK_OBJECT(dblist), "loaded");
		dblist->timeout_handle = -1;
		return 0;
    }
	return 1;
}

/*
 * GnomeDbList widget interface
 */
static void
gnome_db_list_class_init (GnomeDbListClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;

	gnome_db_list_signals[GNOME_DB_LIST_LOADED] =
		gtk_signal_new("loaded", GTK_RUN_LAST, object_class->type,
					   GTK_SIGNAL_OFFSET(GnomeDbListClass, loaded),
					   gtk_signal_default_marshaller,
					   GTK_TYPE_NONE, 0);
	gnome_db_list_signals[GNOME_DB_LIST_SELECT_ROW] =
		gtk_signal_new("select_row", GTK_RUN_LAST, object_class->type,
					   GTK_SIGNAL_OFFSET(GnomeDbListClass, select_row),
					   gtk_signal_default_marshaller,
					   GTK_TYPE_NONE, 0);
	gnome_db_list_signals[GNOME_DB_LIST_ADD_ROW] =
		gtk_signal_new("add_row", GTK_RUN_LAST, object_class->type,
					   GTK_SIGNAL_OFFSET(GnomeDbListClass, add_row),
					   gtk_marshal_NONE__INT,
					   GTK_TYPE_NONE, 1, GTK_TYPE_INT);
	gtk_object_class_add_signals(object_class, gnome_db_list_signals,
								 GNOME_DB_LIST_LAST_SIGNAL);

	klass->loaded = 0;
	klass->select_row = 0;
	klass->add_row = 0;
	object_class->destroy = gnome_db_list_destroy;
}

static void
gnome_db_list_init (GnomeDbList *dblist)
{
	GtkWidget *scroll;

	dblist->recset = NULL;
	dblist->total_rows = 0;
	dblist->col = -1;
	dblist->timeout_handle = -1;

	scroll = gnome_db_new_scrolled_window_widget();
	dblist->list = gnome_db_new_clist_widget(0, 2);
	gtk_signal_connect(GTK_OBJECT(dblist->list), "select_row",
					   GTK_SIGNAL_FUNC(row_selected_cb), (gpointer) dblist);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scroll), 
										  dblist->list);
	gtk_box_pack_start(GTK_BOX(dblist), scroll, 1, 1, 0);
}

guint
gnome_db_list_get_type (void)
{
	static guint db_list_type = 0;

	if (!db_list_type) {
		GtkTypeInfo db_list_info = {
			"GnomeDbList",
			sizeof (GnomeDbList),
			sizeof (GnomeDbListClass),
			(GtkClassInitFunc) gnome_db_list_class_init,
			(GtkObjectInitFunc) gnome_db_list_init,
			(GtkArgSetFunc) NULL,
			(GtkArgGetFunc) NULL
		};
		db_list_type = gtk_type_unique(gtk_vbox_get_type(), &db_list_info);
    }
	return (db_list_type);
}

/**
 * gnome_db_list_new
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Creates a new GnomeDbList widget
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_list_new (GdaRecordset *recset, gint pos)
{
	GtkWidget *dblist = gtk_type_new(gnome_db_list_get_type());

	if (recset != 0)
		gnome_db_list_set_recordset(GNOME_DB_LIST(dblist), recset, pos);
	return (dblist);
}

static void
gnome_db_list_destroy (GtkObject *object, gpointer user_data)
{
	GtkObjectClass *parent_class;
	GnomeDbList *dblist;

	g_return_if_fail(GNOME_DB_IS_LIST(object));

	dblist = GNOME_DB_LIST(object);
	gnome_db_list_clear(dblist);

	if (GDA_IS_RECORDSET (dblist->recset))
		gda_recordset_free (dblist->recset);

	parent_class = gtk_type_class (gtk_vbox_get_type ());
	if (parent_class && parent_class->destroy)
		parent_class->destroy (object);
}

/**
 * gnome_db_list_clear
 * @dblist: GnomeDbList widget
 *
 * Clears the DbList widget. It clears all entries in the list and frees
 * the recordset associated
 */
void
gnome_db_list_clear (GnomeDbList *dblist)
{
	g_return_if_fail(GNOME_DB_IS_LIST(dblist));

	/* stop timeout if active */
	if (dblist->timeout_handle != -1) {
		gtk_timeout_remove(dblist->timeout_handle);
		dblist->timeout_handle = -1;
	}

	if (dblist->recset != 0) {
		gda_recordset_free(dblist->recset);
		dblist->recset = 0;
	}
	dblist->total_rows = 0;
	gnome_db_clear_clist(GTK_CLIST(dblist->list));
}

/**
 * gnome_db_list_get_row_count
 * @dblist: GnomeDbList widget
 *
 * Returns number of rows actually loaded in the recordset. As the list widget loads
 * data asynchronously, you may call this function before all rows have been fetched
 * from the database, in which case, it won't reflect the "real" number of rows
 * returned in the recordset
 *
 * Returns: number of rows in the recordset , or -1 on error
 */
gint
gnome_db_list_get_row_count (GnomeDbList *dblist)
{
	g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), -1);
	return dblist->total_rows;
}

/**
 * gnome_db_list_get_string
 * @dblist: GnomeDbList widget
 *
 * Returns the currently selected item's text. That is, the literal of the first
 * selected item
 *
 * Returns: a string containing the literal selected, or NULL if none is
 * selected or if there is an error
 */
gchar *
gnome_db_list_get_string (GnomeDbList *dblist)
{
	GList *selected;
  
	g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);

	selected = GTK_CLIST(dblist->list)->selection;
	if (selected) {
		gchar *name = NULL;

		guint row = GPOINTER_TO_UINT(selected->data);
		gtk_clist_get_text(GTK_CLIST(dblist->list), row, 1, &name);
		g_warning("name = %s", name);
		return (name);
	}

	return (0);
}

/**
 * gnome_db_list_get_recordset
 * @dblist: GnomeDbList widget
 *
 * Returns the GdaRecordset object associated with the given GnomeDbList
 * widget
 *
 * Returns: a pointer to the GdaRecordset object being used, or NULL if none
 * or if there is an error
 */
GdaRecordset *
gnome_db_list_get_recordset (GnomeDbList *dblist)
{
	g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);
	return (dblist->recset);
}

/**
 * gnome_db_list_set_recordset
 * @dblist: GnomeDbList widget
 * @recset: recordset to be shown
 * @pos: position of the field being shown
 *
 * Associates a GdaRecordset object with the given GnomeDbList
 * widget and shows its contents. The #GnomeDbList widget keeps
 * a reference to the #GdaRecordset object, so you should call
 * #gda_recordset_free after successfully calling this function
 * when you no longer need the recordset.
 */
void
gnome_db_list_set_recordset (GnomeDbList *dblist, GdaRecordset *recset, gint pos)
{
	gulong position;

	g_return_if_fail(GNOME_DB_IS_LIST(dblist));

	gnome_db_list_clear(dblist);

	if (GDA_IS_RECORDSET (dblist->recset))
		gda_recordset_free (dblist->recset);
	dblist->recset = recset;

	if (GDA_IS_RECORDSET(dblist->recset)) {
		gtk_object_ref (GTK_OBJECT (recset));
		gtk_signal_connect(GTK_OBJECT(dblist->recset),
						   "destroy",
						   GTK_SIGNAL_FUNC(recordset_destroyed_cb),
						   (gpointer) dblist);
	}
	dblist->total_rows = 0;
	dblist->col = pos;

	dblist->timeout_handle = gtk_timeout_add (15,
						  (GtkFunction) timeout_callback,
						  (gpointer) dblist);
}

/**
 * gnome_db_list_get_pixmap
 * @dblist: GnomeDbList widget
 * @row: row number
 *
 * Returns the pixmap being used for the given row
 */
GdkPixmap *
gnome_db_list_get_pixmap (GnomeDbList *dblist, gint row)
{
	GdkPixmap* pixmap;
	GdkBitmap* mask;

	g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);

	gtk_clist_get_pixmap(GTK_CLIST(dblist->list), row, 0, &pixmap, &mask);
	return (pixmap);
}

/**
 * gnome_db_list_set_pixmap
 * @dblist: GnomeDbList widget
 * @row: row number
 * @pixmap: GdkPixmap object
 *
 * Sets the pixmap to be shown for the given row
 */
void
gnome_db_list_set_pixmap (GnomeDbList *dblist,
                          gint row,
                          GdkPixmap *pixmap,
                          GdkBitmap *mask)
{
	g_return_if_fail(GNOME_DB_IS_LIST(dblist));

	gtk_clist_set_pixmap(GTK_CLIST(dblist->list),
			     row,
			     0,
			     pixmap,
			     mask);
}

/**
 * gnome_db_list_get_pixmap_mask
 * @dblist: GnomeDbList widget
 * @row: row number
 */
GdkBitmap *
gnome_db_list_get_pixmap_mask (GnomeDbList *dblist, gint row)
{
	GdkPixmap* pixmap;
	GdkBitmap* mask;

	g_return_val_if_fail(GNOME_DB_IS_LIST(dblist), 0);

	gtk_clist_get_pixmap(GTK_CLIST(dblist->list), row, 0, &pixmap, &mask);
	return (mask);
}
