/* GNOME DB library
 * Copyright (C) 1999-2000 Rodrigo Moya
 *
 * This Library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This Library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this Library; see the file COPYING.LIB.  If not,
 * write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gnome-db-util.h>
#include <config.h>

void
gnome_db_clear_clist (GtkCList *clist) {
	g_return_if_fail(GTK_IS_CLIST(clist));
	gtk_clist_clear(clist);
}

GnomePixmap *
gnome_db_get_pixmap (const gchar *stock_pixmap) {
	struct loaded_pixmap_info {
		gchar*       name;
		GnomePixmap* pixmap;
	};

	GList*                     node;
	struct loaded_pixmap_info* info;
	static GList*              loaded_pixmaps = NULL;
	
	g_return_val_if_fail(stock_pixmap != NULL, NULL);
	
	/* first, try to find if the pixmap is already loaded */
	node = g_list_first(loaded_pixmaps);
	while (node) {
		info = (struct loaded_pixmap_info *) node->data;
		if (info && !strcmp(info->name, stock_pixmap))
			return info->pixmap;
		node = g_list_next(node);
	}
	
	/* it's not loaded, so get it */
	info = g_new0(struct loaded_pixmap_info, 1);
	info->name = g_strdup(stock_pixmap);
	info->pixmap = gnome_stock_pixmap_widget(NULL, info->name);
	loaded_pixmaps = g_list_append(loaded_pixmaps, (gpointer) info);
	
	return info->pixmap;
}

void
gnome_db_show_about_dialog (const gchar *title) {
	const gchar *authors[] = { "Michael Lausch <michael@lausch.at>",
	                           "Rodrigo Moya <rodrigo@gnome-db.org>",
	                           "Stephan Heinze <stephan.heinze@xcom.de>",
	                           "Vivien Malerba <malerba@linuxave.net>",
	                           "Nick Gorham <nick@lurcher.org>",
	                           "Chris Wiegand <cwiegand@urgentmail.com>",
	                           "Alvaro del Castillo <acs@barrapunto.com>",
	                           "Akira TAGOH <tagoh@gnome-db.org>",
	                           "Carlos Perell Marn <carlos@gnome-db.org>",
	                           "Holger Thon <holger.thon@gnome-db.org>",
	                           "Joaqun Cuenca <e98cuenc@yahoo.com>",
	                           "Reinhard Mller <reinhard@gnue.org>",
	                           NULL };
	GtkWidget *about = gnome_about_new(title, VERSION,
	                                   _("Copyright The Free Software Foundation (C) 1998-2000"),
	                                   authors,
	                                   _("This program is part of the GNOME project for LINUX. "
	                                     "GNOME Data Access comes with ABSOLUTELY NO WARRANTY. "
	                                     "This is free software, and you are welcome to redistribute "
	                                     "it under the conditions of the GNU General Public Licence."),
	                                    "gnome-db.png");
	gtk_widget_show(about);
}

void
gnome_db_show_error (const gchar *format, ...) {
	va_list args;
	gchar sz[256];
	
	/* build the message string */
	va_start(args, format);
	vsprintf(sz, format, args);
	va_end(args);
	
	gnome_error_dialog(sz);
}

void
gnome_db_show_message (const gchar *format, ...) {
	va_list args;
	gchar sz[256];
	
	/* build the message string */
	va_start(args, format);
	vsprintf(sz, format, args);
	va_end(args);
	
	gnome_ok_dialog(sz);
}

void
gnome_db_show_file (const gchar *filename) {
	GtkWidget* dialog;
	GtkWidget* viewer;
	
	g_return_if_fail(filename != 0);
	
	dialog = gnome_dialog_new(filename, GNOME_STOCK_BUTTON_CLOSE, 0);
	viewer = gnome_less_new();
	gtk_widget_show(viewer);
	gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), viewer, 1, 1, 0);
	if (gnome_less_show_file(GNOME_LESS(viewer), filename)) {
		gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
	}
	else gnome_db_show_error(_("Could not open file\n%s"), filename);
}

void
gnome_db_set_bg_color (GtkWidget *w, gint red, gint green, gint blue) {
	GdkColormap* colormap;
	GdkColor     color;
	
	g_return_if_fail(GTK_IS_WIDGET(w));
	
	colormap = gdk_colormap_get_system();
	color.red = red;
	color.green = green;
	color.blue = blue;
	if (gdk_color_alloc(colormap, &color)) {
		GtkStyle *style, *style_default;
		
		style_default = gtk_widget_get_default_style();
		style = gtk_style_copy(style_default);
		style->base[GTK_STATE_NORMAL] = color;
		gtk_widget_set_style(w, style);
	}
	else g_warning(_("Could not allocate color"));
}

static gint
get_config_int (const gchar *format, ...) {
	gchar   buffer[2048];
	va_list args;
	
	g_return_val_if_fail(format, 0);
	
	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
	
	return gda_config_get_int(buffer);
}

static void
set_config_int (gint value, const gchar *format, ...) {
	gchar   buffer[2048];
	va_list args;
	
	g_return_if_fail(format);
	
	va_start(args, format);
	vsprintf(buffer, format, args);
	va_end(args);
	
	gda_config_set_int(buffer, value);
}

void
gnome_db_load_window_config (const gchar *name, GtkWidget *window) {
	g_return_if_fail(name != NULL);
	g_return_if_fail(GTK_IS_WIDGET(window));
	
	gdk_window_move(window->window,
	                get_config_int("/%s/Placement/%s_X",
	                               g_get_prgname(),
	                               name) + 20,
	                get_config_int("/%s/Placement/%s_Y",
                                 g_get_prgname(),
                                 name) + 20);
	gtk_widget_set_usize(window,
                       get_config_int("/%s/Placement/%s_Width",
                                      g_get_prgname(),
                                      name),
                       get_config_int("/%s/Placement/%s_Height",
                                      g_get_prgname(),
                                      name));
}

void
gnome_db_save_window_config (const gchar *name, GtkWidget *window) {
	gint x, y, width, height;
	
	g_return_if_fail(name != NULL);
	g_return_if_fail(GTK_IS_WIDGET(window));
	
	gdk_window_get_geometry(window->window, &x, &y, &width, &height, NULL);
	set_config_int(x, "/%s/Placement/%s_X", g_get_prgname(), name);
	set_config_int(y, "/%s/Placement/%s_Y", g_get_prgname(), name);
	set_config_int(width, "/%s/Placement/%s_Width", g_get_prgname(), name);
	set_config_int(height, "/%s/Placement/%s_Height", g_get_prgname(), name);
}

/*
 * File selection dialog
 */
static void
filesel_button_clicked (GtkWidget *w, glong *selected) {
	*selected = (glong) gtk_object_get_data(GTK_OBJECT(w), "button");
}

gchar *
gnome_db_select_file (const gchar *title) {
	GtkWidget* filesel;
	glong*     selected;
	gchar*     res = NULL;
	
	selected = (glong *) g_malloc(sizeof(glong));
	*selected = 0;
	
	/* create dialog */
	filesel = gtk_file_selection_new(title);
	gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
	                    "button",
	                    (gpointer) 1);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->ok_button),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(filesel_button_clicked),
	                   (gpointer) selected);
	gtk_object_set_data(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
	                    "button",
	                    (gpointer) 2);
	gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(filesel)->cancel_button),
	                   "clicked",
	                   GTK_SIGNAL_FUNC(filesel_button_clicked),
	                   (gpointer) selected);
	gtk_widget_show(filesel);
	gtk_file_selection_show_fileop_buttons(GTK_FILE_SELECTION(filesel));
	
	/* wait for selection from user */
	while (!*selected)
		gtk_main_iteration_do(TRUE);
	if (*selected == 1) /* OK button */ {
		res = g_strdup(gtk_file_selection_get_filename(GTK_FILE_SELECTION(filesel)));
	}
	else res = NULL;
	
	/* clean up */
	gtk_widget_destroy(filesel);
	g_free((gpointer) selected);
	return res;
}

void
gnome_db_submit_bug_cb (GtkWidget *w, gpointer data) {
	gchar* args[1];
	
	args[0] = "bug-buddy";
	if (gnome_execute_async(NULL, 1, args) == -1)
		gnome_db_show_error(_("Unable to execute Bug Report Tool"));
}
