/* GNOME DB library
 * Copyright (C) 1999 Rodrigo Moya
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#if defined(HAVE_CONFIG_H)
#  include <config.h>
#endif
#include "gnome-db.h"

static void gnome_db_grid_class_init (GnomeDbGridClass *);
static void gnome_db_grid_init       (GnomeDbGrid *);

/*
 * GnomeDbGrid widget signals
 */
enum
{
  GNOME_DB_GRID_ERROR,
  GNOME_DB_GRID_LOADED,
  GNOME_DB_GRID_SELECT_ROW,
  GNOME_DB_GRID_UNSELECT_ROW,
  GNOME_DB_GRID_ADD_ROW,
  GNOME_DB_GRID_LAST_SIGNAL
};

static gint gnome_db_grid_signals[GNOME_DB_GRID_LAST_SIGNAL] = { 0 };

/*
 * GnomeDbGrid callbacks
 */
static void
clear_selection_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  gnome_db_grid_clear_selection(GNOME_DB_GRID(data));
}

static void
copy_cb (GtkWidget *w, gpointer data)
{
}

static void
drag_data_get_cb (GtkWidget *w,
		  GdkDragContext *context,
		  GtkSelectionData *selection_data,
		  guint info,
		  guint time)
{
  g_return_if_fail(GTK_IS_CLIST(w));

  /* FIXME: how to send the grid contents? */
}

static void
select_all_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  gnome_db_grid_select_all(GNOME_DB_GRID(data));
}

static void
select_row_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data)
{
  GnomeDbGrid *grid;
  
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  
  grid = GNOME_DB_GRID(data);
  gtk_signal_emit(GTK_OBJECT(grid), gnome_db_grid_signals[GNOME_DB_GRID_SELECT_ROW], row + 1);
}

static void
unselect_row_cb (GtkCList *clist, gint row, gint column, GdkEvent *event, gpointer data)
{
  GnomeDbGrid *grid;
  
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  
  grid = GNOME_DB_GRID(data);
  gtk_signal_emit(GTK_OBJECT(grid), gnome_db_grid_signals[GNOME_DB_GRID_UNSELECT_ROW], row + 1);
}

static void
hide_columns_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  gtk_clist_column_titles_hide(GTK_CLIST(GNOME_DB_GRID(data)->grid));
  GNOME_DB_GRID(data)->show_col_names = FALSE;
}

static void
refresh_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  
  /* FIXME: must pass GnomeDbGrid widget pointer */
  gnome_db_grid_refresh(GNOME_DB_GRID(data));
}

static void
show_columns_cb (GtkWidget *w, gpointer data)
{
  g_return_if_fail(GNOME_DB_IS_GRID(data));
  gtk_clist_column_titles_show(GTK_CLIST(GNOME_DB_GRID(data)->grid));
  GNOME_DB_GRID(data)->show_col_names = TRUE;
}

static void
hide_row_numbers_cb (GtkWidget *w, gpointer data)
{
  GnomeDbGrid *grid = GNOME_DB_GRID(data);
  
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
}

static void
show_row_numbers_cb (GtkWidget *w, gpointer data)
{
  GnomeDbGrid *grid = GNOME_DB_GRID(data);
  
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
}

static void
view_as_records_cb (GtkWidget *w, gpointer data)
{
  GtkWidget* dialog;
  GtkWidget* dset;

  g_return_if_fail(GNOME_DB_IS_GRID(data));

  dialog = gnome_dialog_new(_("Record View"), GNOME_STOCK_BUTTON_CLOSE, NULL);
  dset = gnome_db_dataset_new(gnome_db_grid_get_recordset(GNOME_DB_GRID(data)));
  gtk_widget_show(dset);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), dset, 1, 1, 0);

  gnome_dialog_run_and_close(GNOME_DIALOG(dialog));
}

/*
 * Signal handlers
 */
void
gnome_db_grid_error (GnomeDbGrid *grid)
{
}

/*
 * Private functions
 */
GnomeUIInfo gridpopupmenu[] =
{
  { GNOME_APP_UI_ITEM, N_("Copy"), N_("Copy selection to clipboard"),
    copy_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_COPY, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Select all"), N_("Select all rows"),
    select_all_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
    NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Clear selection"), N_("Clear all selected rows"),
    clear_selection_cb, NULL, NULL, GNOME_APP_PIXMAP_NONE,
    NULL, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Record view..."), N_("View recordset record-by-record"),
    view_as_records_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_FONT, 0, 0, NULL },
  GNOMEUIINFO_SEPARATOR,
  { GNOME_APP_UI_ITEM, N_("Refresh grid"), N_("Refresh grid contents"),
    refresh_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_REFRESH, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Print grid"), N_("Print grid contents"),
    NULL, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_PRINT, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Show columns"), N_("Show column titles"),
    show_columns_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_TOP, 0, 0, NULL },
  { GNOME_APP_UI_ITEM, N_("Hide columns"), N_("Hide column titles"),
    hide_columns_cb, NULL, NULL, GNOME_APP_PIXMAP_STOCK,
    GNOME_STOCK_MENU_BOTTOM, 0, 0, NULL },
  GNOMEUIINFO_END
};

static GtkWidget *
create_grid (gint cnt, gchar *titles[], GnomeDbGrid *parent)
{
  GtkWidget *grid, *popup_menu;

  /* create widget */
  grid = gnome_db_new_clist_widget(titles, cnt);
  if (parent->show_col_names)
    gtk_clist_column_titles_show(GTK_CLIST(parent->grid));
  else gtk_clist_column_titles_hide(GTK_CLIST(parent->grid));

  gtk_clist_set_selection_mode(GTK_CLIST(grid), GTK_SELECTION_MULTIPLE);
  gtk_signal_connect(GTK_OBJECT(grid),
		     "select_row",
                     GTK_SIGNAL_FUNC(select_row_cb),
		     (gpointer) parent);
  gtk_signal_connect(GTK_OBJECT(grid),
		     "unselect_row",
                     GTK_SIGNAL_FUNC(unselect_row_cb),
		     (gpointer) parent);

  /* drag'n'drop */
  gtk_signal_connect(GTK_OBJECT(grid),
		     "drag_data_get",
		     GTK_SIGNAL_FUNC(drag_data_get_cb),
		     NULL);

  /* set GtkCList properties */
  gtk_widget_show(grid);
  gtk_container_add(GTK_CONTAINER(parent->scrolled_window), grid);
  gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(parent->scrolled_window),
                                 GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
                                 
  /* create the attached popup menu */
  popup_menu = gnome_db_new_popup_menu(grid, gridpopupmenu, (gpointer) parent);
  return (grid);
}

static void
create_empty_grid (GnomeDbGrid *grid)
{
  if (grid->grid != NULL)
    gtk_widget_destroy(grid->grid);
  grid->grid = create_grid(1, 0, grid);
  grid->total_rows = 0;
}

static gint
timeout_callback (GnomeDbGrid *grid)
{
  gint position, cnt;
  
  g_return_val_if_fail(GNOME_DB_IS_GRID(grid), 0);
  g_return_val_if_fail(GTK_IS_CLIST(grid->grid), 0);
  g_return_val_if_fail(IS_GDA_RECORDSET(grid->recset), 0);
  
  position = gda_recordset_move(grid->recset, 1, 0);
  if (position != GDA_RECORDSET_INVALID_POSITION && !gda_recordset_eof(grid->recset))
    {
      gchar *str;
      GPtrArray *row = g_ptr_array_new();

      gtk_clist_freeze(GTK_CLIST(grid->grid));
      for (cnt = 0; cnt < gda_recordset_rowsize(grid->recset); cnt++)
        {
          Gda_Field *field = gda_recordset_field_idx(grid->recset, cnt);
          g_ptr_array_add(row, (gpointer) gda_stringify_value(0, 0, field));
        }
      gtk_clist_append(GTK_CLIST(grid->grid), (gchar **) row->pdata);
      gtk_signal_emit(GTK_OBJECT(grid),
                      gnome_db_grid_signals[GNOME_DB_GRID_ADD_ROW],
                      grid->total_rows);
      gtk_clist_thaw(GTK_CLIST(grid->grid));
      g_ptr_array_free(row, TRUE);
      
      /* move to next row */
      str = g_strdup_printf("%d", grid->total_rows + 1);
      g_free((gpointer) str);
      grid->total_rows++;
    }
  else
    {
      gtk_signal_emit_by_name(GTK_OBJECT(grid), "loaded");
      grid->timeout_handle = -1;
      return (0);
    }
  return (1);
}

static void
show_recordset (GnomeDbGrid *grid, Gda_Recordset *recset)
{
  gulong position;
  register gint cnt, total_fields = 0;

  if (grid->grid != 0)
    {
      gtk_widget_destroy(grid->grid);
      grid->grid = 0;
    }
  grid->recset = recset;
  if (!recset)
    {
      grid->grid = 0;
      create_empty_grid(grid);
      gtk_signal_emit_by_name(GTK_OBJECT(grid), "loaded");
      return;
    }
    
  /* create grid */
  {
    GPtrArray *columns = g_ptr_array_new();
    total_fields = gda_recordset_rowsize(recset);
    if (total_fields > 0)
      {
        for (cnt = 0; cnt < total_fields; cnt++)
          {
            gchar *col_name;
            col_name = gda_field_name(gda_recordset_field_idx(recset, cnt));
            if (col_name != NULL)
              g_ptr_array_add(columns, (gpointer) col_name);
          }
        grid->grid = create_grid(total_fields, (gchar **) columns->pdata, 
                                 grid);

        /* column justification */
        for (cnt = 0; cnt < total_fields; cnt++)
          {
            switch (gda_field_type(gda_recordset_field_idx(recset, cnt)))
              {
                case GDA_TypeBigint :
                case GDA_TypeCurrency :
                case GDA_TypeDecimal :
                case GDA_TypeDouble :
                case GDA_TypeInteger :
                case GDA_TypeNumeric :
                case GDA_TypeSingle :
                case GDA_TypeSmallint :
                case GDA_TypeTinyint :
                case GDA_TypeUBigint :
                case GDA_TypeUSmallint :
                  gtk_clist_set_column_justification(GTK_CLIST(grid->grid), cnt,
                                                     GTK_JUSTIFY_RIGHT);
                  break;
                default :
                  gtk_clist_set_column_justification(GTK_CLIST(grid->grid), cnt,
                                                     GTK_JUSTIFY_LEFT);
                  break;
              }
          }

	grid->total_rows = 0;
	grid->timeout_handle = gtk_timeout_add(15, (GtkFunction) timeout_callback,
					       (gpointer) grid);
	gtk_clist_moveto(GTK_CLIST(grid->grid), 0, 0, 0.0, 0.0);
      }
    else
      {
	create_empty_grid(grid);
	grid->timeout_handle = -1;
      }
    g_ptr_array_free(columns, TRUE);
  }
}

/*
 * GnomeDbGrid widget interface
 */
static void
gnome_db_grid_class_init (GnomeDbGridClass *klass)
{
  GtkObjectClass *object_class = (GtkObjectClass *) klass;

  gnome_db_grid_signals[GNOME_DB_GRID_ERROR] = 
           gtk_signal_new ("error", GTK_RUN_FIRST, object_class->type,
                           GTK_SIGNAL_OFFSET(GnomeDbGridClass, error),
                           gtk_signal_default_marshaller, 
                           GTK_TYPE_NONE, 0);
  gnome_db_grid_signals[GNOME_DB_GRID_LOADED] =
           gtk_signal_new("loaded", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbGridClass, loaded),
                          gtk_signal_default_marshaller,
                          GTK_TYPE_NONE, 0);
  gnome_db_grid_signals[GNOME_DB_GRID_SELECT_ROW] =
           gtk_signal_new("select_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbGridClass, select_row),
                          gtk_marshal_NONE__INT,
                          GTK_TYPE_NONE, 1, GTK_TYPE_INT);
  gnome_db_grid_signals[GNOME_DB_GRID_UNSELECT_ROW] =
           gtk_signal_new("unselect_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbGridClass, unselect_row),
                          gtk_marshal_NONE__INT,
                          GTK_TYPE_NONE, 1, GTK_TYPE_INT);
  gnome_db_grid_signals[GNOME_DB_GRID_ADD_ROW] =
           gtk_signal_new("add_row", GTK_RUN_LAST, object_class->type,
                          GTK_SIGNAL_OFFSET(GnomeDbGridClass, add_row),
                          gtk_marshal_NONE__INT,
                          GTK_TYPE_NONE, 1, GTK_TYPE_INT);
  gtk_object_class_add_signals(object_class, gnome_db_grid_signals,
                               GNOME_DB_GRID_LAST_SIGNAL);

  klass->error = gnome_db_grid_error;
  klass->loaded = 0;
  klass->select_row = 0;
  klass->unselect_row = 0;
  klass->add_row = 0;
  object_class->destroy = gnome_db_grid_destroy;
}

static void
gnome_db_grid_init (GnomeDbGrid *grid)
{ 
  gtk_container_set_border_width(GTK_CONTAINER(grid), 5);
  
  /* create the scrolled window */
  grid->scrolled_window = gnome_db_new_scrolled_window_widget();
  gtk_box_pack_start(GTK_BOX(grid), grid->scrolled_window, 1, 1, 0);
  gtk_widget_show(grid->scrolled_window);

  grid->grid = 0;
  grid->recset = 0;
  grid->total_rows = 0;
  grid->show_row_numbers = TRUE;
  grid->show_col_names = TRUE;
  grid->timeout_handle = -1;
}

guint
gnome_db_grid_get_type (void)
{
  static guint db_grid_type = 0;
  
  if (!db_grid_type)
    {
      GtkTypeInfo db_grid_info =
      {
        "GnomeDbGrid",
        sizeof (GnomeDbGrid),
        sizeof (GnomeDbGridClass),
        (GtkClassInitFunc) gnome_db_grid_class_init,
        (GtkObjectInitFunc) gnome_db_grid_init,
        (GtkArgSetFunc) NULL,
        (GtkArgGetFunc) NULL
      };
      db_grid_type = gtk_type_unique(gtk_vbox_get_type(), &db_grid_info);
    }
  return (db_grid_type);
}

/**
 * gnome_db_grid_new
 * @recset: recordset to be shown
 *
 * Creates a new GnomeDbGrid widget, which displays the given
 * recordset, if given
 *
 * Returns: a pointer to the new widget, or NULL on error
 */
GtkWidget *
gnome_db_grid_new (Gda_Recordset *recset)
{
  GtkWidget *grid;
  grid = gtk_type_new(gnome_db_grid_get_type());
  
  if (recset == NULL)
    create_empty_grid(GNOME_DB_GRID(grid));
  else show_recordset(GNOME_DB_GRID(grid), recset);

  return (grid);
}

/**
 * gnome_db_destroy
 * @grid: the grid to be destroyed
 *
 * Destroys the given grid, freeing any data associated with it, included
 * the Gda_Recordset object
 */
void
gnome_db_grid_destroy (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  
  /* stop timeout if active */
  if (grid->timeout_handle != -1)
    gtk_timeout_remove(grid->timeout_handle);
    
  /* free all memory */
  if (grid->recset != NULL)
    {
      gda_recordset_close(grid->recset);
      gda_recordset_free(grid->recset);
    }
  gtk_widget_destroy(grid->grid);
  gtk_widget_destroy(grid->scrolled_window);
}

/**
 * gnome_db_grid_set_background
 * @grid: the grid widget
 * @row: row number
 * @color: color to be used as background
 *
 * Sets the background color of an entire row. The "color" parameter
 * must already be allocated
 */
void
gnome_db_grid_set_background (GnomeDbGrid *grid, gint row, GdkColor *color)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  g_return_if_fail(color != 0);
  
  gtk_clist_set_background(GTK_CLIST(grid->grid), row, color);
}

/**
 * gnome_db_grid_set_foreground
 * @grid: the grid widget
 * @row: row number
 * @color: color to be used as foreground
 *
 * Sets the foreground color of an entire row. The "color" parameter
 * must already be allocated
 */
void
gnome_db_grid_set_foreground (GnomeDbGrid *grid, gint row, GdkColor *color)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  g_return_if_fail(color != 0);
  
  gtk_clist_set_foreground(GTK_CLIST(grid->grid), row, color);
}

/**
 * gnome_db_grid_select_all
 * @grid: the grid widget
 *
 * Selects all cells in the given grid
 */
void
gnome_db_grid_select_all (GnomeDbGrid *grid)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  gtk_clist_select_all(GTK_CLIST(grid->grid));
}

/**
 * gnome_db_grid_clear_selection
 * @grid: the grid widget
 *
 * Clears the current selection in the given grid
 */ 
void
gnome_db_grid_clear_selection (GnomeDbGrid *grid)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  gtk_clist_unselect_all(GTK_CLIST(grid->grid));
}

/**
 * gnome_db_grid_get_selected_rows
 * @grid: the grid widget
 *
 * Returns: the number of selected rows in the given grid widget
 */
gint
gnome_db_grid_get_selected_rows (GnomeDbGrid *grid)
{
  GList *selected;
  
  g_return_val_if_fail(GNOME_DB_IS_GRID(grid), -1);
  g_return_val_if_fail(GTK_IS_CLIST(grid->grid), -1);
  
  selected = GTK_CLIST(grid->grid)->selection;
  if (selected != 0) return (g_list_length(selected));
  return (0);
}

/**
 * gnome_db_grid_copy_clipboard
 * @grid: the grid widget
 *
 * Copies the current selection to the clipboard
 */
void
gnome_db_grid_copy_clipboard (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
}

/**
 * gnome_db_grid_get_cell
 * @grid: the grid widget
 * @col: column number
 * @row: row number
 *
 * Returns the text of the given cell
 */
const gchar *
gnome_db_grid_get_cell (GnomeDbGrid *grid, gint col, gint row)
{
  gchar *txt;
  
  g_return_val_if_fail(grid != 0, 0);
  g_return_val_if_fail(GNOME_DB_IS_GRID(grid), 0);

  gtk_clist_get_text(GTK_CLIST(grid->grid), row, col, &txt);
  return (txt);
}

/**
 * gnome_db_grid_set_column_visibility
 * @grid: the grid object
 * @col: column to be hidden/shown
 * @vis: hide (0) or show (1)
 *
 * Shows or hides a column in the given grid widget. The default is to show
 * all columns
 */
void
gnome_db_grid_set_column_visibility (GnomeDbGrid *grid, gint col, gint vis)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  gtk_clist_set_column_visibility(GTK_CLIST(GNOME_DB_GRID(grid)->grid), col, vis);
}

/**
 * gnome_db_grid_set_row_visibility
 * @grid: the grid object
 * @row: row to be hidden/shown
 * vis: hide (0) or show (1)
 *
 * Shows or hides a row in the given grid widget. The default is to show
 * all rows
 */
void
gnome_db_grid_set_row_visibility (GnomeDbGrid *grid, gint row, gint vis)
{
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
}

/**
 * gnome_db_grid_hide_column_titles
 * @grid: the grid widget
 *
 * Hides columns in the given grid
 */
void
gnome_db_grid_hide_column_titles (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  hide_columns_cb(GTK_WIDGET(grid), grid);
}

/**
 * gnome_db_grid_show_column_titles
 * @grid: the grid object
 *
 * Shows columns in the given grid
 */
void
gnome_db_grid_show_column_titles (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != 0);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  show_columns_cb(GTK_WIDGET(grid), grid);
}

/**
 * gnome_db_grid_hide_row_numbers
 * @grid: the grid widget
 *
 * Hides row numbers in the given grid
 */
void
gnome_db_grid_hide_row_numbers (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  hide_row_numbers_cb(GTK_WIDGET(grid), grid);
}

/**
 * gnome_db_grid_show_row_numbers
 * @grid: the grid widget
 *
 * Show row numbers in the given grid
 */
void
gnome_db_grid_show_row_numbers (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  show_row_numbers_cb(GTK_WIDGET(grid), grid);
}

/**
 * gnome_db_grid_clear
 * @grid: the grid widget
 *
 * Removes all rows from the given grid
 */
void
gnome_db_grid_clear (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
  create_empty_grid(grid);
}

/**
 * gnome_db_grid_refresh
 * @grid: the grid widget
 *
 * Refresh grid's contents
 */
void
gnome_db_grid_refresh (GnomeDbGrid *grid)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));
}

/**
 * gnome_db_grid_get_recordset
 * @grid: the grid widget
 *
 * Returns the Gda_Recordset object being used by this grid
 *
 * Returns: a pointer to the Gda_Recordsetobject, or NULL if none is being
 * used or if there is an error
 */
Gda_Recordset *
gnome_db_grid_get_recordset (GnomeDbGrid *grid)
{
  g_return_val_if_fail(grid != NULL, NULL);
  g_return_val_if_fail(GNOME_DB_IS_GRID(grid), NULL);
  return (grid->recset);
}

/**
 * gnome_db_grid_set_recordset
 * @grid: the grid widget
 * @recset: recordset to be displayed
 *
 * Sets the recordset to be displayed on the given grid
 */
void
gnome_db_grid_set_recordset (GnomeDbGrid *grid, Gda_Recordset *recset)
{
  g_return_if_fail(grid != NULL);
  g_return_if_fail(GNOME_DB_IS_GRID(grid));

  /* stop timeout if active */
  if (grid->timeout_handle != -1)
    {
      gtk_timeout_remove(grid->timeout_handle);
      grid->timeout_handle = -1;
    }
  
  if (recset == grid->recset && recset != NULL)
    gnome_db_grid_refresh(grid);
  else
    {
      if (grid->recset != NULL)
        {
          gda_recordset_close(grid->recset);
          gda_recordset_free(grid->recset);
        }
      show_recordset(grid, recset);
    }
}
