/*
 *  GMF: The GNOME Media Framework
 *
 *  Copyright (C) 1999 Elliot Lee
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  Author: Elliot Lee <sopwith@redhat.com>
 *
 */

#include <config.h>

#include <string.h>
#include <sys/types.h>
#include <stdio.h>
#include <unistd.h>
#include <dirent.h>

#include "libgmf.h"
#include "gmf-filterregistry.h"

#define FILTER_LISTING_PATH "/GMF/filters"
#define GMF_DEBUG

GMF_Filter
gmf_filter_activate(GoadServerList *slist, const char *goad_id)
{
  GMF_Filter retval;

  retval = goad_server_activate_with_id(slist, goad_id, 0, NULL);

  return retval;
}

static gboolean string_in_array(const char *string, const char **array)
{
  int i;
  for(i = 0; array[i]; i++) {
    if(!strcmp(string, array[i]))
      return TRUE;
  }

  return FALSE;
}

static GMF_Filter
filter_find_activation(GoadServerList *slist,
		       GMFFilterRegistry *reg, GSList *possibilities,
		       const char *intype,
		       const char *outtype)
{
  GSList *ltmp;
  char *buf, *ctmp;
  GMFFilterRegistryEntry *ent = NULL;
  char *outtypes[4];
  int i;

  buf = alloca(strlen(outtype));
  
  ctmp = strchr(outtype, '/');
  if(!ctmp) return CORBA_OBJECT_NIL;
  
  sprintf(buf, "%.*s/*", ctmp - outtype, outtype);

  outtypes[0] = (char *)outtype;
  outtypes[1] = buf;
  outtypes[2] = "*";
  outtypes[3] = NULL;

  for(i = 0; outtypes[i]; i++) {
    for(ltmp = possibilities; ltmp; ltmp = g_slist_next(ltmp)) {
      ent = (GMFFilterRegistryEntry *)ltmp->data;
      
      if(string_in_array(outtypes[i], (const char **)ent->output_types))
	goto out;
    }
  }

  return CORBA_OBJECT_NIL;

 out:
#ifdef GMF_DEBUG
  g_message("Activating %s for %s -> %s", ent->goad_id, intype, outtypes[i]);
#endif

  return gmf_filter_activate(slist, ent->goad_id);
}

GMF_Filter
gmf_filter_activate_for_types(GoadServerList *slist, GMFFilterRegistry *registry,
			      const char *intype, const char *outtype)
{
  GMFFilterRegistry *reg;
  GMF_Filter retval;
  GSList *ltmp;
  char *intypes[4];
  char *buf, *ctmp;
  int i;

  g_return_val_if_fail(intype, CORBA_OBJECT_NIL);
  g_return_val_if_fail(outtype, CORBA_OBJECT_NIL);

  reg = registry;
  if(!reg)
    reg = gmf_filter_registry_get();

  g_return_val_if_fail(reg, CORBA_OBJECT_NIL);

  intypes[0] = (char *)intype;

  buf = alloca(strlen(intype));
  ctmp = strchr(intype, '/');
  if(!ctmp) goto out;
  sprintf(buf, "%.*s/*", ctmp - intype, intype);
  intypes[1] = buf;

  intypes[2] = "*";
  intypes[3] = NULL;

  for(i = 0; intypes[i]; i++) {
    ltmp = g_hash_table_lookup(registry->by_input_type, intypes[i]);
    retval = filter_find_activation(slist, registry, ltmp, intypes[i], outtype);
    if(!CORBA_Object_is_nil(retval, NULL)) break;
  }

 out:
  if(!registry)
    gmf_filter_registry_free(reg);

  return retval;
}

static void
gmf_registry_entry_add_for_type(GHashTable *ht,
				char *type,
				GMFFilterRegistryEntry *ent)
{
  GSList *ltmp;
  ltmp = g_hash_table_lookup(ht, type);

  if(ltmp)
    g_hash_table_insert(ht, type, g_slist_prepend(ltmp, ent));
  else
    g_hash_table_insert(ht, g_strdup(type), g_slist_prepend(NULL, ent));
}

static void
gmf_filter_registry_read_file(GMFFilterRegistry *registry, const char *file, GString *tmpstr)
{
  gpointer iter;
  char *dummy;
  GMFFilterRegistryEntry *ent;
  int i, n;

  gnome_config_push_prefix(file);

  iter = gnome_config_init_iterator_sections(file);

  while((iter = gnome_config_iterator_next(iter, &dummy, NULL))) {
    ent = g_new0(GMFFilterRegistryEntry, 1);

    /* get info for entry */
    if(*file == '=')
      g_string_sprintf(tmpstr, "=%s/=goad_id", dummy);
    else
      g_string_sprintf(tmpstr, "%s/goad_id", dummy);

    ent->goad_id = gnome_config_get_string(tmpstr->str);
    g_assert(ent->goad_id && *ent->goad_id);

    if(*file == '=')
      g_string_sprintf(tmpstr, "=%s/=input_types", dummy);
    else
      g_string_sprintf(tmpstr, "%s/input_types", dummy);
    gnome_config_get_vector(tmpstr->str, &n, &ent->input_types);
    ent->input_types = g_realloc(ent->input_types, sizeof(char *) * (n+1));
    ent->input_types[n] = NULL;

    if(*file == '=')
      g_string_sprintf(tmpstr, "=%s/=output_types", dummy);
    else
      g_string_sprintf(tmpstr, "%s/goad_id", dummy);

    gnome_config_get_vector(tmpstr->str, &n, &ent->output_types);
    ent->output_types = g_realloc(ent->output_types, sizeof(char *) * (n+1));
    ent->output_types[n] = NULL;

    /* put entry into registry */
    registry->list = g_slist_append(registry->list, ent);
    g_hash_table_insert(registry->by_id, ent->goad_id, ent);
    
    for(i = 0; ent->input_types[i]; i++)
      gmf_registry_entry_add_for_type(registry->by_input_type,
				      ent->input_types[i], ent);
    
    for(i = 0; ent->output_types[i]; i++)
      gmf_registry_entry_add_for_type(registry->by_output_type,
				      ent->output_types[i], ent);
  }

  gnome_config_pop_prefix();
}

static void
gmf_filter_registry_read_dir(GMFFilterRegistry *registry, const char *dir)
{
  GString *tmpstr2;
  GString *tmpstr;
  struct dirent *dent;
  DIR *dirh;

  dirh = opendir(dir);
  if(!dirh) return;

  tmpstr = g_string_new(NULL);
  tmpstr2 = g_string_new(NULL);

  for(dent = readdir(dirh); dent; dent = readdir(dirh)) {
    register char *p;

    p = strrchr(dent->d_name, '.');
    if (!p || (strcmp (p, ".gmffilter") != 0))
      continue;

    g_string_sprintf(tmpstr2, "=%s/%s", dir, dent->d_name);
    gmf_filter_registry_read_file(registry, tmpstr2->str, tmpstr);
  }

  g_string_free(tmpstr, TRUE);
  g_string_free(tmpstr2, TRUE);
}

GMFFilterRegistry *
gmf_filter_registry_get()
{
  GMFFilterRegistry *retval;
  GString *tmpstr;
  char *ctmp;

  retval = g_new0(GMFFilterRegistry, 1);
  retval->by_id = g_hash_table_new(g_str_hash, g_str_equal);
  retval->by_output_type = g_hash_table_new(g_str_hash, g_str_equal);
  retval->by_input_type = g_hash_table_new(g_str_hash, g_str_equal);

  tmpstr = g_string_new(NULL);

  g_string_sprintf(tmpstr, "%s" FILTER_LISTING_PATH, gnome_user_dir);
  gmf_filter_registry_read_dir(retval, tmpstr->str);

  ctmp = gnome_unconditional_config_file(FILTER_LISTING_PATH);
  gmf_filter_registry_read_dir(retval, ctmp);
  g_free(ctmp);

  g_string_free(tmpstr, TRUE);

  return retval;
}

static void
gmf_filter_registry_entry_free(GMFFilterRegistryEntry *entry)
{
  g_free(entry->goad_id);
  g_strfreev(entry->input_types);
  g_strfreev(entry->output_types);
}

static gboolean
gmf_bytype_free(gchar *key, GSList *list)
{
  g_free(key);
  g_slist_free(list);

  return TRUE;
}

void
gmf_filter_registry_free(GMFFilterRegistry *registry)
{
  g_hash_table_foreach_remove(registry->by_input_type,
			      (GHRFunc)gmf_bytype_free,
			      registry);
  g_hash_table_foreach_remove(registry->by_output_type,
			      (GHRFunc)gmf_bytype_free,
			      registry);
  g_slist_foreach(registry->list,
		  (GFunc)gmf_filter_registry_entry_free,
		  registry);

  g_hash_table_destroy(registry->by_id);
  g_hash_table_destroy(registry->by_input_type);
  g_hash_table_destroy(registry->by_output_type);
  g_slist_free(registry->list);

  g_free(registry);
}
