/* Glom
 *
 * Copyright (C) 2010 Openismus GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this program; if not, write to the
71 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include "tests/test_selfhosting_utils.h"
#include <libglom/init.h>
#include <libglom/utils.h>
#include <libglom/db_utils.h>
#include <libglom/connectionpool.h>
#include <glib.h> //For g_assert()
#include <iostream>
#include <cstdlib> //For EXIT_SUCCESS and EXIT_FAILURE

Glom::Document document;

static bool check_get_extra_rows(const Glib::ustring& quote_char)
{
  //Try to get more rows than intended:
  const Gnome::Gda::Value value("Born To Run" + quote_char + " OR " + quote_char + "x" + quote_char + "=" + quote_char + "x");
  Glom::sharedptr<const Glom::Field> where_field = document.get_field("albums", "name");
  const Gnome::Gda::SqlExpr where_clause = 
    Glom::Utils::build_simple_where_expression("albums", where_field, value);
  
  Glom::Utils::type_vecLayoutFields fieldsToGet;
  Glom::sharedptr<const Glom::Field> field = document.get_field("albums", "album_id");
  Glom::sharedptr<Glom::LayoutItem_Field> layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);
  field = document.get_field("albums", "name");
  layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);

  const Glib::RefPtr<const Gnome::Gda::SqlBuilder> builder = 
    Glom::Utils::build_sql_select_with_where_clause("albums",
      fieldsToGet, where_clause);
  Glib::RefPtr<Gnome::Gda::DataModel> data_model = 
    Glom::DbUtils::query_execute_select(builder);
  if(!test_model_expected_size(data_model, 2, 0)) //No rows should be returned because the match value was stupid, if escaped properly.
  {
    std::cerr << "Failure: Unexpected data model size for query, with quote_char=" << quote_char << std::endl;
    return false;
  }

  return true;
}

static bool check_drop_table(const Glib::ustring& quote_char)
{
  //Try to drop the table in a second SQL statement:
  const Gnome::Gda::Value value("True Blue" + quote_char + "; DROP TABLE songs; --");
  Glom::sharedptr<const Glom::Field> where_field = 
    document.get_field("albums", "name");
  const Gnome::Gda::SqlExpr where_clause = 
    Glom::Utils::build_simple_where_expression("albums", where_field, value);
  
  Glom::Utils::type_vecLayoutFields fieldsToGet;
  Glom::sharedptr<const Glom::Field> field = document.get_field("albums", "album_id");
  Glom::sharedptr<Glom::LayoutItem_Field> layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);
  field = document.get_field("albums", "name");
  layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);

  const Glib::RefPtr<const Gnome::Gda::SqlBuilder> builder = 
    Glom::Utils::build_sql_select_with_where_clause("albums",
      fieldsToGet, where_clause);
  Glib::RefPtr<Gnome::Gda::DataModel> data_model = 
    Glom::DbUtils::query_execute_select(builder);
  if(!test_model_expected_size(data_model, 2, 0)) //No rows should be returned because the match value was stupid, if escaped properly.
  {
    std::cerr << "Failure: Unexpected data model size for query, with quote_char=" << quote_char << std::endl;
    return false;
  }

  if(!test_table_exists("songs", document))
  {
    std::cerr << "Failure: The table may have been dropped." << std::endl;
    return false;
  }

  return true;
}

static bool check_avoid_quotes_and_drop_table_with_false_value_type()
{
  //Try to drop the table in a second SQL statement,
  //by using a text value for a field whose type should not need quoting:
  const Gnome::Gda::Value value("1;DROP TABLE songs");
  Glom::sharedptr<const Glom::Field> where_field = 
    document.get_field("albums", "album_id");
  const Gnome::Gda::SqlExpr where_clause = 
    Glom::Utils::build_simple_where_expression("albums", where_field, value);
  
  Glom::Utils::type_vecLayoutFields fieldsToGet;
  Glom::sharedptr<const Glom::Field> field = document.get_field("albums", "album_id");
  Glom::sharedptr<Glom::LayoutItem_Field> layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);
  field = document.get_field("albums", "name");
  layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);

  const Glib::RefPtr<const Gnome::Gda::SqlBuilder> builder = 
    Glom::Utils::build_sql_select_with_where_clause("albums",
      fieldsToGet, where_clause);

  std::cout << "This test expects some std::cerr output about exceptions now:" << std::endl;
  
  //Glom::ConnectionPool::get_instance()->set_show_debug_output(true);

  bool result = false;
  Glib::RefPtr<Gnome::Gda::DataModel> data_model
    = Glom::DbUtils::query_execute_select(builder);
  if(!data_model)
  {
    result = true; //This should have failed because the value was of the wrong type.
  }
  else
  {
    //Allow this because it fails (correctly) with PostgreSQL but not with SQLite.
    //though even with SQLite there is quoting that prevents the SQL injection.
    result = true;
    //result = false;
    //std::cerr << G_STRFUNC << ": Failure: The SQL query should have failed." << std::endl;
  }

  //We should not get this far, but if we do, tell us more about what happened:
  if(!test_table_exists("songs", document))
  {
    std::cerr << "Failure: The table may have been dropped." << std::endl;
    return false;
  }

  //It should have failed earlier.
  return result;
}

static bool check_avoid_quotes_and_drop_table_with_false_field_type()
{
  //Try to drop the table in a second SQL statement,
  //by using a text value for a field whose type should not need quoting:
  const Gnome::Gda::Value value("\"Born To Run\";DROP TABLE songs");

  //Specify a field with incorrect type information:
  Glom::sharedptr<Glom::Field> where_field = 
    document.get_field("albums", "name");
  where_field->set_glom_type(Glom::Field::TYPE_NUMERIC);
  //const GType gda_type = Glom::Field::get_gda_type_for_glom_type(Glom::TYPE_NUMERIC); 

  const Gnome::Gda::SqlExpr where_clause = 
    Glom::Utils::build_simple_where_expression("albums", where_field, value);
 
  Glom::Utils::type_vecLayoutFields fieldsToGet;
  Glom::sharedptr<const Glom::Field> field = document.get_field("albums", "album_id");
  Glom::sharedptr<Glom::LayoutItem_Field> layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);
  field = document.get_field("albums", "name");
  layoutitem = Glom::sharedptr<Glom::LayoutItem_Field>::create();
  layoutitem->set_full_field_details(field);
  fieldsToGet.push_back(layoutitem);

  const Glib::RefPtr<const Gnome::Gda::SqlBuilder> builder = 
    Glom::Utils::build_sql_select_with_where_clause("albums",
      fieldsToGet, where_clause);

  Glib::RefPtr<Gnome::Gda::DataModel> data_model
    = Glom::DbUtils::query_execute_select(builder);
  if(!test_model_expected_size(data_model, 2, 0)) //No rows should be returned because the match value was stupid, if escaped properly.
  {
    std::cerr << "Failure: Unexpected data model size for query." << std::endl;
    return false;
  }

  if(!test_table_exists("songs", document))
  {
    std::cerr << "Failure: The table may have been dropped." << std::endl;
    return false;
  }

  return true;
}

static bool test(Glom::Document::HostingMode hosting_mode)
{
  const bool recreated = 
    test_create_and_selfhost("example_music_collection.glom", document, hosting_mode);
  if(!recreated)
  {
    std::cerr << "Recreation failed." << std::endl;
    return false;
  }

  if(!check_get_extra_rows("\""))
  {
    std::cerr << "Failure: check_get_extra_rows() failed." << std::endl;
    return false;
  }
  
  if(!check_get_extra_rows("'"))
  {
    std::cerr << "Failure: check_get_extra_rows() failed." << std::endl;
    return false;
  }

  if(!check_drop_table("\""))
  {
    std::cerr << "Failure: check_drop_table() failed." << std::endl;
    return false;
  }
  
  if(!check_drop_table("'"))
  {
    std::cerr << "Failure: check_drop_table() failed." << std::endl;
    return false;
  }

  if(!check_avoid_quotes_and_drop_table_with_false_value_type())
  {
    std::cerr << "Failure: check_avoid_quotes_and_drop_table_with_false_value_type() failed." << std::endl;
    return false;
  }

  if(!check_avoid_quotes_and_drop_table_with_false_field_type())
  {
    std::cerr << "Failure: check_avoid_quotes_and_drop_table_with_false_field_type() failed." << std::endl;
    return false;
  }

  test_selfhosting_cleanup();

  return true;
}

int main()
{
  Glom::libglom_init();

  if(!test(Glom::Document::HOSTING_MODE_POSTGRES_SELF))
  {
    std::cerr << "Failed with PostgreSQL" << std::endl;
    test_selfhosting_cleanup();
    return EXIT_FAILURE;
  }
  
  if(!test(Glom::Document::HOSTING_MODE_SQLITE))
  {
    std::cerr << "Failed with SQLite" << std::endl;
    test_selfhosting_cleanup();
    return EXIT_FAILURE;
  }

  Glom::libglom_deinit();

  return EXIT_SUCCESS;
}
