/* $Header: /cvs/gnome/gIDE/src/gI_todo.c,v 1.1 2000/03/21 23:36:12 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#ifdef HAVE_STRPTIME
#define _GNU_SOURCE
#endif

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <sys/types.h>

#include <gtk/gtk.h>
#include "gI_todo.h"

#include "structs.h"
#include "gI_project.h"
#include "gI_project_changelog.h"
#include "gI_common.h"

/****************************************************************************
 *
 *
 *
 ****************               TODO LIST           *************************
 *
 *
 *
 ****************************************************************************/

static const gchar *todo_status_strings[] = {
	        "Cancelled",
	        "To do",
	        "In progress",
	        "Done (still open)",
	        "Closed"
        };

/*********************************************************************
 *
 * Todo Items
 *
 * These are the individual items in the todo list.  They hold various
 * data, like timestamps etc.
 *********************************************************************/

/* Local functions declarations */

static void
todo_item_set_status_string(gI_TodoItem *todo);

static void
todo_item_set_time(gchar **todo_time,
                   gchar  *Time);
static gboolean
todo_getline(FILE *file,
             GString *buf);
static gI_TodoItem *
todo_item_read_item_from_file(FILE *file);
static void
todo_item_read_comments_from_file(FILE *file,
                                  gI_TodoItem *todo);

/* Implementation */


gI_TodoItem *
gI_todo_item_create(gchar *item,
                    gchar *time)
{
	gI_TodoItem *todo;

	g_return_val_if_fail(item != NULL, NULL);

	todo = g_new0(gI_TodoItem, 1);

	todo->item = g_new(gchar, strlen(item) + 1);
	strcpy(todo->item, item);

	todo_item_set_time(&(todo->openedTime), time);

	todo->todo_status = TODO_TODO;
	todo_item_set_status_string(todo);

	return todo;
}

void
gI_todo_item_destroy(gI_TodoItem *todo)
{
	gchar *temp;
	GList *list;

	g_return_if_fail(todo != NULL);

	g_free(todo->item);
	g_free(todo->openedTime);
	g_free(todo->closedTime);
	g_free(todo->dueDate);
	g_free(todo->assignedTo);

	list = todo->comments;
	while(list)
	{
		temp = list->data;
		g_free(temp);
		list = list->next;
	}

	g_free(todo);
}


void
gI_todo_item_increment(gI_TodoItem *todo)
{
	g_return_if_fail(todo != NULL);

	if(todo->todo_status >= TODO_DONE)
		return;

	todo->todo_status++;

	todo_item_set_status_string(todo);

	todo_item_set_time(&(todo->modifiedTime), NULL);
}

void
gI_todo_item_decrement(gI_TodoItem *todo)
{
	g_return_if_fail(todo != NULL);

	if(todo->todo_status <= TODO_CANCELLED)
		return;

	todo->todo_status--;

	todo_item_set_status_string(todo);

	todo_item_set_time(&(todo->modifiedTime), NULL);
}

void
gI_todo_item_close(gI_TodoItem *todo)
{
	g_return_if_fail(todo != NULL);

	todo->todo_status = TODO_CLOSED;

	todo_item_set_status_string(todo);

	todo_item_set_time(&(todo->modifiedTime), NULL);
	todo_item_set_time(&(todo->closedTime), NULL);
}

void
gI_todo_item_assign(gI_TodoItem *todo,
                    gchar       *who)
{
	g_return_if_fail(todo != NULL);

	todo->seen = FALSE;

	if(!who)
	{
		g_free(todo->assignedTo);
		todo->assignedTo = NULL;
	}
	else
	{
		todo->assignedTo = g_renew(gchar, todo->assignedTo, strlen(who) + 1);
		strcpy(todo->assignedTo, who);
	}
}

void
gI_todo_item_add_comment(gI_TodoItem *todo,
                         gchar *comment)
{
	gchar *buf;

	g_return_if_fail(todo);
	g_return_if_fail(comment);

	buf = g_new(gchar, strlen(comment) + 1);

	strcpy(buf, comment);

	todo->comments = g_list_append(todo->comments, buf);

	todo_item_set_time(&(todo->modifiedTime), NULL);
}

static void
todo_item_set_status_string(gI_TodoItem *todo)
{
	todo->status = todo_status_strings[todo->todo_status];
}

static void
todo_item_set_time(gchar **todo_time,
                   gchar  *Time)
{
	time_t *tm;
	gchar *temp;
	gint len;

	g_return_if_fail(todo_time != NULL);

	if(Time)
		temp = Time;
	else
	{
		tm = g_new(time_t, 1);
		time(tm);
		temp = ctime(tm);
	}

	len = strlen(temp);

	*todo_time = g_renew(gchar, *todo_time, len + 1);
	strcpy(*todo_time, temp);

	/* Ctime puts a newline in there */
	if(!Time)
		(*todo_time)[len - 1] = '\0';

}

/*********************************************************************
 *
 * Todo Item List
 *
 * This is a list of todo items.  We need a class for this, because
 * we need pointers to all attached Todo List Panes.  (IE, user
 * interfaces).  This is so we can link a sub-window back to the parent
 * and still let it be editable.
 *
 * Hence, we have to have a list of things that need to be updated.
 *
 *********************************************************************/

gI_TodoList *
gI_todo_list_create(void)
{
	return g_new0(gI_TodoList, 1);
}

void
gI_todo_list_destroy(gI_TodoList *tlist)
{
	GList *list;

	g_return_if_fail(tlist != NULL);

	list = tlist->todoItems;
	while(list)
	{
		gI_todo_item_destroy( (gI_TodoItem *) list->data);
		list = g_list_next(list);
	}

	g_free(tlist);
}

void
gI_todo_list_attach(gI_TodoList *tlist,
                    gI_TodoPane *tpane)
{
	g_return_if_fail(tlist != NULL);
	g_return_if_fail(tpane != NULL);

	tlist->todoPanes = g_list_append(tlist->todoPanes, tpane);
}

void
gI_todo_list_detach(gI_TodoList *tlist,
                    gI_TodoPane *tpane)
{
	g_return_if_fail(tlist != NULL);
	g_return_if_fail(tpane != NULL);

	tlist->todoPanes = g_list_remove(tlist->todoPanes, tpane);
}

inline void
gI_todo_list_add(gI_TodoList *tlist,
                 gI_TodoItem *todo)
{
	tlist->todoItems = g_list_append(tlist->todoItems, todo);
}

inline void
gI_todo_list_remove(gI_TodoList *tlist,
                    gI_TodoItem *todo)
{
	GList *tmp;

	tmp = g_list_find(tlist->todoItems, todo);

	g_return_if_fail(tmp != NULL);

	gI_todo_item_destroy( (gI_TodoItem *) tmp->data);

	tlist->todoItems = g_list_remove(tlist->todoItems, todo);
}


/* Print out the todo list to the specified file, using the specified
 * begin and end strings.
 */

gboolean
gI_todo_list_write_to_file(gI_TodoList *tlist,
                           FILE *file,
                           gchar *beginString,
                           gchar *endString)
{
	GString *buf;
	gI_TodoItem *todo;
	GList *items, *comments;
	gint a;

	g_return_val_if_fail(tlist, FALSE);
	g_return_val_if_fail(file, FALSE);
	g_return_val_if_fail(beginString, FALSE);
	g_return_val_if_fail(endString, FALSE);

	buf = g_string_new("");

	/* Print header to Todo List */

	fputs(beginString, file);
	fputs("\n", file);

	/* Print all Todo Items */

	items = tlist->todoItems;
	while(items)
	{
		todo = items->data;

		/* Item list */

		g_string_assign(buf, "$Begin Item\n");
		fputs(buf->str, file);

		/* Item */

		g_string_assign(buf, "Item: ");
		g_string_append(buf, todo->item);
		g_string_append_c(buf, '\n');

		fputs(buf->str, file);

		/* Status FIXME will fail if enum is changed */

		g_string_sprintf(buf, "Status: %d, %s\n",
		                 todo->todo_status,
		                 todo->status);
		fputs(buf->str, file);

		/* Times */

		g_string_sprintf(buf,
		                 "Times:\nOpened: %s\nModified: %s\nClosed: %s\nDue: %s\n",
		                 todo->openedTime,
		                 todo->modifiedTime ? todo->modifiedTime : "",
		                 todo->closedTime ? todo->closedTime : "",
		                 todo->dueDate ? todo->dueDate : "");
		fputs(buf->str, file);


		/* Assignment */

		g_string_sprintf(buf, "Assigned: %s\n",
		                 todo->assignedTo ? todo->assignedTo : "");
		fputs(buf->str, file);

		g_string_sprintf(buf, "Seen: %s\n", todo->seen ? "Yes" : "No");
		fputs(buf->str, file);


		/* Comments */

		g_string_assign(buf, "$Begin Comments:\n");
		fputs(buf->str, file);

		comments = todo->comments;
		while(comments)
		{
			g_string_assign(buf, comments->data);

			/* Comments might have newlines in them,
			 * so we replace the newline with a literal '\' 'n'
			 * and escape the backslashes */

			for(a = 0; a < buf->len; a++)
			{
				if(buf->str[a] == '\n')
				{
					g_string_erase(buf, a, 1);
					g_string_insert(buf, a, "\\n");
					a++;
				}
				else if(buf->str[a] == '\\')
				{
					g_string_insert_c(buf, a, '\\');
					a++;
				}
			}

			g_string_append_c(buf, '\n');
			fputs(buf->str, file);

			comments = g_list_next(comments);
		}

		g_string_assign(buf, "$End Comments\n");
		fputs(buf->str, file);

		/* End item */
		g_string_assign(buf, "$End Item\n");
		fputs(buf->str, file);

		items = g_list_next(items);
	}

	fputs(endString, file);
	fputs("\n", file);

	g_string_free(buf, FALSE);

	return TRUE;
}

/* Read in and create todo list from file.  The begin string must have
 * been seen by the calling function, so we only scan for the ending
 * string */

gI_TodoList *
gI_todo_list_read_from_file(FILE *file,
                            gchar *endString)
{
	gI_TodoList *tlist;
	gI_TodoItem *todo;
	GString *lineIn;

	g_return_val_if_fail(file, NULL);
	g_return_val_if_fail(file, NULL);

	lineIn = g_string_new("");
	tlist = gI_todo_list_create();

	while(todo_getline(file, lineIn) && strcmp(lineIn->str, endString))
	{
		if(strcmp(lineIn->str, "$Begin Item"))
			continue;

		todo = todo_item_read_item_from_file(file);

		if(todo)
			gI_todo_list_add(tlist, todo);
		else
			g_print("File IO Error reading Todo List!");
	}

	g_string_free(lineIn, FALSE);

	return tlist;
}

static gboolean
todo_getline(FILE *file,
             GString *buf)
{
	gchar c = 0;

	g_string_truncate(buf, 0);

	c = fgetc(file);
	while(!feof(file) && (c != '\n'))
	{
		g_string_append_c(buf, c);

		c = fgetc(file);
	}

	return !feof(file);
}

static gI_TodoItem *
todo_item_read_item_from_file(FILE *file)
{
	gI_TodoItem *todo;
	GString *lineIn;

	lineIn = g_string_new("");

	/* Item */

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Item: ", 6))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 6);

	todo = gI_todo_item_create(lineIn->str, "");

	/* Item Status */

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Status: ", 8))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 8);
	todo->todo_status = atoi(lineIn->str);
	todo_item_set_status_string(todo);

	/* Times */

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Times:", 6))
		return NULL;  /* Error! */

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Opened: ", 8))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 8);
	if(lineIn->len < 2)
		return NULL;  /* How can there be no creation date?! */
	todo_item_set_time(&(todo->openedTime), lineIn->str);

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Modified: ", 10))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 10);
	if(lineIn->len > 2)
		todo_item_set_time(&(todo->modifiedTime), lineIn->str);

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Closed: ", 8))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 8);
	if(lineIn->len > 2)
		todo_item_set_time(&(todo->modifiedTime), lineIn->str);

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Due: ", 5))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 5);
	if(lineIn->len > 2)
		todo_item_set_time(&(todo->dueDate), lineIn->str);


	/* Assigned To */


	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Assigned: ", 10))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 10);
	if(lineIn->len > 2)
		gI_todo_item_assign(todo, lineIn->str);


	/* Does this person know it's assigned? */


	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "Seen: ", 6))
		return NULL;  /* Error! */
	g_string_erase(lineIn, 0, 6);
	if(lineIn->str[0] == 'Y')
		todo->seen = TRUE;
	else
		todo->seen = FALSE;

	todo_getline(file, lineIn);
	if(strncmp(lineIn->str, "$Begin Comments", 15))
		return NULL;  /* Error! */

	todo_item_read_comments_from_file(file, todo);

	g_string_free(lineIn, FALSE);

	return todo;
}

static void
todo_item_read_comments_from_file(FILE *file,
                                  gI_TodoItem *todo)
{
	GString *lineIn;
	gint a;

	lineIn = g_string_new("");

	while(todo_getline(file, lineIn) &&
	        strncmp(lineIn->str, "$End Comments", 14))
	{
		/* Strip escape gchars */

		for(a = 0; a < (lineIn->len - 1); a++)
		{
			if(lineIn->str[a] == '\\')
			{
				if(lineIn->str[a + 1] == 'n')
				{
					g_string_erase(lineIn, a, 2);
					g_string_insert_c(lineIn, a, '\n');
				}
				else if(lineIn->str[a + 1] == '\\')
				{
					g_string_erase(lineIn, a, 1);
				}
				else
					break; /* ERROR! */
			}
		}

		gI_todo_item_add_comment(todo, lineIn->str);
	}

	g_string_free(lineIn, FALSE);
}

/*************************************************************************
 *
 *
 *
 ****************   TODO PANE - Draws todo list in a vbox ****************
 *
 *
 *
 *************************************************************************/


/* Local strings *****************************************************/

static const gchar *listFrameTitle = N_("gIDE Todo List");
static const gchar *listEditFrameTitle = N_("List");
static const gchar *itemEditFrameTitle = N_("Item");
static const gchar *miscFrameTitle = N_("Misc");
static const gchar *newItemFrameTitle = N_("Enter new item name & assignment:");
static const gchar *commentFrameTitle = N_("Add comment:");

static const gchar *listEditNewTitle = N_(" New Item ");
static const gchar *listEditEditTitle = N_(" Edit ");
static const gchar *listEditDeleteTitle = N_(" Delete ");

static const gchar *itemEditIncrementTitle = N_(" Inc. Status ");
static const gchar *itemEditDecrementTitle = N_(" Dec. Status ");
static const gchar *itemEditCloseTitle = N_(" Close Item ");

static const gchar *miscAddCommentTitle = N_(" Add Comment ");
static const gchar *miscRemoveCommentTitle = N_(" Remove ");
static const gchar *miscAssignTitle = N_(" Assign to: ");

static const gchar *newItemAdd = N_(" Apply to List ");
static const gchar *newItemCancel = N_(" Cancel ");

static const gchar *commentAdd = N_(" Add Comment ");

static gchar *todoCTreeTitles[] = { "Item", "Assigned", "Status" };

static const gchar *todoItemOpenedDate =   N_("Created:        ");
static const gchar *todoItemModifiedDate = N_("Modified:       ");
static const gchar *todoItemClosedDate =   N_("Closed:         ");
static const gchar *todoItemDueDate =      N_("Completion Due: ");
static const gchar *todoItemComments =     N_("Comments:");

#if 0
static const gchar *errorNeedNameTitle = N_("Error");
#endif
static const gchar *errorNeedName = N_("Please enter a name for your Todo item");
static const gchar *errorNeedComment = N_("Please enter a comment to add" );

static const gint todo_sort_order[LAST_TODO] = {
	        10, /* TODO_CANCELLED */
	        1, /* TODO_TODO */
	        2, /* TODO_INPROGRESS */
	        3, /* TODO_DONE */
	        9 /* TODO_CLOSED */
        };

/* Local functions defs **********************************************/

static GList *
todo_pane_ctree_get_todo_selected(gI_TodoPane *tpane);

static void
todo_pane_set_row_text(gI_TodoPane *tpane,
                       GtkCTreeNode *node,
                       gchar *rowData[]);
static void
todo_pane_pack_node_from_todo(gI_TodoPane *tpane,
                              TodoPaneData *tdata,
                              gI_TodoItem *todo);
/* Local Callbacks ***************************************************/

static gint
listEditNewButtonPress(GtkWidget *widget,
                       gpointer data);
static gint
listEditEditButtonPress(GtkWidget *widget,
                        gpointer data);
static gint
listEditDeleteButtonPress(GtkWidget *widget,
                          gpointer data);
static gint
itemIncrementButtonPress(GtkWidget *widget,
                         gpointer data);
static gint
itemDecrementButtonPress(GtkWidget *widget,
                         gpointer data);
static gint
itemCloseButtonPress(GtkWidget *widget,
                     gpointer data);
static gint
miscAddCommentButtonPress(GtkWidget *widget,
                          gpointer data);
static gint
miscRemoveCommentButtonPress(GtkWidget *widget,
                             gpointer data);
static gint
miscAssignButtonPress(GtkWidget *widget,
                      gpointer data);
static gint
newItemAddButtonPress(GtkWidget *widget,
                      gpointer data);
static gint
newItemCancelButtonPress(GtkWidget *widget,
                         gpointer data);
static gint
commentAddButtonPress(GtkWidget *widget,
                      gpointer data);
static gint
assignedColumnSortButtonPress(GtkWidget *widget,
                              gpointer data);
static gint
statusColumnSortButtonPress(GtkWidget *widget,
                            gpointer data);

static gint
statusColumnSortFunc(GtkCList *clist,
                     GtkCListRow *ptr1,
                     GtkCListRow *ptr2);
#ifdef HAVE_STRPTIME
static gint
itemColumnSortButtonPress(GtkWidget *widget,
                          gpointer data);
static gint
itemColumnSortFunc(GtkCList *clist,
                   GtkCListRow *ptr1,
                   GtkCListRow *ptr2);
#endif /* HAVE_STRPTIME */


/* General functions *************************************************/

gI_TodoPane *
gI_todo_pane_create(void)
{
	GtkWidget *temp, *temp2;

	gI_TodoPane *tpane;

	tpane = g_new0(gI_TodoPane, 1);

	/* Top level elements */

	tpane->vbox = gtk_vbox_new(FALSE, FALSE);

	tpane->listFrame = gtk_frame_new( listFrameTitle );
	gtk_box_pack_start(GTK_BOX(tpane->vbox), tpane->listFrame,
	                   TRUE, TRUE, 0);

	tpane->ctree = gtk_ctree_new_with_titles( 3, 0, todoCTreeTitles );
	gtk_signal_connect(GTK_OBJECT(
	                       ((GtkCListColumn)(GTK_CLIST(tpane->ctree)->column[1]))
	                       .button), "clicked",
	                   GTK_SIGNAL_FUNC(assignedColumnSortButtonPress),
	                   tpane);
	gtk_signal_connect(GTK_OBJECT(
	                       ((GtkCListColumn)(GTK_CLIST(tpane->ctree)->column[2]))
	                       .button), "clicked",
	                   GTK_SIGNAL_FUNC(statusColumnSortButtonPress),
	                   tpane);

#ifdef HAVE_STRPTIME
	gtk_signal_connect(GTK_OBJECT(
	                       ((GtkCListColumn)(GTK_CLIST(tpane->ctree)->column[0]))
	                       .button), "clicked",
	                   GTK_SIGNAL_FUNC(itemColumnSortButtonPress),
	                   tpane);
#endif /* HAVE_STRPTIME */

	/* For 1.1.5 */
	temp = gtk_scrolled_window_new(NULL, NULL);

	gtk_container_add(GTK_CONTAINER(tpane->listFrame), temp);
	gtk_container_add(GTK_CONTAINER(temp), tpane->ctree);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(temp),
	                               GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	/* 1.1.5 */

	gtk_clist_set_column_width(GTK_CLIST(tpane->ctree), 0, 300);
	gtk_clist_set_column_width(GTK_CLIST(tpane->ctree), 1, 60);
	gtk_clist_set_column_width(GTK_CLIST(tpane->ctree), 2, 75);
#if 0 /* doesn't work right */
	gtk_clist_set_column_auto_resize(GTK_CLIST(tpane->ctree),
	                                 0, TRUE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(tpane->ctree),
	                                 1, FALSE);
	gtk_clist_set_column_auto_resize(GTK_CLIST(tpane->ctree),
	                                 2, FALSE);
#endif
	gtk_clist_set_selection_mode(GTK_CLIST(tpane->ctree),
	                             GTK_SELECTION_EXTENDED);

	tpane->buttonHBox = gtk_hbox_new(FALSE, FALSE);
	gtk_box_pack_start(GTK_BOX(tpane->vbox), tpane->buttonHBox,
	                   FALSE, FALSE, 0);

	tpane->listEditFrame = gtk_frame_new( listEditFrameTitle );
	gtk_box_pack_start(GTK_BOX(tpane->buttonHBox), tpane->listEditFrame,
	                   FALSE, FALSE, 0);

	tpane->itemEditFrame = gtk_frame_new( itemEditFrameTitle );
	gtk_box_pack_start(GTK_BOX(tpane->buttonHBox), tpane->itemEditFrame,
	                   FALSE, FALSE, 0);

	tpane->miscFrame = gtk_frame_new( miscFrameTitle );
	gtk_box_pack_start(GTK_BOX(tpane->buttonHBox), tpane->miscFrame,
	                   FALSE, FALSE, 0);

	/* We pack the new item frame and the new comment frame into a
	   notebook with no tabs */

	tpane->editArea = gtk_notebook_new();

	/* Is this ok? */
	gtk_notebook_set_show_border(GTK_NOTEBOOK(tpane->editArea), FALSE);


	gtk_box_pack_start(GTK_BOX(tpane->buttonHBox), tpane->editArea,
	                   TRUE, TRUE, 0);

	temp = gtk_label_new("");
	tpane->newItemFrame = gtk_frame_new(newItemFrameTitle);
	gtk_notebook_append_page(GTK_NOTEBOOK(tpane->editArea),
	                         tpane->newItemFrame, temp);

	temp = gtk_label_new("");

	tpane->commentFrame = gtk_frame_new(commentFrameTitle);
	gtk_notebook_append_page(GTK_NOTEBOOK(tpane->editArea),
	                         tpane->commentFrame, temp);

	/* Blank page */
	temp = gtk_label_new("");
	temp2 = gtk_hbox_new(FALSE, FALSE);
	gtk_notebook_append_page(GTK_NOTEBOOK(tpane->editArea),
	                         temp2, temp);

	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(tpane->editArea), FALSE);

	/* Buttons in the List Edit frame */

	tpane->listEditVBox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(tpane->listEditFrame),
	                  tpane->listEditVBox);

	tpane->listEditNew = gtk_button_new_with_label(listEditNewTitle);
	tpane->listEditEdit = gtk_button_new_with_label(listEditEditTitle);
	tpane->listEditDelete = gtk_button_new_with_label(listEditDeleteTitle);

	gtk_box_pack_start(GTK_BOX(tpane->listEditVBox), tpane->listEditNew,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->listEditVBox), tpane->listEditEdit,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->listEditVBox), tpane->listEditDelete,
	                   FALSE, FALSE, 0);

	gtk_signal_connect(GTK_OBJECT(tpane->listEditNew), "clicked",
	                   GTK_SIGNAL_FUNC(listEditNewButtonPress),
	                   tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->listEditEdit), "clicked",
	                   GTK_SIGNAL_FUNC(listEditEditButtonPress),
	                   tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->listEditDelete), "clicked",
	                   GTK_SIGNAL_FUNC(listEditDeleteButtonPress),
	                   tpane);


	/* Buttons in the Item Edit frame */

	tpane->itemEditVBox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(tpane->itemEditFrame),
	                  tpane->itemEditVBox);

	tpane->itemEditIncrement = gtk_button_new_with_label(
	                               itemEditIncrementTitle);
	tpane->itemEditDecrement = gtk_button_new_with_label(
	                               itemEditDecrementTitle);
	tpane->itemEditClose = gtk_button_new_with_label(
	                           itemEditCloseTitle);

	gtk_box_pack_start(GTK_BOX(tpane->itemEditVBox), tpane->itemEditIncrement,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->itemEditVBox), tpane->itemEditDecrement,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->itemEditVBox), tpane->itemEditClose,
	                   FALSE, FALSE, 0);

	gtk_signal_connect(GTK_OBJECT(tpane->itemEditIncrement), "clicked",
	                   GTK_SIGNAL_FUNC(itemIncrementButtonPress),
	                   tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->itemEditDecrement), "clicked",
	                   GTK_SIGNAL_FUNC(itemDecrementButtonPress),
	                   tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->itemEditClose), "clicked",
	                   GTK_SIGNAL_FUNC(itemCloseButtonPress),
	                   tpane);



	/* Buttons in the Misc frame */



	tpane->miscEditVBox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(tpane->miscFrame),
	                  tpane->miscEditVBox);

	tpane->miscAddComment = gtk_button_new_with_label(miscAddCommentTitle);
	tpane->miscRemoveComment = gtk_button_new_with_label(
	                               miscRemoveCommentTitle);
	gtk_widget_set_sensitive(tpane->miscRemoveComment, FALSE);
	tpane->miscAssign = gtk_button_new_with_label(
	                        miscAssignTitle);
	gtk_widget_set_sensitive(tpane->miscAssign, FALSE);

	gtk_box_pack_start(GTK_BOX(tpane->miscEditVBox), tpane->miscAddComment,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->miscEditVBox), tpane->miscRemoveComment,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->miscEditVBox), tpane->miscAssign,
	                   FALSE, FALSE, 0);


	gtk_signal_connect(GTK_OBJECT(tpane->miscAddComment), "clicked",
	                   GTK_SIGNAL_FUNC(miscAddCommentButtonPress), tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->miscRemoveComment), "clicked",
	                   GTK_SIGNAL_FUNC(miscRemoveCommentButtonPress), tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->miscAssign), "clicked",
	                   GTK_SIGNAL_FUNC(miscAssignButtonPress), tpane);



	/* Buttons & Entries in New Item frame */



	tpane->newItemVBox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(tpane->newItemFrame), tpane->newItemVBox);

	tpane->itemNameEntry = gtk_entry_new();
	tpane->itemAssignedEntry = gtk_entry_new();

	temp = gtk_hbox_new(FALSE, FALSE);
	tpane->itemAdd = gtk_button_new_with_label(newItemAdd);
	tpane->itemCancel = gtk_button_new_with_label(newItemCancel);

	gtk_box_pack_start(GTK_BOX(temp), tpane->itemAdd, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(temp), tpane->itemCancel, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(tpane->newItemVBox), tpane->itemNameEntry,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->newItemVBox), tpane->itemAssignedEntry,
	                   FALSE, FALSE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->newItemVBox), temp,
	                   FALSE, FALSE, 0);

	gtk_signal_connect(GTK_OBJECT(tpane->itemAdd), "clicked",
	                   GTK_SIGNAL_FUNC(newItemAddButtonPress), tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->itemCancel), "clicked",
	                   GTK_SIGNAL_FUNC(newItemCancelButtonPress), tpane);



	/* Buttons and text area in the add comment frame ******/



	tpane->commentVBox = gtk_vbox_new(FALSE, FALSE);
	gtk_container_add(GTK_CONTAINER(tpane->commentFrame), tpane->commentVBox);

	tpane->commentText = gtk_text_new(NULL, NULL);
	gtk_widget_set_usize(GTK_WIDGET(tpane->commentText), -1, 10);
	gtk_text_set_editable(GTK_TEXT(tpane->commentText), TRUE);

	temp = gtk_hbox_new(FALSE, FALSE);
	tpane->commentAdd = gtk_button_new_with_label(commentAdd);
	tpane->commentCancel = gtk_button_new_with_label(newItemCancel);

	gtk_box_pack_start(GTK_BOX(temp), tpane->commentAdd, FALSE, FALSE, 0);
	gtk_box_pack_end(GTK_BOX(temp), tpane->commentCancel, FALSE, FALSE, 0);

	gtk_box_pack_start(GTK_BOX(tpane->commentVBox), tpane->commentText,
	                   TRUE, TRUE, 0);
	gtk_box_pack_start(GTK_BOX(tpane->commentVBox), temp,
	                   FALSE, FALSE, 0);

	gtk_signal_connect(GTK_OBJECT(tpane->commentAdd), "clicked",
	                   GTK_SIGNAL_FUNC(commentAddButtonPress), tpane);
	gtk_signal_connect(GTK_OBJECT(tpane->commentCancel), "clicked",
	                   GTK_SIGNAL_FUNC(newItemCancelButtonPress), tpane);




	/* End of contstructor */


	gtk_widget_show_all(tpane->vbox);

	/* New item frame is not visible by default */

	gtk_notebook_set_page(GTK_NOTEBOOK(tpane->editArea), 2);

	return tpane;
}


gI_TodoPane *
gI_todo_pane_create_from_todo(gI_TodoList *tlist)
{
	gI_TodoPane *tpane;
	GList *list;
	gI_TodoItem *todo;


	g_return_val_if_fail(tlist != NULL, NULL);

	tpane = gI_todo_pane_create();

	tpane->tlist = tlist;

	list = tlist->todoItems;
	while(list)
	{
		todo = list->data;

		todo_pane_pack_node_from_todo(tpane, NULL, todo);

		list = g_list_next(list);
	}

	gI_todo_list_attach(tlist, tpane);

	return tpane;
}

void
gI_todo_pane_destroy(gI_TodoPane *tpane)
{
	g_return_if_fail(tpane);

	gI_todo_list_detach(tpane->tlist, tpane);

	g_free(tpane);
}


/* Local helper functions *******************************************/

static void
todo_pane_set_row_text(gI_TodoPane *tpane,
                       GtkCTreeNode *node,
                       gchar *rowData[])
{
	gint a;
	for(a = 0; a < 3; a++)
		if(rowData[a])
			gtk_ctree_node_set_text(GTK_CTREE(tpane->ctree),
			                        node, a, rowData[a]);
}

static void
todo_pane_pack_node_from_todo(gI_TodoPane *tpane,
                              TodoPaneData *tdata,
                              gI_TodoItem *todo)
{
	GString *string;
	gchar *rowData[3];
	GtkCTreeNode *last_node = NULL;
	GtkCTreeNode *node;
	GList *comments, *commentsNodes;

	string = g_string_new("");

	if(!tdata)
	{
		tdata = g_new0(TodoPaneData, 1);
		last_node = NULL;
	}
	else
	{
		last_node = tdata->top;
	}

	tdata->todo = todo;

	/* Initial row */

	rowData[0] = todo->item;
	rowData[1] = todo->assignedTo;
	rowData[2] = (gchar *)todo->status;

	if(!tdata->top)
		tdata->top = gtk_ctree_insert_node(GTK_CTREE(tpane->ctree),
		                                   NULL, last_node,
		                                   NULL, 0,
		                                   NULL, NULL, NULL, NULL,
		                                   FALSE, FALSE);

	todo_pane_set_row_text(tpane, tdata->top, rowData);

	gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree), tdata->top, tdata);

	/* Create date - always available */

	rowData[0] = rowData[1] = rowData[2] = NULL;

	string = g_string_assign(string, todoItemOpenedDate);
	string = g_string_append(string, todo->openedTime);
	rowData[0] = string->str;

	if(!tdata->createDate)
		tdata->createDate = gtk_ctree_insert_node(GTK_CTREE(tpane->ctree),
		                    tdata->top, NULL,
		                    NULL, 0,
		                    NULL, NULL, NULL, NULL,
		                    TRUE, FALSE);
	todo_pane_set_row_text(tpane, tdata->createDate, rowData);

	gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree),
	                            tdata->createDate, tdata);

	if(todo->modifiedTime)
	{
		string = g_string_assign(string, todoItemModifiedDate);
		string = g_string_append(string, todo->modifiedTime);

		rowData[0] = string->str;

		if(!tdata->modifiedDate)
			tdata->modifiedDate = gtk_ctree_insert_node(
			                          GTK_CTREE(tpane->ctree),
			                          tdata->top, tdata->closedDate ? tdata->closedDate :
			                          tdata->dueDate ? tdata->dueDate : tdata->commentsRoot,
			                          NULL, 0,
			                          NULL, NULL, NULL, NULL,
			                          TRUE, FALSE);
		todo_pane_set_row_text(tpane, tdata->modifiedDate, rowData);

		gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree),
		                            tdata->modifiedDate, tdata);

	}

	if(todo->closedTime)
	{
		string = g_string_assign(string, todoItemClosedDate);
		string = g_string_append(string, todo->closedTime);

		rowData[0] = string->str;

		if(!tdata->closedDate)
			tdata->closedDate = gtk_ctree_insert_node(
			                        GTK_CTREE(tpane->ctree),
			                        tdata->top,
			                        tdata->dueDate ? tdata->dueDate : tdata->commentsRoot,
			                        rowData, 0,
			                        NULL, NULL, NULL, NULL,
			                        TRUE, FALSE);
		todo_pane_set_row_text(tpane, tdata->closedDate, rowData);

		gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree),
		                            tdata->closedDate, tdata);

	}

	if(todo->dueDate)
	{
		string = g_string_assign(string, todoItemDueDate);
		string = g_string_append(string, todo->dueDate);

		rowData[0] = string->str;

		if(!tdata->dueDate)
			tdata->dueDate = gtk_ctree_insert_node(
			                     GTK_CTREE(tpane->ctree),
			                     tdata->top, tdata->commentsRoot,
			                     rowData, 0,
			                     NULL, NULL, NULL, NULL,
			                     TRUE, FALSE);
		todo_pane_set_row_text(tpane, tdata->dueDate, rowData);

		gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree),
		                            tdata->dueDate, tdata);

	}

	if(todo->comments)
	{
		comments = todo->comments;
		commentsNodes = tdata->commentsNodes;

		if(!tdata->commentsRoot)
		{
			rowData[0] = (gchar *)todoItemComments;

			tdata->commentsRoot = gtk_ctree_insert_node(
			                          GTK_CTREE(tpane->ctree),
			                          tdata->top, NULL,
			                          rowData, 0,
			                          NULL, NULL, NULL, NULL,
			                          FALSE, TRUE);
		}
		else
		{
			while(comments && commentsNodes)
			{
				comments = g_list_next(comments);
				commentsNodes = g_list_next(commentsNodes);
			}
		}

		gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree),
		                            tdata->commentsRoot, tdata);

		while(comments)
		{
			rowData[0] = comments->data;

			node = gtk_ctree_insert_node(
			           GTK_CTREE(tpane->ctree),
			           tdata->commentsRoot, NULL,
			           rowData, 0,
			           NULL, NULL, NULL, NULL,
			           TRUE, TRUE);
			gtk_ctree_node_set_row_data(GTK_CTREE(tpane->ctree),
			                            node, tdata);

			tdata->commentsNodes = g_list_append(tdata->commentsNodes, node);

			comments = g_list_next(comments);
		}
	}

	g_string_free(string, FALSE);
}

/* Grovels through the Ctree and spews out a unique list of items which
 * were selected.  100% probability that this is the wrong way to do it.
 * If course, CTree is inherently sick, so I don't care. */

static GList *
todo_pane_ctree_get_todo_selected(gI_TodoPane *tpane)
{
	GtkCTreeNode *select;
	GList *selected_data = NULL;
	TodoPaneData *tdata;

	g_return_val_if_fail(GTK_IS_CTREE(tpane->ctree), 0);

	select = GTK_CTREE_NODE(GTK_CLIST(tpane->ctree)->selection);

	if(!select)
	{
		return NULL;
	}

	while(select)
	{
		tdata = gtk_ctree_node_get_row_data(GTK_CTREE(tpane->ctree),
		                                    GTK_CTREE_NODE(
		                                        GTK_CTREE_ROW(select)));

		if(!tdata)
			g_assert_not_reached();

		if(!g_list_find(selected_data, tdata))
			selected_data = g_list_append(selected_data, tdata);

		select = GTK_CTREE_NODE_NEXT(select);
	}

	return selected_data;
}

/* Local callback functions ******************************************/


static gint
listEditNewButtonPress(GtkWidget *widget,
                       gpointer data)
{
	gI_TodoPane *tpane;

	tpane = data;

	if(tpane->editingData)
		newItemCancelButtonPress(widget, data);

	gtk_notebook_set_page(GTK_NOTEBOOK(tpane->editArea), 0);

	return 0;
}

static gint
listEditEditButtonPress(GtkWidget *widget,
                        gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	gI_TodoItem *todo;
	GList *selected;

	tpane = data;

	selected = todo_pane_ctree_get_todo_selected(tpane);

	if(!selected)
		return 0;

	gtk_notebook_set_page(GTK_NOTEBOOK(tpane->editArea), 0);

	tdata = selected->data;
	todo = tdata->todo;

	gtk_entry_set_text(GTK_ENTRY(tpane->itemNameEntry), todo->item);
	if(todo->assignedTo)
		gtk_entry_set_text(GTK_ENTRY(tpane->itemAssignedEntry),
		                   todo->assignedTo);

	tpane->editingData = tdata;

	g_list_free(selected);

	return 0;
}

static gint
listEditDeleteButtonPress(GtkWidget *widget,
                          gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	GList *selected;

	tpane = data;

	selected = todo_pane_ctree_get_todo_selected(tpane);

	if(!selected)
		return 0;

	newItemCancelButtonPress(widget, data);

	gtk_clist_freeze(GTK_CLIST(tpane->ctree));

	while(selected)
	{
		tdata = selected->data;

		gI_todo_list_remove(tpane->tlist, tdata->todo);

		gtk_ctree_remove_node(GTK_CTREE(tpane->ctree), tdata->top);

		g_free(tdata);

		selected = g_list_next(selected);
	}

	gtk_clist_thaw(GTK_CLIST(tpane->ctree));

	return 0;
}

static gint
itemIncrementButtonPress(GtkWidget *widget,
                         gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	GList *selected;

	tpane = data;

	selected = todo_pane_ctree_get_todo_selected(tpane);

	if(!selected)
		return 0;

	gtk_clist_freeze(GTK_CLIST(tpane->ctree));

	while(selected)
	{
		tdata = selected->data;

		gI_todo_item_increment(tdata->todo);

		todo_pane_pack_node_from_todo(tpane, tdata, tdata->todo);

		selected = g_list_next(selected);
	}

	gtk_clist_thaw(GTK_CLIST(tpane->ctree));

	g_list_free(selected);
	return 0;
}

static gint
itemDecrementButtonPress(GtkWidget *widget,
                         gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	GList *selected;

	tpane = data;

	selected = todo_pane_ctree_get_todo_selected(tpane);

	if(!selected)
		return 0;

	gtk_clist_freeze(GTK_CLIST(tpane->ctree));

	while(selected)
	{
		tdata = selected->data;

		gI_todo_item_decrement(tdata->todo);

		todo_pane_pack_node_from_todo(tpane, tdata, tdata->todo);

		selected = g_list_next(selected);
	}

	gtk_clist_thaw(GTK_CLIST(tpane->ctree));

	g_list_free(selected);
	return 0;
}

static gint
itemCloseButtonPress(GtkWidget *widget,
                     gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	GList *selected,
	*comments;
	GString *buf;
	gchar *user;
	gI_project *project;
	extern gI_config *cfg;

	gchar *changelogFile;

	tpane = data;

	selected = todo_pane_ctree_get_todo_selected(tpane);

	if(!selected)
		return 0;

	gtk_clist_freeze(GTK_CLIST(tpane->ctree));

	while(selected)
	{
		tdata = selected->data;

		gI_todo_item_close(tdata->todo);

		todo_pane_pack_node_from_todo(tpane, tdata, tdata->todo);

		/* Add changelog entry if necessary */

		if(cfg->changelog)
		{
			project = gI_project_get_current();

			if( isempty( project->changelog ) )
			{
				/* error handling */
				gI_error_dialog( _("No ChangeLog-File specified.") );
				return 0;
			}

			changelogFile = project->changelog;

			user = getenv( "USER" );

			buf = g_string_new("");

			if( user )
				g_string_sprintf(buf, "%s <%s>\n",
				                 tdata->todo->closedTime, user );
			else
				g_string_sprintf(buf, "%s\n", tdata->todo->closedTime );
			g_string_sprintfa( buf, "\t* %s\n", tdata->todo->item );

			comments = tdata->todo->comments;
			while(comments)
			{
				g_string_sprintfa( buf, "\t* %s\n", (gchar *)(comments->data));
				comments = g_list_next(comments);
			}

			g_string_sprintfa( buf, "\n" );

			gI_changelog_popup(buf->str, changelogFile);

			g_string_free(buf, FALSE);


		}
		selected = g_list_next(selected);
	}

	gtk_clist_thaw(GTK_CLIST(tpane->ctree));

	g_list_free(selected);

	return 0;
}

static gint
miscAddCommentButtonPress(GtkWidget *widget,
                          gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	gI_TodoItem *todo;
	GList *selected;

	tpane = data;

	newItemCancelButtonPress(widget, data);

	selected = todo_pane_ctree_get_todo_selected(tpane);

	if(!selected)
		return 0;

	tdata = selected->data;
	todo = tdata->todo;

	tpane->editingData = tdata;

	gtk_notebook_set_page(GTK_NOTEBOOK(tpane->editArea), 1);

	return 0;
}

static gint
miscRemoveCommentButtonPress(GtkWidget *widget,
                             gpointer data)
{
	/* FIXME STUB */

	return 0;
}

static gint
miscAssignButtonPress(GtkWidget *widget,
                      gpointer data)
{
#if 0 /* JUST FOR TESTING */
	gI_TodoPane *tpane;
	gI_TodoList *tlist;
	FILE *file;
	/* FIXME Just a stub */

	tpane = data;
	tlist = tpane->tlist;

	file = fopen("TestTodoFile.txt", "w");

	gI_todo_list_write_to_file(tlist, file, "$Begin Todo List",
	                           "$End Todo List");

	fclose(file);
#endif
	return 0;
}

static gint
newItemAddButtonPress(GtkWidget *widget,
                      gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata = NULL;
	gI_TodoItem *todo;
	gchar *name;
	gchar *assign;

	tpane = data;

	name = gtk_entry_get_text(GTK_ENTRY(tpane->itemNameEntry));
	assign = gtk_entry_get_text(GTK_ENTRY(tpane->itemAssignedEntry));

	if(isempty(name))
	{
		gI_ok_dialog((gchar *)errorNeedName);
		return 0;
	}


	if(tpane->editingData)
	{
		tdata = tpane->editingData;
		todo = gI_todo_item_create(name, tdata->todo->openedTime);

		gI_todo_list_remove(tpane->tlist, tdata->todo);
		tdata->todo = todo;
	}
	else
		todo = gI_todo_item_create(name, NULL);

	if(!isempty(assign))
		gI_todo_item_assign(todo, assign);

	gI_todo_list_add(tpane->tlist, todo);

	gtk_clist_freeze(GTK_CLIST(tpane->ctree));

	todo_pane_pack_node_from_todo(tpane, tdata, todo);

	gtk_clist_thaw(GTK_CLIST(tpane->ctree));

	newItemCancelButtonPress(widget, data);

	return 0;
}

static gint
newItemCancelButtonPress(GtkWidget *widget,
                         gpointer data)
{
	gI_TodoPane *tpane;

	tpane = data;

	gtk_entry_set_text(GTK_ENTRY(tpane->itemNameEntry), "");
	gtk_entry_set_text(GTK_ENTRY(tpane->itemAssignedEntry), "");

	tpane->editingData = NULL;

	gtk_notebook_set_page(GTK_NOTEBOOK(tpane->editArea), 2);

	return 0;
}

static gint
commentAddButtonPress(GtkWidget *widget,
                      gpointer data)
{
	gI_TodoPane *tpane;
	TodoPaneData *tdata;
	gI_TodoItem *todo;
	gint len;
	gchar *comment;

	tpane = data;

	tdata = tpane->editingData;

	g_return_val_if_fail(tdata, 0);

	todo = tdata->todo;

	len = gtk_text_get_length(GTK_TEXT(tpane->commentText));
	comment = gtk_editable_get_chars(GTK_EDITABLE(tpane->commentText),
	                                 0, len);
	if(isempty(comment))
	{
		gI_ok_dialog((gchar *)errorNeedComment);
		return 0;
	}

	gI_todo_item_add_comment(todo, comment);

	gtk_clist_freeze(GTK_CLIST(tpane->ctree));
	todo_pane_pack_node_from_todo(tpane, tdata, todo);
	gtk_clist_thaw(GTK_CLIST(tpane->ctree));

	newItemCancelButtonPress(widget, data);

	gtk_editable_select_region(GTK_EDITABLE(tpane->commentText), 0, len);
	gtk_editable_delete_selection(GTK_EDITABLE(tpane->commentText));

	return 0;
}

static gint
assignedColumnSortButtonPress(GtkWidget *widget,
                              gpointer data)
{
	GtkWidget *ctree;

	ctree = ((gI_TodoPane *)data)->ctree;

	if(GTK_CLIST(ctree)->sort_column == 1)
		gtk_clist_set_sort_type(GTK_CLIST(ctree),
		                        !(GTK_CLIST(ctree)->sort_type));

	gtk_clist_set_sort_column(GTK_CLIST(ctree), 1);

	gtk_clist_freeze(GTK_CLIST(ctree));
	gtk_clist_sort(GTK_CLIST(ctree));
	gtk_clist_thaw(GTK_CLIST(ctree));

	return 0;
}

static gint
statusColumnSortButtonPress(GtkWidget *widget,
                            gpointer data)
{
	GtkWidget *ctree;

	ctree = ((gI_TodoPane *)data)->ctree;

	if(GTK_CLIST(ctree)->sort_column == 2)
		gtk_clist_set_sort_type(GTK_CLIST(ctree),
		                        !(GTK_CLIST(ctree)->sort_type));

	gtk_clist_set_sort_column(GTK_CLIST(ctree), 2);

	gtk_clist_set_compare_func(GTK_CLIST(ctree),
	                           (GtkCListCompareFunc)statusColumnSortFunc);
	gtk_clist_freeze(GTK_CLIST(ctree));
	gtk_clist_sort(GTK_CLIST(ctree));
	gtk_clist_thaw(GTK_CLIST(ctree));
	gtk_clist_set_compare_func(GTK_CLIST(ctree), NULL);

	return 0;
}

static gint
statusColumnSortFunc(GtkCList *clist,
                     GtkCListRow *ptr1,
                     GtkCListRow *ptr2)
{
	TodoPaneData *tdata1,
	*tdata2;

	tdata1 = ptr1->data;
	tdata2 = ptr2->data;

	if(tdata1 == tdata2)
		return 0;

	if(tdata1->todo->todo_status == tdata2->todo->todo_status)
		return 0;

	if(todo_sort_order[tdata1->todo->todo_status] >
	        todo_sort_order[tdata2->todo->todo_status])
		return 1;

	return -1;
}

#ifdef HAVE_STRPTIME
static gint
itemColumnSortButtonPress(GtkWidget *widget,
                          gpointer data)
{
	GtkWidget *ctree;

	ctree = ((gI_TodoPane *)data)->ctree;

	if(GTK_CLIST(ctree)->sort_column == 0)
		gtk_clist_set_sort_type(GTK_CLIST(ctree),
		                        !(GTK_CLIST(ctree)->sort_type));

	gtk_clist_set_sort_column(GTK_CLIST(ctree), 0);

	gtk_clist_set_compare_func(GTK_CLIST(ctree),
	                           (GtkCListCompareFunc)itemColumnSortFunc);
	gtk_clist_freeze(GTK_CLIST(ctree));
	gtk_ctree_sort_node(GTK_CTREE(ctree), NULL);
	gtk_clist_thaw(GTK_CLIST(ctree));
	gtk_clist_set_compare_func(GTK_CLIST(ctree), NULL);

	return 0;
}

static gint
itemColumnSortFunc(GtkCList *clist,
                   GtkCListRow *ptr1,
                   GtkCListRow *ptr2)
{
	TodoPaneData *tdata1,
	*tdata2;
	struct tm tm1, tm2;

	tdata1 = ptr1->data;
	tdata2 = ptr2->data;

	if(tdata1 == tdata2)
		return 0;

	strptime(tdata1->todo->openedTime, "%c", &tm1);
	strptime(tdata2->todo->openedTime, "%c", &tm2);

	return(difftime( mktime(&tm1), mktime(&tm2) ) );
}
#endif /* HAVE_STRPTIME */

