/* $Header: /cvs/gnome/gIDE/src/gI_debug.c,v 1.15 2000/05/02 16:28:03 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "gide.h"
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <sys/un.h>
#include <time.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include "gI_file.h"
#include "gI_common.h"
#include "gI_project.h"
#include "gI_compile.h"
#include "gI_debug.h"
#include "gI_menus.h"
#include "gI_debug_watch.h"
#include "gI_debug_breakpoint.h"
#include "gI_debug_backtrace.h"
#include "gI_debug_p.h"

/* status of a command in the gdb command queue */
/* not sent yet */
#define N_GDBCMD_TODO					1
/* sent */
#define N_GDBCMD_DONE					2

/* state of gdb */
/* undefined, set on initialization */
#define N_GDB_UNDEF						0
/* idle, prompt is there */
#define N_GDB_IDLE						1
/* running, that does not have to mean that the program is running,
   it just means that gdb is busy */
#define N_GDB_RUNNING					2
/* break */
#define N_GDB_BREAK						3

/* set this to 0 and no gdb output */
static glong							show_output = 1;

/* local prototypes */
static void debug_start_gdb(gI_debug_window* dw);
static void debug_stop_gdb(gI_debug_window* dw);
static gint debug_get_child_pid(gI_debug_window* dw);
static void debug_run(GtkWidget* widget, gI_debug_window* dw);
static gint debug_check_for_data(void);
static void install_debug_popup(gI_debug_window* dw);
static gint popup_debug_menu(GtkWidget* widget, GdkEvent* ev);
static void debug_kill(GtkWidget* widget, gI_debug_window* dw);
static void debug_step_out(GtkWidget* widget, gI_debug_window* dw);
static void debug_run_to_cursor(GtkWidget* widget, gI_debug_window* dw);
static void debug_backtrace(GtkWidget* widget, gI_debug_window* dw);
static void debug_expressions(GtkWidget* widget, gI_debug_window* dw);
/*
static void debug_exceptions(GtkWidget* widget, gI_debug_window* dw);
*/
static void debug_file(GtkWidget* widget, gpointer data);
static gI_debug_window* debug_window_new(GideDocument* document);
static void debug_window_destroy(GtkWidget* widget, gI_debug_window* dw);
static void debug_window_parse_state(gI_debug_window* dw, gchar* buf);
static void debug_process_whatis(gI_debug_window* dw, gchar* msg);
static void debug_process_expressions(gI_debug_window* dw, gchar* msg);
static void debug_update_view(gI_debug_window* dw, gchar* ptr);
static void debug_process_exceptions(gI_debug_window* dw, gchar* msg);
static gchar* debug_get_project_exe(gI_project* project);
static gchar* debug_get_exe(GideDocument* doc);
static gI_gdbcmd* gI_gdbcmd_new(void);
static void debug_window_parse_state(gI_debug_window* dw, gchar* buf);


/* private functions */
/*
 * Add a new expression to the watch window
 */
static void
add_expression(
	gI_debug_window*					dw,
	gchar*								locale
)
{
	gchar*								row[3];
	gchar*								ptr;
	gchar*								dup;
	gchar*								dup2;
	gchar*								dup3 = NULL;
	gchar*								sptr;

	dup = g_strdup(locale);
	ptr = strchr(dup, '=');
	--ptr;
	*ptr = '\0';
	row[0] = dup;

	dup2 = g_strdup(locale);
	ptr = strchr(dup2, '=');
	ptr += 2;

	sptr = ptr;
	if(*sptr == '(')
	{
		dup3 = g_strdup(++sptr);

		sptr = dup3;
		while(*sptr != ')')
		{
			sptr++;
		}

		*sptr = '\0';
		ptr = sptr + 2;

		row[1] = dup3;
	}
	else
	{
		row[1] = NULL;
	}
	row[2] = ptr;

	gtk_clist_append(GTK_CLIST(dw->watch_window->list), row);

	g_free(dup);
	g_free(dup2);
	if(dup3 != NULL)
	{
		g_free(dup3);
	}
}


/*
 * Process the whatis debugger output
 */
static void
debug_process_whatis(
	gI_debug_window*					dw,
	gchar*								msg
)
{
	glong								rowcnt;
	gchar*								type;
	glong								found_empty = -1;
	gchar*								ptr;
	gchar*								retptr;
	gchar*								expr;
	gchar								cmdstr[100];

	/* get first empty types fields of the list */
	for(rowcnt = 0; rowcnt < GTK_CLIST(dw->watch_window->list)->rows;
		rowcnt++)
	{
		type = NULL;
		gtk_clist_get_text(GTK_CLIST(dw->watch_window->list), rowcnt, 1,
			&type);
		if(isempty(type))
		{
			found_empty = rowcnt;
			break;
		}
	}

	if(found_empty == -1)
	{
		gtk_clist_thaw(GTK_CLIST(dw->watch_window->list));
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	if(!strncmp(msg, "No symbol \"", 10))
	{
		gtk_clist_set_text(GTK_CLIST(dw->watch_window->list), found_empty,
			1, "???");
	}
	else
	{
		ptr = strchr(msg, '=');
		if(ptr)
		{
			retptr = strchr(ptr, '\n');
			*retptr = '\0';
			ptr += 2;

			/* strip '(' & ')' */
			if(*ptr == '(')
			{
				ptr++;
			}
			if(*--retptr == ')')
			{
				*retptr = '\0';
			}

			gtk_clist_set_text(GTK_CLIST(dw->watch_window->list),
				found_empty, 1, ptr);
		}
	}

	found_empty = -1;

	/* get first empty types fields of the list */
	for(rowcnt = 0; rowcnt < GTK_CLIST(dw->watch_window->list)->rows;
		rowcnt++)
	{
		type = NULL;
		gtk_clist_get_text(GTK_CLIST(dw->watch_window->list), rowcnt, 1,
			&type);
		if(isempty(type))
		{
			found_empty = rowcnt;
			break;
		}
	}

	if(found_empty == -1)
	{
		gtk_clist_thaw(GTK_CLIST(dw->watch_window->list));
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	gtk_clist_get_text(GTK_CLIST(dw->watch_window->list), found_empty, 0,
		&expr);

	g_snprintf(cmdstr, sizeof(cmdstr), "whatis %s", expr);

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
	dw->wait_for = N_WAIT_WHATIS;
}


/*
 * Process the expressions debugger output
 */
static void
debug_process_expressions(
	gI_debug_window*					dw,
	gchar*								msg
)
{
	gchar*								ptr;
	gchar*								retptr;
	gchar								locale[MAXLEN];
	gchar*								sptr;
	glong								found_empty = -1;
	glong								rowcnt;
	gchar*								type;
	gchar*								expr;
	gchar								cmdstr[100];

	if(dw->watch_window)
	{
		gtk_clist_clear(GTK_CLIST(dw->watch_window->list));
	}

	ptr = &msg[0];
	if(!strncmp(ptr, "No locals", 9) ||
		!strncmp(ptr, "No symbol table", 15) ||
		!strncmp(ptr, "No frame selected", 17) ||
		!strncmp(ptr, "The program is not being run.", 29))
	{
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	/* new watch window */
	if(dw->watch_window == NULL)
	{
		dw->watch_window = gI_watch_window_new();
	}

	if(!strstr(ptr, GDB_PROMPT))
	{
		g_warning("Parsing error: endless loop?!");
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	gtk_clist_freeze(GTK_CLIST(dw->watch_window->list));

	while(strncmp(ptr, GDB_PROMPT, strlen(GDB_PROMPT)))
	{
		/* this may get an endless loop */
		retptr = strchr(ptr, '\n');
		--retptr;
		strncpy(locale, ptr, retptr - (ptr - 1));

		locale[retptr - (ptr - 1)] = '\0';

		if(*ptr == ' ')
		{
			ptr = retptr + 2;
			continue;
		}

		sptr = strchr(locale, '=');
		if(!strncmp(sptr, "= {", 3))
		{
			sptr = strstr(locale, "= {");
			sptr += 2;
			*sptr = '\0';
			strcat(locale, "???");
		}
		else
		{
			if(strchr(locale, '{') || strchr(locale, '}'))
			{
				ptr = retptr + 2;
				continue;
			}
		}

		add_expression(dw, locale);

		ptr = retptr + 2;
	}
	dw->wait_for = N_WAIT_NOTHING;

	/* get first empty types field of the list */
	found_empty = -1;

	for(rowcnt = 0; rowcnt < GTK_CLIST(dw->watch_window->list)->rows;
		rowcnt++)
	{
		type = NULL;
		gtk_clist_get_text(GTK_CLIST(dw->watch_window->list), rowcnt, 1,
			&type);
		if(isempty(type))
		{
			found_empty = rowcnt;
			break;
		}
	}

	if(found_empty == -1)
	{
		gtk_clist_thaw(GTK_CLIST(dw->watch_window->list));
		dw->wait_for = N_WAIT_NOTHING;
		return;
	}

	gtk_clist_get_text(GTK_CLIST(dw->watch_window->list), found_empty, 0,
		&expr);

	g_snprintf(cmdstr, sizeof(cmdstr), "whatis %s", expr);

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
	dw->wait_for = N_WAIT_WHATIS;
}


/*
 * Process the exceptions debugger output
 */
static void
debug_process_exceptions(
	gI_debug_window*					dw,
	gchar*								msg
)
{
	/* TODO: implement parsing for exceptions stuff */
	dw->wait_for = N_WAIT_NOTHING;
}


/*
 * Update the debugger view
 */
static void
debug_update_view(
	gI_debug_window*					dw,
	gchar*								ptr
)
{
	gchar*								retptr;
	gchar*								data[6];
	gchar*								file;
	glong								line;

	ptr += 2;

	retptr = strchr(ptr, '\n');
	if(retptr)
	{
		*retptr = '\0';
	}

	SK_GetFields(ptr, data, ':');

	if(atoi(data[0]) != 0)
	{
		file = data[1];
		line = atoi(data[2]);

		if(check_file_open(file))
		{
			/* if file is already opened, just switch the notebook page */
			goto_file(file);
		}
		else
		{
			/* ...open it */
			file_open_by_name(main_window, file);
		}

		/* scroll to the line */
		goto_line(line);
	}

	/* can we be sure that there are not more than 2 fields
	 * to free????
	 */
	g_free(data[0]);
	g_free(data[1]);
	g_free(data[2]);

	/* break'd, now we request information about locals
	 * and maybe the break- and watchpoints!?
	 */
	debug_send_dbg_cmd(dw, "info locals", TRUE);
	dw->wait_for = N_WAIT_EXPRESSIONS;

	/* does this actually work, sending 2 commands in a row? */
	if(dw->brkpnt_window && GTK_WIDGET_VISIBLE(dw->brkpnt_window))
	{
		/* update breakpoints */
		debug_send_dbg_cmd(dw, "info breakpoints", TRUE);
		dw->wait_for = N_WAIT_BREAKPOINTS;
	}

	/* should we update the backtrace window here, too? */
}


/*
 * Process debugger output
 */
static void
process_debug_output(
	gI_debug_window*					dw,
	gchar*								outmsg
)
{
	gchar*								ptr;
	gchar*								msg;

	msg = g_strdup(outmsg);

	/* set state */
	debug_window_parse_state(dw, msg);

	gtk_widget_set_sensitive(dw->entry, dw->state == N_GDB_RUNNING);

	if((ptr = strstr(msg, "\032\032")))
	{
		debug_update_view(dw, ptr);
	}
	else if(dw->wait_for == N_WAIT_EXPRESSIONS)
	{
		debug_process_expressions(dw, msg);
	}
	else if(dw->wait_for == N_WAIT_BREAKPOINTS)
	{
		gI_breakpoint_process_output(dw, msg);
	}
	else if(dw->wait_for == N_WAIT_BACKTRACE)
	{
		gI_backtrace_process_output(dw, msg);
	}
	else if(dw->wait_for == N_WAIT_WHATIS)
	{
		debug_process_whatis(dw, msg);
	}
	else if(dw->wait_for == N_WAIT_EVALEXPR)
	{
		gI_watch_process_output(dw, msg);
	}
	else if(dw->wait_for == N_WAIT_EXCEPTIONS)
	{
		debug_process_exceptions(dw, msg);
	}
	else if(dw->wait_for > 0)
	{
		g_warning("Debugger in unknown state\n");
	}

	g_free(msg);
}


/*
 * Input-handler function for the gdb output pipe
 */
static void
handle_debug_output(
	gI_debug_window*					dw,
	gint								fh,
	GdkInputCondition					condition
)
{
	/*
	 * we need very very big buffer or gdk_events_wait()
	 * segsegv's on large gdb/program output
	 * another reason is that (at the moment), we need all
	 * the expressions (or breakpoints) in one buffer
	 */
	gchar								buf[MAXLEN * 9];
	gint								count;

	count = read(dw->fh_out, buf, sizeof(buf));

	if(count <= 0)
	{
		/* stop debugger */
		debug_stop_gdb(dw);

		/* set statusbar */
		gI_window_set_statusbar(main_window);

		return;
	}

	buf[count] = '\0';

	if(count >= (MAXLEN * 9))
	{
		/* buffer reached maximum size, can this happen?? */
		g_warning("count reached maximum!");
	}

	process_debug_output(dw, buf);

	if(!show_output)
	{
		return;
	}

	gtk_text_insert(GTK_TEXT(dw->text), NULL, NULL, NULL, buf, count);
}


/*
 * Set state of items in Debug-Menu (sensitive true/false)
 */
static void
debug_set_menu(
	gI_debug_window*					dw,
	glong								debug
)
{
	if(debug)
	{
		gI_menus_set_sensitive_debugon(main_window);
	}
	else
	{
		gI_menus_set_sensitive_debugoff(main_window);
	}
}


/*
 * Start GDB (create socket, system gdbio, etc.)
 */
static void
debug_start_gdb(
	gI_debug_window*					dw
)
{
	gchar*								sockname;
	gchar*								execstr;
	gchar								buf[10];
	struct sockaddr_un					sock;
	gint								fh;
	gint								addrlen;
	gint								ret;

	if(!dw)
	{
		return;
	}

	/* generate socket name */
	sockname = g_strdup_printf("%s/gide_deb.%ld.%ld", cfg->tmpdir, time(NULL),
		(glong)getpid());

	/* unlink existing files */
	remove(sockname);

	fh = socket(AF_UNIX, SOCK_STREAM, 0);
	if(fh < 0)
	{
		g_free(sockname);
		perror("socket");
		return;
	}

	/* someone reported problems with the setsockopt stuff, so we disable
	 * it for now
	 */
#if 0
	/* set socket options: recv buffer size */
	if(setsockopt(fh, SOL_SOCKET, SO_RCVBUF, &bufsize, 0) != 0)
	{
		g_free(sockname);
		perror("setsockopt");
		return;
	}

	/* set socket options: send buffer size */
	if(setsockopt(fh, SOL_SOCKET, SO_SNDBUF, &bufsize, 0) != 0)
	{
		g_free(sockname);
		perror("setsockopt");
		return;
	}
#endif

	sock.sun_family = AF_UNIX;
	strcpy(sock.sun_path, sockname);

	/* bind name to socket */
	if(bind(fh, (struct sockaddr*)&sock, sizeof(struct sockaddr_un)) < 0)
	{
		g_free(sockname);
		perror("bind");
		return;
	}

	/* set mode of socket */
	chmod(sockname, 00600);

	/* queue for incoming connections */
	if(listen(fh, 1) < 0)
	{
		g_free(sockname);
		perror("listen");
		return;
	}

	/* run gdbio */
	execstr = g_strconcat("gdbio", " ", cfg->db, " ", sockname, " &", NULL);
	ret = system(execstr);
	g_free(execstr);

	if(ret == 127 || ret == -1)
	{
		gI_error_dialog(_("Unable to start 'gdbio' !"));
		remove(sockname);
		g_free(sockname);
		return;
	}

	/* accept queued connection */
	addrlen = sizeof(struct sockaddr_un);
	fh = accept(fh, (struct sockaddr*)&sock, &addrlen);
	if(fh < 0)
	{
		perror("accept");
		remove(sockname);
		g_free(sockname);
		return;
	}

	/* socket is full-duplex, same handle for input and output */
	dw->fh_in = fh;
	dw->fh_out = fh;

	/* read pid of gdb */
	read(dw->fh_out, buf, sizeof(buf));
	dw->pid = atoi(buf);

	/* save name of socket */
	dw->sockname = sockname;

	/* install input handler */
	dw->input_id = gdk_input_add(dw->fh_out, GDK_INPUT_READ,
		(GdkInputFunction)handle_debug_output, (gpointer)dw);

	/* set debug menu */
	debug_set_menu(dw, TRUE);

	/* set statusbar */
	gI_window_set_statusbar(main_window);
}


/*
 * Stop GDB (close and unlink socket, etc.)
 */
static void
debug_stop_gdb(
	gI_debug_window*					dw
)
{
	GList*								tmp;

	if(dw->pid == -1)
	{
		return;
	}

	/* remove input handler */
	gdk_input_remove(dw->input_id);

	/* send 'quit' command */
/*	debug_send_dbg_cmd(dw, "quit");*/

	/* close socket */
	close(dw->fh_in);

	/* process gtk stuff */
	while(gtk_events_pending())
	{
		gtk_main_iteration();
	}

	/* kill debugger, if it's still there... */
	kill(dw->pid, SIGKILL);

	/* display mesg */
	gI_ok_dialog(_("GDB process terminated!"));

	/* set pid in dw struct */
	dw->pid = -1;

	/* unlink fifos */
	remove(dw->sockname);
	g_free(dw->sockname);
	dw->sockname = NULL;

	/* set debug menu */
	debug_set_menu(dw, FALSE);

	/* we should free the command queue list here */
	tmp = g_list_first(dw->cmd_queue);
	while(tmp)
	{
		gI_gdbcmd* gdbcmd = (gI_gdbcmd*)tmp->data;
		if(gdbcmd)
		{
			g_free(gdbcmd->cmd);
		}
		tmp = g_list_next(tmp);
	}

	g_list_free(dw->cmd_queue);
	dw->cmd_queue = NULL;
}


/*
 * Callback for the entry widget in the debugger window
 */
static void
send_command_cb(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	gchar*								p;

	p = gtk_entry_get_text(GTK_ENTRY(widget));
	if(!p)
	{
		return;
	}

	/* if the state is running, we send the stuff directly to the debugger */
	if(dw->state == N_GDB_RUNNING)
	{
		write(dw->fh_in, p, strlen(p));
		write(dw->fh_in, "\n", 1);

		gtk_text_insert(GTK_TEXT(dw->text), NULL, NULL, NULL, p, strlen(p));
		gtk_text_insert(GTK_TEXT(dw->text), NULL, NULL, NULL, "\n", 1);

		gtk_entry_set_text(GTK_ENTRY(widget), "");

		return;
	}

	debug_send_dbg_cmd(dw, p, TRUE);
	gtk_entry_set_text(GTK_ENTRY(widget), "");
}


/*
 * Callback used when starting a new program
 */
static void
debug_run_ok(
	GtkWidget*							widget,
	GtkWidget*							entry
)
{
	gchar*								cmdstr;
	gI_debug_window*					dw;
	gchar*								run_par;

	dw = main_window->debug_window;
	if(!dw)
	{
		return;
	}

	run_par = gtk_entry_get_text(GTK_ENTRY(entry));	

	if(!run_par || isempty(run_par))
	{
		cmdstr = g_strdup("run");
	}
	else
	{
		cmdstr = g_strdup_printf("run %s", run_par);
	}

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
	g_free(cmdstr);

	gtk_widget_set_sensitive(dw->entry, TRUE);
}


/*
 * Run a program from within the debugger
 */
static void
debug_run(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	entry_dialog(_("Enter Parameters: "), _("Debug - Run program"),
		debug_run_ok);
}


/*
 * Continue a stopped program
 */
static void
debug_cont(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	debug_send_dbg_cmd(dw, "cont", TRUE);
}


/*
 * Callback used when setting a new breakpoint
 */
static void
debug_brkpnt_ok(
	GtkWidget*							widget,
	GtkWidget*							entry
)
{
	gchar*								cmdstr;
	gchar*								ptr;
	gI_debug_window*					dw;

	dw = main_window->debug_window;
	if(!dw)
	{
		return;
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(entry));
	if(!ptr || isempty(ptr))
	{
		cmdstr = g_strdup("break");
	}
	else
	{
		cmdstr = g_strdup_printf("break %s", ptr);
	}

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
	g_free(cmdstr);
	if(dw->brkpnt_window != NULL &&
		GTK_WIDGET_VISIBLE(dw->brkpnt_window->window))
	{
		/* update breakpoint window */
		gtk_clist_clear(GTK_CLIST(dw->brkpnt_window->list));
		debug_send_dbg_cmd(main_window->debug_window, "info breakpoints", TRUE);
		main_window->debug_window->wait_for = N_WAIT_BREAKPOINTS;
	}
}


/*
 * Set a new breakpoint
 */
static void
debug_brkpnt(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	entry_dialog(_("Enter breakpoint: "), _("Debug - Set breakpoint"),
		debug_brkpnt_ok);
}


/*
 * Create a new debug window
 */
static gI_debug_window*
debug_window_new(
	GideDocument*						document
)
{
	gI_debug_window*					dw;
	GtkWidget*							vbox;
	GtkWidget*							hbox;

	dw = g_malloc0(sizeof(gI_debug_window));
	dw->watch_window = NULL;
	dw->brkpnt_window = NULL;
	dw->sockname = NULL;
	dw->fh_in = -1;
	dw->fh_out = -1;
	dw->pid = -1;
	dw->state = N_GDB_UNDEF;
	dw->cmd_queue = NULL;
	dw->wait_for = -1;
	dw->breakpoints = NULL;

	dw->window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(dw->window), _("gIDE Debug Output Window"));
	gtk_widget_set_usize(dw->window, 600, 225);
	gtk_signal_connect(GTK_OBJECT(dw->window), "destroy",
		GTK_SIGNAL_FUNC(debug_window_destroy), (gpointer)dw);

	install_debug_popup(dw);

	vbox = gtk_vbox_new(0, FALSE);
	gtk_container_add(GTK_CONTAINER(dw->window), vbox);

	dw->text = gtk_text_new(NULL, NULL);
	gtk_box_pack_start(GTK_BOX(vbox), dw->text, TRUE, TRUE, 0);
	gtk_signal_connect_object(GTK_OBJECT(dw->text), "event",
		GTK_SIGNAL_FUNC(popup_debug_menu), GTK_OBJECT(dw->popup_menu));
	gtk_widget_grab_focus(dw->text);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, TRUE, 0);

	gtk_box_pack_start(GTK_BOX(hbox), gtk_label_new("gdb>"), FALSE, TRUE, 2);

	dw->entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), dw->entry, TRUE, TRUE, 2);
	gtk_signal_connect(GTK_OBJECT(dw->entry), "activate",
		GTK_SIGNAL_FUNC(send_command_cb), dw);
	gtk_widget_set_sensitive(dw->entry, FALSE);

	gtk_widget_show_all(vbox);

	return dw;
}


/*
 * Destroy the debug window
 */
static void
debug_window_destroy(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	/* stop the debugger */
	debug_stop_gdb(dw);

	/* destroy popup menu */
	gtk_widget_destroy(dw->popup_menu);

	/* free memory */
	g_free(dw);
	main_window->debug_window = NULL;

	/* set statusbar */
	gI_window_set_statusbar(main_window);
}


/*
 * Callback used when trying to attach the debugger to a running program
 */
static void
debug_attach_ok(
	GtkWidget*							widget,
	GtkWidget*							entry
)
{
	gchar*								cmdstr;
	gint								pid;

	if(!main_window->debug_window)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	pid = atoi(gtk_entry_get_text(GTK_ENTRY(entry)));
	if(pid > 0)
	{
		cmdstr = g_strdup_printf("attach %d", pid);
		debug_send_dbg_cmd(main_window->debug_window, cmdstr, TRUE);
		g_free(cmdstr);
	}
	else
	{
		gI_error_dialog(_("Invalid PID!"));
	}
}


/*
 * Create a new debugger command
 */
static gI_gdbcmd*
gI_gdbcmd_new(
	void
)
{
	gI_gdbcmd*							gdbcmd;

	gdbcmd = g_malloc0(sizeof(gI_gdbcmd));
	gdbcmd->cmd = NULL;
	gdbcmd->show = 1;
	gdbcmd->status = N_GDBCMD_TODO;

	return gdbcmd;
}


/*
 * Returns the the child PID of a process running under GDB control (uses ps)
 */
static gint
debug_get_child_pid(
	gI_debug_window*					dw
)
{
	FILE*								fp;
	gchar								buf[STRLEN];
	glong								pid;
	glong								ppid;
	glong								child_pid = -1;

	fp = popen("ps -j", "r");
	if(!fp)
	{
		perror("popen");
		return -1;
	}

	/* skip title line */
	fgets(buf, sizeof(buf), fp);

	while(fgets(buf, sizeof(buf), fp))
	{
		if(sscanf(buf, "%ld %ld", &ppid, &pid) == 2)
		{
			if(dw->pid == ppid)
			{
				child_pid = pid;
				break;
			}
		}
	}

	if(pclose(fp) != 0)
	{
		perror("pclose");
	}

	return child_pid;
}


/*
 * Handle pending GTK+ events
 */
static gint
debug_check_for_data(
	void
)
{
	gint								iteration = 0;

	while(gtk_events_pending() || gdk_events_pending())
	{
		iteration += gtk_main_iteration();
	}

	return iteration;
}


/*
 * Create the popup menu for the debugger window
 */
static void
install_debug_popup(
	gI_debug_window*					dw
)
{
	GnomeUIInfo							breakpoint_menu[] =
	{
		{
			GNOME_APP_UI_ITEM, _("Set"),
			_("Set a new breakpoint"),
			debug_brkpnt, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F9, 0, NULL*/
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Clear"),
			_("Clear an existing breakpoint"),
			gI_breakpoint_clear, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("View"),
			_("View all breakpoints"),
			gI_breakpoint_view, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F9, GDK_MOD1_MASK, NULL*/
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};
	GnomeUIInfo							program_menu[] =
	{
		{
			GNOME_APP_UI_ITEM, _("Break"),
			_("Break the program"),
			gI_debug_break, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F12, 0, NULL*/
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Continue"),
			_("Continue the program"),
			debug_cont, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F6, 0, NULL*/
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("File"),
			_("Set the program file"),
			debug_file, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Kill"),
			_("Kill the program"),
			debug_kill, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F5, GDK_CONTROL_MASK, NULL*/
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Run"),
			_("Start running the program"),
			debug_run, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F5, 0, NULL*/
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};
	GnomeUIInfo							popup_items[] =
	{
		GNOMEUIINFO_SUBTREE(_("Breakpoints"), breakpoint_menu),
		GNOMEUIINFO_SUBTREE(_("Program"), program_menu),
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, _("Backtrace"),
			_("Show the stack trace"),
			debug_backtrace, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
/*		{
			GNOME_APP_UI_ITEM, _("Exceptions"),
			_("Show the exceptions"),
			debug_exceptions, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},*/
		{
			GNOME_APP_UI_ITEM, _("Expressions"),
			_("Create a new expression"),
			debug_expressions, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		GNOMEUIINFO_SEPARATOR,
		{
			GNOME_APP_UI_ITEM, _("Run to cursor"),
			_("Continue until the cursor position"),
			debug_run_to_cursor, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Step into"),
			_("Next instruction, follow subroutines"),
			gI_debug_step_into, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F11, 0, NULL*/
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Step out"),
			_("Step out of the current function"),
			debug_step_out, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
			0, 0, NULL
		},
		{
			GNOME_APP_UI_ITEM, _("Step over"),
			_("Next instruction"),
			gI_debug_step_over, NULL, NULL,
			GNOME_APP_PIXMAP_NONE, NULL,
/*			GDK_F10, 0, NULL*/
			0, 0, NULL
		},
		GNOMEUIINFO_END
	};

	dw->popup_menu = gnome_popup_menu_new(popup_items);
}


/*
 * Show the popup menu for the debugger window
 */
static gint
popup_debug_menu(
	GtkWidget*							widget,
	GdkEvent*							ev
)
{
	if(ev->type == GDK_BUTTON_PRESS)
	{
		GdkEventButton*					event = (GdkEventButton*)ev;

		if(event->button == 3)
		{
			gnome_popup_menu_do_popup_modal(widget,
				NULL, NULL, event, main_window->debug_window);
			return TRUE;
		}
	}

	return FALSE;
}


/*
 * Kill the program being debugged
 */
static void
debug_kill(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_send_dbg_cmd(dw, "kill", TRUE);
}


/*
 * Step out of the current routine
 */
static void
debug_step_out(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_send_dbg_cmd(dw, "finish", TRUE);
}


/*
 * Run until the current cursor position in the current document
 */
static void
debug_run_to_cursor(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	GideDocument*						doc;
	gchar*								cmdstr;
	glong								line;
	gchar*								file;

	doc = gI_window_get_current_doc(main_window);
	if(!doc)
	{
		gI_error_dialog(_("No document available"));
		return;
	}

	if(!doc->filename)
	{
		gI_error_dialog(_("Current document doesn't have a filename"));
		return;
	}

	file = doc->filename;
	line = get_line_from_point(gI_text_get_point(doc));

	cmdstr = g_strdup_printf("until %s:%ld", file, line);

	debug_send_dbg_cmd(dw, cmdstr, TRUE);
	g_free(cmdstr);
}


/*
 * Get the current stacktrace
 */
static void
debug_backtrace(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	/* since we decided that there can be only one debug window.. */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_send_dbg_cmd(dw, "bt", TRUE);
	dw->wait_for = N_WAIT_BACKTRACE;
}


/*
 * Get information on local variables
 */
static void
debug_expressions(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	/* since we decided that there can be only one debug window.. */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_send_dbg_cmd(dw, "info locals", TRUE);
	dw->wait_for = N_WAIT_EXPRESSIONS;
}


#if 0
static void
debug_exceptions(
	GtkWidget*							widget,
	gI_debug_window*					dw
)
{
	/* since we decided that there can be only one debug window.. */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		return;
	}

	debug_send_dbg_cmd(dw, "info catch", TRUE);
	dw->wait_for = N_WAIT_EXCEPTIONS;

	/* needed for c++ debugging, output processing is not yet implemented */
}
#endif


/*
 * Callback which sets the program file to debug
 */
static void
debug_file_ok(
	GtkWidget*							widget,
	GtkWidget*							entry
)
{
	gchar*								ptr;
	gchar*								cmdstr;
	gI_debug_window*					dw;

	dw = main_window->debug_window;
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	ptr = gtk_entry_get_text(GTK_ENTRY(entry));
	if(!ptr || isempty(ptr))
	{
		return;
	}

	cmdstr = g_strdup_printf("file %s", ptr);
	debug_send_dbg_cmd(dw, cmdstr, TRUE);
	g_free(cmdstr);
}


/*
 * Set the program file to debug
 */
static void
debug_file(
	GtkWidget*							widget,
	gpointer							data
)
{
	entry_dialog(_("Enter file: "), _("Debug - Set program file"),
		debug_file_ok);
}


/*
 * Get the name of the main executable from the project
 */
static gchar*
debug_get_project_exe(
	gI_project*							project
)
{
	gchar*								filename;
	gchar*								str;

	filename = g_strdup(project->mtarget);

	if(!file_exist(filename))
	{
		gchar*							mtarget;

		if(isempty(project->prjroot))
		{
			str = g_strdup_printf(_("Can't find executable '%s'!"),
				filename);
			gI_error_dialog(str);
			g_free(str);
			g_free(filename);
			return NULL;
		}

		mtarget = g_strconcat(project->prjroot, "/", filename, NULL);
		g_free(filename);

		if(!file_exist(mtarget))
		{
			str = g_strdup_printf(_("Can't find executable '%s'!"), mtarget);
			gI_error_dialog(str);
			g_free(str);
			g_free(mtarget);
			return NULL;
		}

		return mtarget;
	}
	return filename;
}


/*
 * Try to determine the exe file to debug
 */
static gchar*
debug_get_exe(
	GideDocument*						doc
)
{
	gchar*								filename;
	gchar*								str;
	gchar*								ptr;
	glong								exec_mod;
	glong								src_mod;

	/* check whether the current document was saved */
	if(!doc->filename)
	{
		if(!doc->changed)
		{
			gI_error_dialog(_("Can't debug an empty file"));
			return NULL;
		}

		gI_error_dialog(_("Save and compile your source first"));
		return NULL;
	}

	/* check whether executable file exists */
	filename = g_strdup(doc->filename);
	ptr = strrchr(filename, '.');
	if(!ptr)
	{
		g_free(filename);
		gI_error_dialog(_("Don't know how to find the executable"));
		return NULL;
	}
	*ptr = '\0';
	if(!file_exist(filename))
	{
		str = g_strdup_printf(_("Can't find executable '%s'!"), filename);
		gI_error_dialog(str);
		g_free(str);
		g_free(filename);
		return NULL;
	}

	/* see whether the exe is up to date */
	exec_mod = get_last_mod(filename);
	src_mod = get_last_mod(doc->filename);
	if(src_mod > exec_mod)
	{
		gI_error_dialog(_("Source file is newer than executable "
			"file!\n(maybe you should recompile it)"));
	}

	return filename;
}


/*
 * Process the debugger output and determine the state
 */
static void
debug_window_parse_state(
	gI_debug_window*					dw,
	gchar*								buf
)
{
	gchar*								ptr;

	ptr = &buf[strlen(buf)];

	ptr -= strlen(GDB_PROMPT);

	if(!strcmp(ptr, GDB_PROMPT))
	{
		if(strstr(buf, "\032\032"))
		{
			dw->state = N_GDB_BREAK;
		}
		else
		{
			dw->state = N_GDB_IDLE;
		}
	}
	else
	{
		dw->state = N_GDB_RUNNING;
	}
}


/* debugger functions, callable from debug sources only */
/*
 * Queue/Send Command to GDB
 */
glong
debug_send_dbg_cmd(
	gI_debug_window*					dw,
	gchar*								cmd,
	glong								show
)
{
	GList*								queue;
	gI_gdbcmd*							gdbcmd;
	gI_gdbcmd*							newcmd;

	/* since we decided that there can be only one debug window.. */
	if(!dw)
	{
		dw = main_window->debug_window;
	}
	if(!dw)
	{
		return 0;
	}

	if(dw->pid == -1)
	{
		return 0;
	}

	if(isempty(cmd))
	{
		strcpy(cmd, "\n");
	}

	/* add new cmd to queue */
	newcmd = gI_gdbcmd_new();
	newcmd->cmd = g_strdup(cmd);
	newcmd->show = show;
	dw->cmd_queue = g_list_append(dw->cmd_queue, (gpointer)newcmd);

	/* process command queue */
	queue = dw->cmd_queue;
	while(queue != NULL)
	{
		gdbcmd = (gI_gdbcmd*)queue->data;
		if(!gdbcmd)
		{
			queue = queue->next;
			continue;
		}

		if(!gdbcmd->cmd)
		{
			queue = queue->next;
			continue;
		}

		if(gdbcmd->status == N_GDBCMD_DONE)
		{
			queue = queue->next;
			continue;
		}

		while(dw->state != N_GDB_IDLE && dw->state != N_GDB_BREAK)
		{
			gtk_main_iteration_do(TRUE);
		}

		/* ...or process queue */
		if(gdbcmd->cmd[0] == '\n')
		{
			write(dw->fh_in, "\n", 1);

			if(gdbcmd->show)
			{
				gtk_text_insert(GTK_TEXT(dw->text), NULL, NULL, NULL, "\n", 1);
			}
		}
		else
		{
			if(write(dw->fh_in, gdbcmd->cmd, strlen(gdbcmd->cmd)) > 0)
			{
				write(dw->fh_in, "\n", 1);

				if(gdbcmd->show)
				{
					gtk_text_insert(GTK_TEXT(dw->text), NULL, NULL, NULL,
						gdbcmd->cmd, strlen(gdbcmd->cmd));
					gtk_text_insert(GTK_TEXT(dw->text), NULL, NULL, NULL, "\n",
						1);
				}

				gdbcmd->status = N_GDBCMD_DONE;
			}

			/* set state to running */
			dw->state = N_GDB_RUNNING;

			/* bsd 4.3 function, hope thats works for all */
			usleep(500);
		}
		queue = queue->next;
	}

	return 1;
}


/* public functions */
/*
 * Start a debugging session
 */
void
gI_debug_start(
	GtkWidget*							widget,
	gpointer							data
)
{
	GideDocument*						currentDoc;
	gchar*								filename;
	gI_project*							project;
	gchar*								cmdstr;

	g_assert(IS_GIDE_WINDOW(data));

	currentDoc = gI_window_get_current_doc(main_window);

	if(!currentDoc)
	{
		gI_error_dialog(_("No current document; nothing to debug"));
		return;
	}

	if(main_window->debug_window)
	{
		gI_error_dialog(_("Already a debugging session in progress"));
		return;
	}

	main_window->debug_window = debug_window_new(currentDoc);
	gtk_widget_show(main_window->debug_window->window);
	debug_start_gdb(main_window->debug_window);

	/* special handling for projects
	 * project handling has priority, maybe we need a pref. switch to
	 * decide this in the future...
	 */
	project = gI_project_get_current();
	if(project && project->mtarget && !isempty(project->mtarget))
	{
		filename = debug_get_project_exe(project);
	}
	else
	{
		filename = debug_get_exe(currentDoc);
	}

	if(!filename)
	{
		return;
	}

	/* process gtk stuff */
	debug_check_for_data();

	/* set prompt */
	debug_send_dbg_cmd(main_window->debug_window, "set prompt " GDB_PROMPT,
		TRUE);

	/* set confirm off */
	debug_send_dbg_cmd(main_window->debug_window, "set confirm off", TRUE);

	/* set limit to 50 gchars */
	debug_send_dbg_cmd(main_window->debug_window, "set print elements 50",
		TRUE);

	/* stop on NULL */
	debug_send_dbg_cmd(main_window->debug_window, "set print null-stop", TRUE);

	/* load executable */
	cmdstr = g_strdup_printf("file %s", filename);
	debug_send_dbg_cmd(main_window->debug_window, cmdstr, TRUE);
	g_free(cmdstr);

	g_free(filename);
}


/*
 * Go the the next instruction, ignoring subroutines
 */
void
gI_debug_step_over(
	GtkWidget*							widget,
	gpointer							data
)
{
	GideDocument*						current;

	current = gI_window_get_current_doc(main_window);
	if(!current)
	{
		gI_error_dialog(_("No document available"));
		return;
	}

	if(!main_window->debug_window)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_send_dbg_cmd(main_window->debug_window, "next", TRUE);
}


/*
 * Go to the next insctruction and follow subroutines
 */
void
gI_debug_step_into(
	GtkWidget*							widget,
	gpointer							data
)
{
	GideDocument*						current;

	current = gI_window_get_current_doc(main_window);
	if(!current)
	{
		gI_error_dialog(_("No document available!"));
		return;
	}

	if(!main_window->debug_window)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_send_dbg_cmd(main_window->debug_window, "step", TRUE);
}


/*
 * Break the executable being debugged
 */
void
gI_debug_break(
	GtkWidget*							widget,
	gpointer							data
)
{
	gI_debug_window*					dw;
	gint								pid;

	if(!(dw = main_window->debug_window))
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	pid = debug_get_child_pid(dw);
	if(pid <= 0)
	{
		gI_error_dialog(_("Can't find the process being debugged"));
		return;
	}

	kill(pid, SIGINT);
}


/*
 * Stop a debugging session
 */
void
gI_debug_stop(
	GtkWidget*							widget,
	gpointer							data
)
{
	gI_debug_window*					dw;

	dw = main_window->debug_window;
	if(!dw)
	{
		gI_error_dialog(_("No debug window available"));
		return;
	}

	debug_stop_gdb(dw);

	/* close all debug windows */
	if(dw->brkpnt_window != NULL &&
		GTK_WIDGET_VISIBLE(dw->brkpnt_window->window))
	{
		gtk_widget_destroy(dw->brkpnt_window->window);
	}

	if(dw->watch_window != NULL &&
		GTK_WIDGET_VISIBLE(dw->watch_window->window))
	{
		gtk_widget_destroy(dw->watch_window->window);
	}

	if(dw->bt_window != NULL && GTK_WIDGET_VISIBLE(dw->bt_window->window))
	{
		gtk_widget_destroy(dw->bt_window->window);
	}

	if(GTK_WIDGET_VISIBLE(dw->window))
	{
		gtk_widget_destroy(dw->window);
	}
}


/*
 * Attach the debugger to an existing process
 */
void
gI_debug_attach(
	GtkWidget*							widget,
	gpointer							data
)
{
	/* FIXME: offer process (c)list here... */

	entry_dialog(_("Enter PID: "), _("Attach to process"), debug_attach_ok);
}
