/*
 * dialog-search-replace.c: Dialog used to search/replace in gIDE
 * spreadsheet
 *
 * Authors:
 *   JP Rosevear (jpr@arcavia.com)
 */
#include <config.h>
#include <dirent.h>
#include <fnmatch.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_GNU_REGEXP
#include <regex.h>
#endif

#include "gide.h"
#include <glade/glade.h>
#include "dialogs.h"
#include "gI_plugin.h"
#include "gI_common.h"
#include "gI_search.h"

#define GLADE_FILE "file-search-replace.glade"

typedef struct 
{
	GideWindow *window;
	GideDocument *document;
	GtkWidget *dialog;
	GtkWidget *file_entry;
	GtkWidget *pat_entry;
	GtkWidget *search_entry;
	GtkWidget *replace_entry;

	GtkWidget *re;
	GtkWidget *cs;
	GtkWidget *sub;
} SearchReplace;

enum { BUTTON_SEARCH = 0, BUTTON_REPLACE, BUTTON_CLOSE };

static void
search_not_found( SearchReplace *sr, FileSearchReplaceState *state )
{
	gchar *error;
	
	error = g_strdup_printf( _("[%s] not found!\n"), state->search );
	gI_error_dialog( error );
	g_free( error );
}

static void
search_no_dir( SearchReplace *sr, FileSearchReplaceState *state )
{
	gchar *error;
	
	error = g_strdup_printf( _("No directory was entered!") );
	gI_error_dialog( error );
	g_free( error );
}

static void
search_no_text( void )
{
	gchar *error;
	
	error = g_strdup_printf( _("No search text was entered!") );
	gI_error_dialog( error );
	g_free( error );
}

static void
search_no_regexp( void )
{
	gchar *error;

	error = g_strdup_printf( _("Regular Expression Search only supported with the GNU regexp lib") );	
	gI_error_dialog( error );
	g_free( error );
}

static void
search_regexp_compile_error( gchar *expr )
{
	gchar *error;

	error = g_strdup_printf( _("Regex Compile Error: %s"), expr );	
	gI_error_dialog( error );
	g_free( error );
}

static void
search_regex_not_impl( void )
{
	gchar *error;

	error = g_strdup_printf( _("Regular Expression search and replace in files not supported!") );	
	gI_error_dialog( error );
	g_free( error );
}

static void
search_state_extract( SearchReplace *sr, FileSearchReplaceState *state,
		      gboolean replace )
{
	/* The directory and files to search */
	if (state->dir)
		g_free( state->dir );
	state->dir = gtk_editable_get_chars(GTK_EDITABLE(sr->file_entry),
					    0, -1);
	if (state->pat)
		g_free( state->pat );
	state->pat = gtk_editable_get_chars(GTK_EDITABLE(sr->pat_entry),
					    0, -1);
	if( isempty( state->pat ) ) {
		state->pat = g_strdup("*.*");
	}

	/* The search variables */
	if (state->search)
		g_free( state->search );
	state->search = gtk_editable_get_chars(GTK_EDITABLE(sr->search_entry),
					       0, -1);

	state->re = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sr->re));
	state->cs = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sr->cs));
	state->sub = gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(sr->sub));

	/* The replace variables */
	if (replace) {
		if ( state->replace )
			g_free( state->replace );	
		state->replace =
			gtk_editable_get_chars(GTK_EDITABLE(sr->replace_entry),
					       0, -1);
	} else {
		if ( state->replace )
			g_free( state->replace );	
		state->replace = NULL;
	}
}

static void
search_file_find( GideDocument *document, FileSearchReplaceState *state,
		  gchar *filename )
{
	FILE *fp;
	gchar MsgStr[STRLEN];
	gint line;
	gint hit;
	gint i;
	gchar buf[STRLEN];
#ifdef HAVE_GNU_REGEXP
	gchar errbuf[256];
	regex_t buffer;
	gint compile;
	regmatch_t regs[1];
	gint match;
#endif

	fp = fopen( filename, "r" );
	if( !fp ) {
		g_snprintf( MsgStr, sizeof(MsgStr),
			    _("\nUnable to Open File: '%s'\n"),
			    filename );
		gI_document_insert_text_at_point( document, MsgStr );
		return;
	}

	if( state->re ) {
#ifdef HAVE_GNU_REGEXP
		compile = regcomp( &buffer, state->search, REG_EXTENDED );
		if( compile != 0 ) {
			regerror( compile, &buffer, errbuf, 256 );
			search_regexp_compile_error( errbuf );
			return;
		}
#else
		search_no_regexp();
		return;
#endif
	}

	line = 0;

	while( fgets( buf, sizeof(buf), fp ) ) {
		line++;

		if( state->re ) {
#ifdef HAVE_GNU_REGEXP
			match = regexec( &buffer, buf, 1, regs, 0 );
			if( match == 0 ) {
				/* matched */
				g_snprintf( MsgStr, sizeof(MsgStr),
					    "%s (%d): %s\n",
					    filename, line, buf );
				gI_document_insert_text_at_point( document,
								  MsgStr );
			}
#endif

			continue;
		}

		if( strlen(buf) < strlen(state->search) )
			continue;

		for(i=0; i<=(strlen(buf)-strlen(state->search)); i++) {
			if( state->cs ) {
				hit = strncasecmp( &buf[i], state->search,
						   strlen(state->search) );
			} else {
				hit = strncmp( &buf[i], state->search,
					       strlen(state->search) );
			}

			if( hit == 0 ) {
				g_snprintf( MsgStr, sizeof(MsgStr),
					    "%s (%d): %s\n", filename,
					    line, buf );

				gI_document_insert_text_at_point( document,
								  MsgStr );
				break;
			}
		}
	}

	fclose( fp );

#ifdef HAVE_GNU_REGPEXP
	regfree( &buffer );
#endif
}

static void
search_file_replace( GideDocument *document, FileSearchReplaceState *state,
		     gchar *filename )
{
	FILE *fp, *fp_out;
	gchar MsgStr[STRLEN];
	gint line;
	gint hit;
	gint i;
	gboolean found = FALSE;
	gchar buf[STRLEN];
	gchar tmp[STRLEN];
	gchar *tmpn;
	gchar *bakfile;
#ifdef HAVE_GNU_REGEXP
	gchar errbuf[256];
	gchar* errmsg = NULL;
	regex_t buffer;
	gint compile;
	regmatch_t regs[1];
	gint match;
#endif

	fp = fopen( filename, "r" );
	if (!fp) {
		g_snprintf( MsgStr, sizeof(MsgStr),
			    _("\nUnable to Open File: '%s'\n"), filename );
		gI_document_insert_text_at_point( document, MsgStr );
		return;
	}

	tmpn = tempnam(g_dirname(filename), "gide");
	fp_out = fopen( tmpn, "w" );
	if (!fp_out) {
		g_snprintf( MsgStr, sizeof(MsgStr),
			    _("\nUnable to Open File: '%s'\n"), tmpn );
		gI_document_insert_text_at_point( document, MsgStr );
		return;
	}

	line = 0;

	while( fgets( buf, sizeof(buf), fp ) ) {
		line++;

		if( state->re ) {
#ifdef HAVE_GNU_REGEXP
#if 0
			compile = regcomp( &buffer, state->search,
					   REG_EXTENDED );
			if (compile != 0) {
				regerror( compile, &buffer, errbuf, 256 );
				search_regexp_compile_error(errbuf);
				return;
			}

			match = regexec( &buffer, buf, 1, regs, 0 );
			if( match == 0 ) {
				/* matched */
				g_print( "replace not implemented!\n" );
				return;
			}
#endif
#else
			search_no_regexp();
			return;
#endif

			continue;
		}

		if( strlen(buf) < strlen(state->search) ) {
			fputs( buf, fp_out );
			continue;
		}
		
		for(i=0; i<=(strlen(buf)-strlen(state->search)); i++) {
			if( state->cs ) {
				hit = strncasecmp( &buf[i], state->search,
						   strlen(state->search) );
			} else {
				hit = strncmp( &buf[i], state->search,
					       strlen(state->search) );
			}

			if (hit == 0) {
				g_snprintf( MsgStr, sizeof(MsgStr),
					    "%s (%d): %s\n", filename,
					    line, buf );
				gI_document_insert_text_at_point( document,
								  MsgStr );
				strncpy( tmp, buf, i );
				tmp[i] = '\0';
				strcat( tmp, state->replace );
				strcat( tmp, &buf[i+(strlen(state->search))] );
				strcpy( buf, tmp );
				i += strlen(state->replace);
				found = TRUE;
			}
		}

		fputs( buf, fp_out );
	}

	fclose( fp );
	fclose( fp_out );

	if (found) {
		bakfile = g_strconcat( filename, ".bak", NULL );
		remove( bakfile );
		rename( filename, bakfile );
		g_free( bakfile );
		rename( tmpn, filename );
	} else {
		remove( tmpn );
	}
	free( tmpn );
}

static gint
search_dir( GideDocument *document, FileSearchReplaceState *state,
	    gboolean repl )
{
	struct stat sb;
	DIR *d;
	struct dirent *entry;
	gchar *fnwp;

	if( stat( state->dir, &sb ) < 0 ) {
		return -1;
	}

	if( !S_ISDIR( sb.st_mode ) ) {
		return -1;
	}

	d = opendir( state->dir );
	if( !d ) {
		return -1;
	}

	while( (entry = readdir( d )) ) {
		/* avoid endless recursion... */
		if( !strcmp( entry->d_name, "." )
		        || !strcmp( entry->d_name, ".." ) )
			continue;

		fnwp = g_strconcat( state->dir, "/", entry->d_name, NULL );

		if( stat( fnwp, &sb ) < 0 ) {
			return -1;
		}

		if( S_ISDIR( sb.st_mode ) && !state->sub ) {
			continue;
		}

		if( S_ISDIR( sb.st_mode ) ) {
			/* recursive.. */
			gchar *temp = state->dir;

			state->dir = fnwp;
			search_dir( document, state, repl );
			state->dir = temp;
			continue;
		}

		/* check if its a regular file (or a symlink), here ... */
		if( !(S_ISREG( sb.st_mode ) || S_ISLNK( sb.st_mode )) ) {
			continue;
		}

		if( fnmatch( state->pat, entry->d_name, FNM_NOESCAPE ) == 0 ) {
			/* matched */
			if( repl ) {
				search_file_replace( document, state, fnwp );
			} else {
				search_file_find( document, state, fnwp );
			}
		}

		g_free( fnwp );
	}

	closedir( d );

	return 0;
}

static void
search_cb (SearchReplace *sr, FileSearchReplaceState *state)
{
	GideDocument *document;
	
	search_state_extract( sr, state, FALSE );

	/* Trap some errors */
	if( isempty( state->dir ) ) {
		search_no_dir( sr, state );
		return;
	}
	if( isempty( state->search ) ) {
		search_no_text();
		return;
	}
		
#ifndef HAVE_GNU_REGEXP
	if( state->re ) {
		search_no_regexp();
		return;
	}
#endif

	document = GIDE_DOCUMENT(gI_document_new());
	gI_window_add_doc( sr->window, document );
	gI_window_set_doc_label (sr->window, document,
				 _("--- Find Results ---") );

	search_dir( document, state, FALSE );
}

static void
replace_cb (SearchReplace *sr, FileSearchReplaceState *state)
{
	GideDocument *document;
	
	search_state_extract( sr, state, TRUE );

		/* Trap some errors */
	if( isempty( state->dir ) ) {
		search_no_dir( sr, state );
		return;
	}
	if( isempty( state->search ) ) {
		search_no_text();
		return;
	}
		
#ifndef HAVE_GNU_REGEXP
	if( state->re ) {
		search_no_regexp();
		return;
	}
#else
	if (state->re) {
		search_regex_not_impl();
		return;
	}
#endif

	document = GIDE_DOCUMENT(gI_document_new());
	gI_window_add_doc( sr->window, document );
	gI_window_set_doc_label (sr->window, document,
				 _("--- Find/Replace Results ---") );

	search_dir( document, state, TRUE );
}

static gint
sr_key_event (GtkWidget *sr, GdkEventKey *event)
{
	if (event->keyval == GDK_Escape) {
		gtk_widget_destroy (sr);
		return 1;
	}
	return 0;
}

static void
dialog_file_search_replace_impl (GideWindow *window, GladeXML *gui,
				 FileSearchReplaceState *state)
{
	SearchReplace *sr = g_new0 (SearchReplace, 1);
	gint bval = BUTTON_SEARCH;

	g_return_if_fail (sr);
	g_return_if_fail (window);
	
	sr->window = window;
	sr->document = gI_window_get_current_doc (window);
	sr->dialog = glade_xml_get_widget (gui, "dialog");
	sr->file_entry = glade_xml_get_widget (gui, "file_entry");
	sr->pat_entry = glade_xml_get_widget (gui, "pat_entry");
	sr->search_entry = glade_xml_get_widget (gui, "search_entry");
	sr->replace_entry = glade_xml_get_widget (gui, "replace_entry");
	sr->re = glade_xml_get_widget (gui, "re");
	sr->cs = glade_xml_get_widget (gui, "cs");
	sr->sub = glade_xml_get_widget (gui, "sub");
	
	/* Set defaults */
	if ( state->dir )
		gtk_entry_set_text( GTK_ENTRY(sr->file_entry),
		state->dir );
	if ( state->pat )
		gtk_entry_set_text( GTK_ENTRY(sr->pat_entry),
		state->pat );
	if ( state->search )
		gtk_entry_set_text( GTK_ENTRY(sr->search_entry),
		state->search );

	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sr->re), state->re );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sr->cs), state->cs );
	gtk_toggle_button_set_active( GTK_TOGGLE_BUTTON(sr->sub),
				      state->sub );

	gnome_dialog_set_default (GNOME_DIALOG (sr->dialog), BUTTON_CLOSE);
	gtk_window_set_policy (GTK_WINDOW (sr->dialog), FALSE, TRUE, FALSE);

	gtk_widget_show_all (GNOME_DIALOG (sr->dialog)->vbox);

	bval = gnome_dialog_run (GNOME_DIALOG (sr->dialog));
	switch (bval) {
		
	case BUTTON_SEARCH:
		search_cb (sr, state);
		break;
		
	case BUTTON_REPLACE:
		replace_cb (sr, state);
		break;
		
	case BUTTON_CLOSE:
		break;
		
	case -1: /* close window */
		return;
		
	default: /* should never happen */
		break;
	}

	/* If the user canceled we have already returned */
	gnome_dialog_close (GNOME_DIALOG (sr->dialog));
}

/*
 * Wrapper around dialog_file_search_replace_impl
 * To libglade'ify it
 */
void
dialog_file_search_replace( GideWindow *window, FileSearchReplaceState *state )
{
	GladeXML *gui;

	g_return_if_fail (window != NULL);

	gui = glade_xml_new (GIDE_GLADEDIR "/" GLADE_FILE , NULL);
	if (!gui) {
		printf ("Could not find " GLADE_FILE "\n");
		return;
	}

	dialog_file_search_replace_impl( window, gui, state );
	gtk_object_unref (GTK_OBJECT (gui));
}



















