/* GDM - The Gnome Display Manager
 * Copyright (C) 1998, 1999 Martin Kasper Petersen <mkp@SunSITE.auc.dk>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <confdefs.h>
#include <config.h>
#include <gnome.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <strings.h>
#include <pwd.h>
#include <grp.h>
#include <syslog.h>
#include <sys/wait.h>
#include <errno.h>

#include "gdm.h"

static const gchar RCSid[]="$Id: gdm.c,v 1.37 1999/01/27 22:11:22 mkp Exp $";

extern void gdm_slave_start(gdmDisplayType *);
extern gchar **gdm_arg_munch(const gchar *p);
extern void gdm_abort(const gchar *, ...);
extern void gdm_server_start(gdmDisplayType *d);
extern void gdm_server_stop(gdmDisplayType *d);
extern void gdm_server_restart(gdmDisplayType *d);
extern void gdm_debug(const gchar *format, ...);

void gdm_display_manage(gdmDisplayType *d);
static void gdm_local_servers_start(gdmDisplayType *d);
static void gdm_display_unmanage(gdmDisplayType *d);

static GSList *gdmDisplays;
static gchar *gdmUser;
static gchar *gdmGroup;

uid_t gdmUserId;
gid_t gdmGroupId;
gchar *gdmSessDir = NULL;
gchar *gdmGreeter = NULL;
gchar *gdmLogDir = NULL;
gchar *gdmDisplayInit = NULL;
gchar *gdmPreSession = NULL;
gchar *gdmPostSession = NULL;
gchar *gdmHalt = NULL;
gchar *gdmReboot = NULL;
gchar *gdmAuthDir = NULL;
gchar *gdmUserAuthFile = NULL;
gchar *gdmSuspend = NULL;
gchar *gdmPidFile = NULL;
gchar *gdmDefaultPath = NULL;
gint  gdmKillInitClients = 0;
gint  gdmUserMaxFile = 0;

typedef struct _childstat ChildStat;
struct _childstat { pid_t pid; gint status; };


static gdmDisplayType * 
gdm_display_alloc (gint id, gchar *command)
{
    gchar dname[1024];

    gdmDisplayType *d = g_malloc(sizeof(gdmDisplayType));

    sprintf(dname, ":%d", id);    
    d->name = strdup(dname);
    d->id = id;
    d->servpid = 0;
    d->slavepid = 0;
    d->greetpid = 0;
    d->sesspid = 0;
    d->command = g_strdup(command);
    d->type=0;
    d->auth = NULL;
    d->servstat = SERVER_DEAD;
    d->dispstat = DISPLAY_DEAD;
    d->cookie = NULL;

    return (d);
}


static void 
gdm_config_parse (void)
{
    gchar *k, *v;
    void *iter;
    struct passwd *pwent;
    struct group *grent;
    struct stat statbuf;
    
    gdmDisplays = NULL;

    if(stat(GDM_CONFIG_FILE, &statbuf) == -1)
	gdm_abort(_("gdm_config_parse: No configuration file: %s. Aborting."), GDM_CONFIG_FILE);

    gnome_config_push_prefix ("=" GDM_CONFIG_FILE "=/");
    gdmPidFile=g_strdup(gnome_config_get_string("daemon/pidfile=/var/run/gdm.pid"));
    gdmUser=g_strdup(gnome_config_get_string("daemon/user=gdm"));
    gdmGroup=g_strdup(gnome_config_get_string("daemon/group=gdm"));
    gdmGreeter=g_strdup(gnome_config_get_string("daemon/greeter"));
    gdmSessDir=g_strdup(gnome_config_get_string("daemon/sessiondir"));
    gdmLogDir=g_strdup(gnome_config_get_string("daemon/logdir"));
    gdmDisplayInit=g_strdup(gnome_config_get_string("daemon/displayinitdir"));
    gdmPreSession=g_strdup(gnome_config_get_string("daemon/presessionscriptdir"));
    gdmPostSession=g_strdup(gnome_config_get_string("daemon/postsessionscriptdir"));
    gdmHalt=g_strdup(gnome_config_get_string("daemon/haltcommand=/sbin/shutdown -h now"));
    gdmReboot=g_strdup(gnome_config_get_string("daemon/rebootcommand=/sbin/shutdown -r now"));
    gdmSuspend=g_strdup(gnome_config_get_string("daemon/suspendcommand"));
    gdmAuthDir=g_strdup(gnome_config_get_string("daemon/authdir"));
    gdmUserAuthFile=g_strdup(gnome_config_get_string("daemon/userauthfile=.Xauthority"));
    gdmKillInitClients=gnome_config_get_int("daemon/killinitclients=1");
    gdmUserMaxFile=gnome_config_get_int("system/UserFileCutoffSize=65536");
    gdmDefaultPath=g_strdup(gnome_config_get_string("daemon/defaultpath=/bin:/usr/bin:/usr/X11/bin:/usr/local/bin"));

    if(gdmGreeter==NULL && stat("/usr/local/bin/gdmgreeter", &statbuf)==0)
    	gdmGreeter="/usr/local/bin/gdmgreeter";

    if(gdmGreeter==NULL && stat("/usr/bin/gdmgreeter", &statbuf)==0)
    	gdmGreeter="/usr/bin/gdmgreeter";

    if(gdmGreeter==NULL && stat("/opt/gnome/bin/gdmgreeter", &statbuf)==0)
    	gdmGreeter="/opt/gnome/bin/gdmgreeter";
    	
    if(gdmGreeter==NULL)
	gdm_abort(_("gdm_config_parse: No greeter specified and default not found."));

    if(gdmAuthDir==NULL && stat("/var/gdm", &statbuf)==0)
    	gdmGreeter="/var/gdm";

    if(gdmAuthDir==NULL && stat("/var/lib/gdm", &statbuf)==0)
    	gdmGreeter="/var/lib/gdm";

    if(gdmAuthDir==NULL && stat("/opt/gnome/var/gdm", &statbuf)==0)
    	gdmGreeter="/opt/gnome/var/gdm";
    	
    if(gdmAuthDir==NULL)
	gdm_abort(_("gdm_config_parse: No authdir specified and default not found."));

    if(gdmLogDir==NULL) 
	gdmLogDir=gdmAuthDir;

    if(gdmSessDir==NULL && stat("/usr/local/etc/gdm/Sessions", &statbuf)==0)
    	gdmSessDir="/usr/local/etc/gdm/Sessions";

    if(gdmSessDir==NULL && stat("/usr/etc/gdm/Sessions", &statbuf)==0)
    	gdmSessDir="/usr/etc/gdm/Sessions";    

    if(gdmSessDir==NULL && stat("/etc/gdm/Sessions", &statbuf)==0)
    	gdmSessDir="/etc/gdm/Sessions";    

    if(gdmSessDir==NULL && stat("/etc/X11/gdm/Sessions", &statbuf)==0)
    	gdmSessDir="/etc/X11/gdm/Sessions";    

    if(gdmSessDir==NULL) 
	gdm_abort(_("gdm_config_parse: No sessions directory specified and default not found."));

    gnome_config_pop_prefix();

    iter=gnome_config_init_iterator("=" GDM_CONFIG_FILE "=/servers");
    iter=gnome_config_iterator_next (iter, &k, &v);
    
    while (iter) {
	gdmDisplays=g_slist_append(gdmDisplays, gdm_display_alloc(atoi(k), v));
	iter=gnome_config_iterator_next (iter, &k, &v);
    };

    if(gdmDisplays==NULL)	/* Alan, this is perfectly legal. Think XDMCP. */
	syslog(LOG_INFO, _("gdm_config_parse: No local servers defined. XDMCP only."));

    pwent = getpwnam(gdmUser);

    if(!pwent)
	gdm_abort(_("gdm_config_parse: Can't find the gdm user (%s). Aborting!"), gdmUser);
    else 
	gdmUserId=pwent->pw_uid;    

    grent = getgrnam(gdmGroup);

    if(!grent)
	gdm_abort(_("gdm_config_parse: Can't find the gdm group (%s). Aborting!"), gdmGroup);
    else 
	gdmGroupId=grent->gr_gid;   

    setegid(gdmGroupId);	/* gid remains `gdm' */
    seteuid(gdmUserId);

    /* Enter paranoia mode */
    if(stat(gdmAuthDir, &statbuf) == -1) 
	gdm_abort(_("gdm_config_parse: Authdir %s does not exist. Aborting."), gdmAuthDir);

    if(! S_ISDIR(statbuf.st_mode))
	gdm_abort(_("gdm_config_parse: Authdir %s is not a directory. Aborting."), gdmAuthDir);

    if(statbuf.st_uid != gdmUserId || statbuf.st_gid != gdmGroupId) 
	gdm_abort(_("gdm_config_parse: Authdir %s is not owned by user %s, group %s. Aborting."), gdmAuthDir, gdmUser, gdmGroup);

    if(statbuf.st_mode != (S_IFDIR|S_IRWXU|S_IRGRP|S_IXGRP)) 
	gdm_abort(_("gdm_config_parse: Authdir %s has wrong permissions. Should be 750. Aborting."), gdmAuthDir, statbuf.st_mode);

    seteuid(0);
}


static void 
gdm_local_servers_start(gdmDisplayType *d)
{
    if(d->type == DISPLAY_LOCAL)
	gdm_server_start(d);
}


void 
gdm_display_manage(gdmDisplayType *d)
{
    sigset_t mask, omask;

    gdm_debug(_("gdm_display_manage: Managing %s"), d->name);

    /* If we have an old subprocess hanging around, kill it */
    if(d->slavepid) {
	sigemptyset(&mask);
	sigaddset(&mask, SIGCHLD);
	sigprocmask(SIG_BLOCK, &mask, &omask);

	kill(d->slavepid, SIGINT);
	waitpid(d->slavepid, 0, 0);

	sigprocmask(SIG_SETMASK, &omask, NULL);
    };

    switch(d->slavepid=fork()) {

    case 0:
	if(d->type == DISPLAY_LOCAL && d->servstat == SERVER_RUNNING)
	    gdm_slave_start(d);
	break;

    case -1:
	gdm_abort(_("gdm_display_manage: Failed forking gdm slave process for %d"), d->name);

    default:
	gdm_debug(_("gdm_display_manage: Forked slave: %d...\n"), d->slavepid);
	break;

    };
}


static void
gdm_child_action(gdmDisplayType *d, ChildStat *slave)
{
    sigset_t mask, omask;
    gchar **argv;

    gdm_debug(_("gdm_child_action: %s"), d->name);

    sigemptyset(&mask);
    sigaddset(&mask, SIGCHLD);
    sigprocmask(SIG_BLOCK, &mask, &omask);

    /* X server died */
    if(slave->pid == d->servpid) {
	d->servpid=0;

	switch(slave->status) {

	case SERVER_SUCCESS:
	case SERVER_FAILURE:
	case SERVER_RUNNING:

	    d->dispstat=DISPLAY_MANAGED;

	    gdm_server_start(d); /* FIXME: # of retries in case of failure */
	    break;

	case SERVER_NOTFOUND:
	case SERVER_ABORT:
	    d->dispstat=DISPLAY_ABORT;
	    break;

	default:
	    gdm_debug(_("gdm_child_action: Server process returned unknown status %d"), slave->status);
	    break;
	}
    };
    
    /* Slave died */
    if(slave->pid==d->slavepid) {
	d->slavepid=0;
	
	switch(slave->status) {
	    
	case DISPLAY_REMANAGE:

	    if(d->type == DISPLAY_LOCAL && d->dispstat != DISPLAY_ABORT && d->servstat != SERVER_PHOENIX) {
		d->dispstat=DISPLAY_DEAD;
		gdm_server_restart(d);
	    };

	    break;

	case DISPLAY_ABORT:
	    syslog(LOG_INFO, _("gdm_child_action: Aborting display %s"), d->name);
	    d->dispstat=DISPLAY_ABORT;
	    break;

	case DISPLAY_REBOOT:
	    syslog(LOG_INFO, _("gdm_child_action: Master rebooting..."));
	    g_slist_foreach(gdmDisplays, (GFunc) gdm_display_unmanage, NULL);
	    closelog();
	    unlink(gdmPidFile);
	    argv=gdm_arg_munch(gdmReboot);
	    execv(argv[0], argv);
	    syslog(LOG_ERR, _("gdm_child_action: Reboot failed: %s"), strerror(errno));
	    break;

	case DISPLAY_HALT:
	    syslog(LOG_INFO, _("gdm_child_action: Master halting..."));
	    g_slist_foreach(gdmDisplays, (GFunc) gdm_display_unmanage, NULL);
	    closelog();
	    unlink(gdmPidFile);
	    argv=gdm_arg_munch(gdmHalt);
	    execv(argv[0], argv);
	    syslog(LOG_ERR, _("gdm_child_action: Halt failed: %s"), strerror(errno));
	    break;

	default:
	    syslog(LOG_ERR, _("gdm_child_action: Slave process returned unknown status!"));
	    break;
	}
    }

    sigprocmask(SIG_SETMASK, &omask, NULL);
}


static void 
gdm_child_handler(gint sig)
{
    gint status;
    ChildStat *slave;

    slave=g_new0(ChildStat, 1);
    slave->pid=waitpid(-1, &status, WNOHANG);

    if(WIFEXITED(status))
	slave->status=WEXITSTATUS(status);

    if(slave->pid > 0) {
	gdm_debug(_("gdm_child_handler: child %d returned %d"), slave->pid, slave->status);
	g_slist_foreach(gdmDisplays, (GFunc) gdm_child_action, slave);
    };

    g_free(slave);
}


static void
gdm_display_unmanage(gdmDisplayType *d)
{
    gdm_debug(_("gdm_display_unmanage: Stopping %s"), d->name);

    if(d->type == DISPLAY_LOCAL) {

	if(d->slavepid) {
	    kill(d->slavepid, SIGINT);
	    waitpid(d->slavepid, 0, 0);
	    d->slavepid=0;
	}

	if(d->servpid) {
	    kill(d->servpid, SIGTERM);
	    waitpid(d->servpid, 0, 0);
	    d->servpid=0;
	}

	d->dispstat=DISPLAY_ABORT;
    }
    else {
	/* Kill remote X server */
    };
}


static void
gdm_term_handler(int sig)
{
    sigset_t mask;

    gdm_debug(_("gdm_term_handler: Got TERM/INT. Going down!"));

    sigemptyset(&mask);
    sigaddset(&mask, SIGCHLD);
    sigprocmask(SIG_BLOCK, &mask, NULL);

    g_slist_foreach(gdmDisplays, (GFunc) gdm_display_unmanage, NULL);

    closelog();
    unlink(gdmPidFile);
    exit(EXIT_SUCCESS);
}


static void
gdm_daemonify(void)
{
    FILE *pf;
    pid_t pid;

    if((pid=fork())) {
	if((pf=fopen(gdmPidFile, "w"))) {
	    fprintf(pf, "%d\n", pid);
	    fclose(pf);
	};
        exit(EXIT_SUCCESS);
    };

    if(pid<0) 
	gdm_abort(_("gdm_daemonify: fork() failed!"));

    if(setsid() < 0)
	gdm_abort(_("gdm_daemonify: setsid() failed: %s!"), strerror(errno));

    chdir(gdmAuthDir);
    umask(022);

    close(0);
    close(1);
    close(2);

    open ("/dev/null", O_RDONLY);
    dup2 (0, 1);
    dup2 (0, 2);
}


int 
main (int argc, char *argv[])
{
    sigset_t mask;
    struct sigaction term, child;
    FILE *pf;
 
    /* xdm compatible error message */
    if(getuid()) {
	fprintf(stderr, _("Only root wants to run gdm\n"));
	exit(EXIT_FAILURE);
    };

    umask (022);
    openlog("gdm", LOG_PID, LOG_DAEMON);

    gdm_config_parse();

    if(!access(gdmPidFile, R_OK)) {
        /* Check the process is still alive. Don't abort otherwise. */
        int pidv;

        pf=fopen(gdmPidFile, "r");
        if(pf && fscanf(pf, "%d", &pidv)==1 && kill(pidv,0)!=-1) {
        	fclose(pf);
		fprintf(stderr, _("gdm already running. Aborting!\n\n"));
		exit(EXIT_FAILURE);
	};

	fclose(pf);
	fprintf(stderr, _("gdm was already running (%d) but seems to have been murdered mysteriously.\n"), pidv);
	unlink(gdmPidFile);
    };

    /* Become daemon unless started in -nodaemon mode or child of init */
    if( (argc==2 && strcmp(argv[1],"-nodaemon")==0) || getppid()==1) {
	if((pf=fopen(gdmPidFile, "w"))) {
	    fprintf(pf, "%d\n", getpid());
	    fclose(pf);
	};
    }
    else
	gdm_daemonify();

    /* Signal handling */
    term.sa_handler = gdm_term_handler;
    term.sa_flags = SA_RESTART;
    sigemptyset(&term.sa_mask);

    if(sigaction(SIGTERM, &term, NULL) < 0) 
	gdm_abort(_("gdm_main: Error setting up TERM signal handler"));

    if(sigaction(SIGINT, &term, NULL) < 0) 
	gdm_abort(_("gdm_main: Error setting up INT signal handler"));

    child.sa_handler = gdm_child_handler;
    child.sa_flags = SA_RESTART|SA_NOCLDSTOP;
    sigemptyset(&child.sa_mask);
    sigaddset(&child.sa_mask, SIGCHLD);

    if(sigaction(SIGCHLD, &child, NULL) < 0) 
	gdm_abort(_("gdm_main: Error setting up CHLD signal handler"));

    sigemptyset(&mask);
    sigaddset(&mask, SIGINT);
    sigaddset(&mask, SIGTERM);
    sigaddset(&mask, SIGCHLD);
    sigprocmask(SIG_UNBLOCK, &mask, NULL);

    /* Start local X servers */
    gdm_debug(_("gdm_main: Here we go..."));
    g_slist_foreach(gdmDisplays, (GFunc) gdm_local_servers_start, NULL);

    /* XDMCP goes here */
    for(;;) {
	gdm_debug("Main loop");
	pause();
    };

    return (EXIT_SUCCESS);
}

/* EOF */
