# Copyright (C) 2004,2005 by SICEm S.L. and Imendio
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gobject
import gtk

from gazpacho import project
from gazpacho import util
from gazpacho.loader import tags
from gazpacho.placeholder import Placeholder
from gazpacho.property import Property
from gazpacho.popup import Popup
from gazpacho.widgetregistry import widget_registry
from gazpacho.l10n import _

def copy_widget(src):
    """Returns a copy of the GWidget src"""

    copy = Widget(src.klass, src.project, interactive=False,
                  call_custom_functions=False)

    if isinstance(src.widget, gtk.Container):
        # hack: this is needed because when setting self._widget=widget
        # in the construction of copy, some children are added when
        # setting some properties
        for child in copy.widget.get_children():
            copy.widget.remove(child)
            
        for child in src.widget.get_children():
            gwidget = Widget.get_from_gtk_widget(child)
            if gwidget is None:
                continue
            child_copy = copy_widget(gwidget)
            copy.widget.add(child_copy.widget)
            child_copy._set_packing_properties(copy,
                                               gwidget._packing_properties)
            
    # now copy the properties
    copy._apply_properties(src._properties)

    return copy


class Widget(gobject.GObject):
    """ 
    This is essentially a wrapper around regular GtkWidgets with more data and
    functionality needed for Gazpacho.
    
    It has access to the GtkWidget it is associated with, the Project it
    belongs to and the instance of its WidgetClass.
    
    Given a GtkWidget you can ask which (Gazpacho) Widget is associated with it
    using the get_from_gtk_widget method or, if no one has been created yet you
    can create one with the create_from_gtk_widget method.
    
    A Widget have a signal dictionary where the keys are signal names and the
    values are lists of signal handler names. There are methods to
    add/remove/edit these.
    
    It store all the Properies (wrappers for GtkWidgets properties) and you can
    get any of them with get_glade_property. It knows how to handle normal
    properties and packing properties.
    """
    __gproperties__ = {
        'name' :    (str,
                     'Name',
                     _('The name of the widget'),
                     '',
                     gobject.PARAM_READWRITE),
        'internal': (str,
                     'Internal name',
                     _('The internal name of the widget'),
                     None,
                     gobject.PARAM_READWRITE),
        'widget':   (gtk.Widget,
                     'Widget',
                     _('The gtk+ widget associated'),
                     gobject.PARAM_READWRITE),
        'klass':    (object,
                     'Class',
                     _('The class of the associated gtk+ widget'),
                     gobject.PARAM_READWRITE),
        'project':  (project.Project,
                     'Project',
                     _('The project that this widget belongs to'),
                     gobject.PARAM_READWRITE)                    
        }
    
    __gsignals__ = {
        'add_signal_handler' :    (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_signal_handler' : (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'change_signal_handler' : (gobject.SIGNAL_RUN_LAST, None,
                                   (object, object))
        }
    
    def __init__(self, klass, project, loading=False, internal_widget=None,
                  internal_name=None, interactive=True, call_custom_functions=True):
        # BIG FAT HACK (Has to be removed as soon as possible):
        # call_custom_functions: this was needed to implement
        # copy_widget. I need to rewrite GWidget initialization code
        # so it is much more flexible. This means deep thinking and
        # search for every code that create GWidgets

        gobject.GObject.__init__(self)

        if loading: return

        # If the widget is an internal child of another widget this is the
        # name of the  internal child, otherwise is None
        # Internal children cannot be deleted.
        self.internal = internal_name

        # A list of GladeProperty. A GladeProperty is an instance of a
        # GladePropertyClass. If a GladePropertyClass for a gtkbutton is label,
        # its property is "Ok"
        self._properties = []

        #  A list of GladeProperty. Note that these properties are related to
        # the container of the widget, thus they change after pasting the
        # widget to a different container. Toplevels widget do not have
        # packing properties. See also child_properties of GladeWidgetClass.
        self._packing_properties = []

        # A dictionary of signals (signal, handlers) indexed by its name
        self._signals = {}

        # A pointer to the widget that was created. and is shown as a "view" of
        # the GladeWidget. This widget is updated as the properties are
        # modified for the GladeWidget. We should not destroy this widget once
        # created, we should just hide them
        self._widget = None # we need an initial dummy value

        if internal_widget is not None:
            widget = internal_widget
        else:
            widget = klass.type()
            if klass.pre_create_function and call_custom_functions:
                klass.pre_create_function(widget, interactive)
        
        self.klass = klass
        self.project = project
        # The name of the widet. For example window1 or button2. This is a
        # unique name and is the one used when loading widget with libglade
        self.name = project.new_widget_name(klass.generic_name)

        self.widget = widget
        
        if internal_widget is not None: return
        
        if klass.post_create_function and call_custom_functions:
            klass.post_create_function(self._widget, interactive)

        if klass.fill_empty_function and call_custom_functions:
            klass.fill_empty_function(self._widget)

        # sync widget properties with our properties
        for p_class in self.klass.properties:
            if p_class.optional and not p_class.optional_default:
                continue
            
            prop = Property(p_class, self)
            try:
                prop.value = self.widget.get_property(p_class.id)
            except TypeError:
                pass
            for p in self._properties:
                if p.klass.id == p_class.id:
                    p.value = prop.value

    def do_get_property(self, prop):
        try:
            return getattr(self, 'get_' + prop.name)(self)
        except:
            raise AttributeError(_('Unknown property %s') % prop.name)

    def do_set_property(self, prop, value):
        try:
            getattr(self, 'set_' + prop.name)(self, value)
        except:
            raise AttributeError(_('Unknown property %s') % prop.name)

    # the properties
    def get_name(self): return self._name
    def set_name(self, value):
        self._name = value
        self.notify('name')
    name = property(get_name, set_name)

    def get_internal(self): return self._internal
    def set_internal(self, value): self._internal = value
    internal = property(get_internal, set_internal)

    def get_widget(self): return self._widget
    def set_widget(self, gtk_widget):
        if self._widget:
            self._widget.set_data('GladeWidgetDataTag', None)
            self._widget = None

        self._widget = gtk_widget
        self._widget.set_data('GladeWidgetDataTag', self)
        self._widget.add_events(gtk.gdk.BUTTON_PRESS_MASK \
                                | gtk.gdk.BUTTON_RELEASE_MASK \
                                | gtk.gdk.KEY_PRESS_MASK)

        if self._widget.flags() & gtk.TOPLEVEL:
            self._widget.connect('delete_event', self._hide_on_delete)

        self._widget.connect('popup_menu', self._popup_menu)
        self._widget.connect('key_press_event', self._key_press)

        self._connect_signal_handlers(self._widget)

        if self._internal is None:
            self._apply_properties()
        else:
            parent = self.get_parent()
            if parent:
                self._set_packing_properties(parent)
        
    widget = property(get_widget, set_widget)

    def get_klass(self): return self._klass
    def set_klass(self, value):
        self._klass = value
        for prop in self._klass.properties:
            p = Property(prop, self)
            self._properties.append(p)
        
    klass = property(get_klass, set_klass)

    # utility getter methods
    def get_parent(self):
        if self._widget.flags() & gtk.TOPLEVEL:
            return None

        return util.get_parent(self._widget)

    def get_from_gtk_widget(gtk_widget):
        return gtk_widget.get_data('GladeWidgetDataTag')

    get_from_gtk_widget = staticmethod(get_from_gtk_widget)

    def create_from_gtk_widget(gtk_widget, project=None, real_widgets=[],
                               signals=[]):
        """  This class method create a GWidget from a GtkWidget. It
        recursively creates any children of the original gtk_widget. Then
        it adds the widget to the project argument.
        This method is mainly used when loading a .glade file with
        gazpacho.loader
        It only handles the widgets whose names are in the
        real_widgets list parameter.
        The signal list parameter is used to set the widget's signals"""

        internal_child = None
        if gtk_widget.name not in real_widgets:
            # check if this is an internal widget
            internal_child = gtk_widget.get_data('internal_child')
            if internal_child is None:
                # this is not an important widget but maybe its children are
                if isinstance(gtk_widget, gtk.Container):
                    for child in gtk_widget.get_children():
                        Widget.create_from_gtk_widget(child, project,
                                                      real_widgets, signals)
                return None

        klass = widget_registry.get_by_name(gobject.type_name(gtk_widget))
        if klass is None:
            print _('Warning: could not get the class from widget name: %s') %\
                  gtk_widget
            return
        
        widget = Widget(klass, project, loading=True)

        widget._internal = internal_child
        widget._properties = []
        widget._packing_properties = []
        widget._signals = {}

        widget.project = project

        # set the class
        widget._klass = klass
        for prop in klass.properties:
            p = Property(prop, widget)
            try:
                p.value = gtk_widget.get_property(prop.id)
            except TypeError:
                pass

            # Get i18n metadata for translatable string properties
            if prop.is_translatable and prop.type == gobject.TYPE_STRING:
                comment = util.gtk_widget_get_metadata(gtk_widget,
                                                       util.METADATA_I18N_COMMENT,
                                                       prop.id)
                if comment is not None:
                    p.i18n_comment = comment

                p.is_translatable = util.gtk_widget_get_metadata(gtk_widget,
                                                                 util.METADATA_I18N_IS_TRANSLATABLE,
                                                                 prop.id)

                p.has_i18n_context = util.gtk_widget_get_metadata(gtk_widget,
                                                                  util.METADATA_I18N_HAS_CONTEXT,
                                                                  prop.id)


            widget._properties.append(p)

        widget.name = gtk_widget.name

        # set the widget
        widget._widget = gtk_widget
        widget._widget.set_data('GladeWidgetDataTag', widget)
        widget._widget.add_events(gtk.gdk.BUTTON_PRESS_MASK \
                                  | gtk.gdk.BUTTON_RELEASE_MASK \
                                  | gtk.gdk.KEY_PRESS_MASK)

        if widget._widget.flags() & gtk.TOPLEVEL:
            widget._widget.connect('delete_event', widget._hide_on_delete)

        widget._widget.connect('popup_menu', widget._popup_menu)
        widget._widget.connect('key_press_event', widget._key_press)

        widget._connect_signal_handlers(widget._widget)

        parent = widget.get_parent()
        if parent:
            widget._set_packing_properties(parent)

        # create the children
        if isinstance(gtk_widget, gtk.Container) \
           and not isinstance(gtk_widget, (gtk.MenuBar, gtk.Toolbar)):
            # for menubars and toolbars we don handle their children here
            for child in gtk_widget.get_children():
                Widget.create_from_gtk_widget(child, project, real_widgets,
                                              signals)

        # set the signals
        widget_signals = [s[1:] for s in signals if s[0] == gtk_widget]
        for signal_name, signal_handler, after in widget_signals:
            handler = {'name': signal_name, 'handler': signal_handler,
                       'after': after}
            widget.add_signal_handler(handler)

        # if the widget is a toplevel we need to attach the accel groups
        # of the application
        if widget.is_toplevel():
            for ag in project._app.get_accel_groups():
                widget._widget.add_accel_group(ag)
        
        return widget

    create_from_gtk_widget = staticmethod(create_from_gtk_widget)
    
    def add_signal_handler(self, signal_handler):
        self.emit('add_signal_handler', signal_handler)
        
    def do_add_signal_handler(self, signal_handler):
        signals = self.list_signal_handlers(signal_handler['name'])
        if not signals:
            self._signals[signal_handler['name']] = []

        self._signals[signal_handler['name']].append(signal_handler)

    def remove_signal_handler(self, signal_handler):
        self.emit('remove_signal_handler', signal_handler)
        
    def do_remove_signal_handler(self, signal_handler):
        signals = self.list_signal_handlers(signal_handler.name)
        # trying to remove an inexistent signal?
        assert signals != []

        self._signals[signal_handler['name']].remove(signal_handler)
        
    def change_signal_handler(self, old_signal_handler, new_signal_handler):
        self.emit('change_signal_handler',
                   old_signal_handler, new_signal_handler)
        
    def do_change_signal_handler(self, old_signal_handler,
                                  new_signal_handler):
        if old_signal_handler['name'] != new_signal_handler['name']:
            return

        signals = self.list_signal_handlers(old_signal_handler['name'])
        # trying to remove an inexistent signal?
        assert signals != []

        index = signals.index(old_signal_handler)
        signals[index]['handler'] = new_signal_handler['handler']
        signals[index]['after'] = new_signal_handler['after']

    def _expose_event(self, widget, event):
        util.queue_draw_nodes(event.window)

    def _find_inside_container(self, widget, data):
        window = data['toplevel']
        coords = window.translate_coordinates(widget, data['x'], data['y'])
        if len(coords) == 2:
            x, y = coords
        else:
            data['found'] = None
            return
        
        if (0 <= x < widget.allocation.width and
            0 <= y < widget.allocation.height and
            Widget.get_from_gtk_widget(widget)):
            data['found'] = widget
            
    def _find_deepest_child_at_position(self, toplevel, container,
                                        top_x, top_y):
        data = {'x': top_x, 'y': top_y, 'toplevel': toplevel, 'found': None}
        container.forall(self._find_inside_container, data)
        if data['found'] and isinstance(data['found'], gtk.Container):
            return self._find_deepest_child_at_position(toplevel,
                                                        data['found'],
                                                        top_x, top_y)
        elif data['found']:
            return Widget.get_from_gtk_widget(data['found'])
        else:
            return Widget.get_from_gtk_widget(container)
    
    def _retrieve_from_position(self, base, x, y):
        """ Returns the real GtkWidget in x, y of base.
        This is needed because for GtkWidgets than does not have a
        GdkWindow the click event goes right to their parent.
        """
        toplevel_widget = base.get_toplevel()
        if not (toplevel_widget.flags() & gtk.TOPLEVEL):
            return None
        top_x, top_y = base.translate_coordinates(toplevel_widget, x, y)
        return self._find_deepest_child_at_position(toplevel_widget,
                                                    toplevel_widget,
                                                    top_x, top_y)
        
    def _button_press_event(self, widget, event):
        glade_widget = self._retrieve_from_position(widget,
                                                    int(event.x),
                                                    int(event.y))
        widget = glade_widget.widget
        # Make sure to grab focus, since we may stop default handlers
        if widget.flags() & gtk.CAN_FOCUS:
            widget.grab_focus()

        if event.button == 1:
            # if it's already selected don't stop default handlers,
            # e.g toggle button
            if util.has_nodes(widget):
                return False

            self.project.selection_set(widget, True)
            return True
        elif event.button == 3:
            app = self.project._app
            popup = Popup(app, glade_widget)
            popup.pop(event)
            return True

        return False
    
    def _event(self, widget, event):
        if event.type == gtk.gdk.EXPOSE:
            return False # XXX TODO
        return False
    
    def _popup_menu(self, widget):
        return True # XXX TODO

    def _key_press(self, widget, event):
        gwidget = Widget.get_from_gtk_widget(widget)

        if event.keyval == gtk.keysyms.Delete:
            # We will delete all the selected items
            gwidget.project.delete_selection()
            return True
        
        return False

    def _hide_on_delete(self, widget, event):
        return widget.hide_on_delete()
    
    def _connect_signal_handlers(self, gtk_widget):
        # don't connect handlers for placeholders
        if isinstance(gtk_widget, Placeholder):
            return

        # check if we've already connected an event handler
        if not gtk_widget.get_data(tags.EVENT_HANDLER_CONNECTED):
            gtk_widget.connect("expose-event", self._expose_event)
            gtk_widget.connect("button-press-event", self._button_press_event)
            gtk_widget.set_data(tags.EVENT_HANDLER_CONNECTED, 1)

        # we also need to get expose events for any children
        if isinstance(gtk_widget, gtk.Container):
            gtk_widget.forall(self._connect_signal_handlers)

    def replace(self, old_widget, new_widget, parent):
        # XXX Document this! This should be a function and not a method
        gnew_widget = Widget.get_from_gtk_widget(new_widget)
        gold_widget = old_widget and Widget.get_from_gtk_widget(old_widget)

        real_new_widget = gnew_widget and gnew_widget.widget or new_widget
        real_old_widget = gold_widget and gold_widget.widget or old_widget

        if parent is None:
            parent = util.get_parent(old_widget)

        if parent.klass.replace_child:
            parent.klass.replace_child(real_old_widget, real_new_widget,
                                        parent.widget)
        else:
            print _("Could not replace a placeholder because a replace function has not been implemented for '%s'") % parent.klass.name
            return

        if gnew_widget:
            gnew_widget._set_packing_properties(parent)

    def _apply_properties(self, properties=[]):
        self.widget.freeze_notify()

        props = properties or self._properties
        
        for i, prop in enumerate(props):
            # XXX TODO we don't support yet this properties
            if prop.klass.type in [gobject.TYPE_ENUM, gobject.TYPE_FLAGS,
                                   gobject.TYPE_OBJECT]:
                continue
            if prop.klass._set_function:
                self._properties[i].value = prop._value
                prop.klass._set_function(self.widget, prop._value)
            else:
                self.widget.set_property(prop.klass.id, prop.value)

        self.widget.thaw_notify()

    def _create_packing_properties(self, container):
        new_list = []
        for property_class in container.klass.child_properties:
            if not container.klass.child_property_applies(container.widget,
                                                          self.widget,
                                                          property_class.id):
                continue
            prop = Property(property_class, self)
            new_list.append(prop)

        parent = container.get_parent()
        if parent:
            ancestor_list = self._create_packing_properties(parent)
            new_list += ancestor_list

        return new_list

    def _set_default_packing_properties(self, container):
        for property_class in container.klass.child_properties:
            if not container.klass.child_property_applies(container.widget,
                                                          self.widget,
                                                          property_class.id):
                continue

            v = self.klass.get_packing_default(container.klass,
                                               property_class.id)
            if not v:
                continue

            value = util.get_property_value_from_string (property_class, v)
            container.widget.child_set_property(self.widget, property_class.id,
                                                value)

    def _set_packing_properties(self, container, properties=[]):
        self._set_default_packing_properties(container)
        self._packing_properties = self._create_packing_properties(container)
        props = properties or self._packing_properties
        # update the values of the properties to the ones we get from gtk
        for prop in props:
            prop.value = container.widget.child_get_property(self.widget,
                                                             prop.klass.id)
            
    def get_glade_property(self, prop_id):
        # XXX Need to changed all 'glade' names for 'gazpacho'
        l = [p for p in self._properties if p.klass.id == prop_id]
        if l: return l[0]

        l = [p for p in self._packing_properties if p.klass.id == prop_id]
        if l: return l[0]

        print _('Could not get property %s for widget %s') % \
              (prop_id, self._name)
        return None

    def is_toplevel(self):
        return self.klass.is_toplevel()

    def _get_glade_widgets(self, gtk_widget):
        """ Return the first child (and its brothers) of gtk_widget which is a
        GladeWidget """
        if not isinstance(gtk_widget, gtk.Container):
            return []

        result = []
        for child in gtk_widget.get_children():
            gchild = Widget.get_from_gtk_widget(child)
            if gchild is not None:
                result.append(child)
            elif isinstance(child, gtk.Container):
                result += self._get_glade_widgets(child)

        return result
    
    def write(self, document):
        node = document.createElement(tags.XML_TAG_WIDGET)
        node.setAttribute(tags.XML_TAG_CLASS, self.klass.name)
        node.setAttribute(tags.XML_TAG_ID, self.name)

        gtk_props = [pspec.name for pspec in
                                    gobject.list_properties(self.widget)]

        visible_node = None
        # write the properties
        for prop in self._properties:
            # we don't store glade specific properties because libglade gets
            # confused with them
            if prop.klass.id in gtk_props:
                if not prop.klass.packing:
                    child = prop.write(document)
                    # the visible property of GtkWidget must be the last one
                    # because otherwise props like default-width or
                    # default-height doesn't work when loading the widget
                    if child is not None:
                        if prop.klass.id == 'visible':
                            visible_node = child
                        else:
                            node.appendChild(child)
        if visible_node is not None:
            node.appendChild(visible_node)

        # Signals
        for signal_name, handlers in self._signals.items():
            for handler in handlers:
                child = document.createElement(tags.XML_TAG_SIGNAL)
                child.setAttribute(tags.XML_TAG_NAME, handler['name'])
                child.setAttribute(tags.XML_TAG_HANDLER, handler['handler'])
                child.setAttribute(tags.XML_TAG_AFTER,
                                   (tags.TRUE, tags.FALSE)[handler['after']])
                node.appendChild(child)
                
        # Children
        if isinstance(self.widget, gtk.Container):
            for child_widget in self.widget.get_children():
                gwidget = Widget.get_from_gtk_widget(child_widget)
                if isinstance(child_widget,Placeholder) or gwidget is not None:
                    child = self._write_child(child_widget, document)
                    if child is not None:
                        node.appendChild(child)
                else:
                    # see if this child has GladeWidgets by its own
                    children = self._get_glade_widgets(child_widget)
                    for c in children:
                        cn = self._write_child(c, document)
                        if cn:
                            node.appendChild(cn)
                    
        return node

    def _write_child(self, gtk_widget, document):
        child_tag = document.createElement(tags.XML_TAG_CHILD)
        if isinstance(gtk_widget, Placeholder):
            child = document.createElement(tags.XML_TAG_PLACEHOLDER)
            child_tag.appendChild(child)
            return child_tag

        child_widget = Widget.get_from_gtk_widget(gtk_widget)
        if child_widget is None:
            return None

        if child_widget.internal is not None:
            child_tag.setAttribute(tags.XML_TAG_INTERNAL_CHILD,
                                   child_widget.internal)

        child = child_widget.write(document)
        child_tag.appendChild(child)

        # Append the packing properties
        if child_widget._packing_properties:
            packing = document.createElement(tags.XML_TAG_PACKING)
            child_tag.appendChild(packing)
            for prop in child_widget._packing_properties:
                packing_property = prop.write(document)
                if packing_property is not None:
                    packing.appendChild(packing_property)

        return child_tag

    def list_signal_handlers(self, signal_name):
        result = []
        try:
            result = self._signals[signal_name]
        except KeyError:
            pass

        return result

gobject.type_register(Widget)
