# Copyright (C) 2004,2005 by SICEm S.L. and Imendio
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import gobject

from gazpacho.loader import tags
from gazpacho.l10n import _

# UNICHAR and UINT are internally the same thing but we need to
# handle them as separate. Luckily there is only a Unichar prop in
# Gtk+, which is 'insible_char', so that's why we put it here.
# See http://www.sicem.biz/pipermail/gazpacho/2005-January/000097.html
UNICHAR_PROPERTIES = ('invisible-char', )

class PropertyException(Exception): pass

class PropertyClass(object):
    supported_types = (gobject.TYPE_BOOLEAN, gobject.TYPE_FLOAT,
                       gobject.TYPE_INT, gobject.TYPE_UINT,
                       gobject.TYPE_DOUBLE, gobject.TYPE_STRING,
                       gobject.TYPE_ENUM, gobject.TYPE_FLAGS,
                       gobject.TYPE_OBJECT)
    
    def __init__(self, spec=None):
        from gazpacho import choice
        self._set_default_values()

        if not spec: return

        self._spec = spec
        
        # The id of the property. Like "label" or "xpad" this is
        # a non-translatable string
        self.id = spec.name

        # The type of property from GladePropertyType
        self._type = self._get_type_from_spec(spec)

        if self._type is None:
            raise PropertyException(_('Type Error'))
            
        # The name of the property. Like "Label" or "X Pad" this is
        # a translatable string
        self._name = spec.nick
        if not self.id or not self._name:
            raise PropertyException(_('Failed to create property class from spec'))

        # The tooltip. Currently unimplemented. Not sure if it should go here
        self._tooltip = spec.blurb

        # The default value for this property
        self._default = self._get_default_from_spec(spec)

        if self._type == gobject.TYPE_ENUM:
            self._choices = choice.list_new_from_spec(spec)
            self._enum_type = spec.value_type
        elif self._type == gobject.TYPE_FLAGS:
            self._enum_type = spec.value_type
        elif self._type == gobject.TYPE_DOUBLE:
            self._parameters = [] # XXX self._get_parameters_numeric(spec)
        elif self._type == gobject.TYPE_OBJECT:
            raise PropertyException(_('Type not suported'))

    def _set_default_values(self):
        # initial values
        self._type = None
        self.id = self._name = self._tooltip = self._default = None
        self._spec = None
        self._disabled = False

        # list of GladeParameter objects. This list provides with an extra
        # set of key-value pairs to specify aspects of this property. Like the
        # number of "VisibleLines" of a text property. Or the range and default
        # values of a spin button entry. Also the default choice for
        # a type == CHOICE
        self._parameters = []

        # list of GladeChoice items. This is only used for propeties of type
        # GLADE_PROPERTY_TYPE_CHOICE and is [] for other poperties.
        self._choices = []

        #  If it is TYPE_ENUM or TYPE_FLAGS, this holds the GType of the enum
        # or flags, otherwise it's 0.
        self._enum_type = 0
        
        # Some properties are optional by nature like default width. It can be
        # set or not set. A default property has a check box in the left that
        # enables/disables de input
        self._optional = False

        # For optional values, what the default is
        self._optional_default = True

        self._visibility_function = None
        
        # A  GladeWidgetClass pointer of objects that we need to set for this
        # widget for example : GtkSpinButton has a Adjustment inside a GtkCombo
        # has an entry inside and a GtkClist which makes a drop dowm menu. This
        # is only valid when the type is object.
        self._child = False
        
        # Common properties go in the common tab
        self._common = False

        # Packing properties go in the packing tab
        self._packing = False

        # If true, this property_class has been "modified" from the the
        # standard property by a xml file
        self._is_modified = False

        # If this property can't be set with g_object_set then we need to
        # implement it inside glade. This is a pointer to the function that
        # can set this property. The functions to work arround this problems
        # are inside gladegtk.py
        self._set_function = None

        # If this property can't be get with g_object_get then we need to
        # implement it inside glade. This is a pointer to the function that
        # can get this property. The functions to work arround this problems
        # are inside gladegtk.py
        self._get_function = None

        # Some properties requires a special widget for editing the property.
        # This function creates the widget.
        self._editor_create_function = None
        self._editor_load_function = None

        # If true, additional i18n metadata can be edited for string
        # properties
        self._is_translatable = True

    # property accesors
    
    def get_type(self): return self._type
    def set_type(self, value): self._type = value
    type = property(get_type, set_type)

    def get_spec(self): return self._spec
    spec = property(get_spec)
    
    def get_default(self): return self._default
    def set_default(self, value): self._default = value
    default = property(get_default, set_default)

    def get_optional(self): return self._optional
    def set_optional(self, value): self._optional = value
    optional = property(get_optional, set_optional)
    
    def get_optional_default(self): return self._optional_default
    def set_optional_default(self, value): self._optional_default = value
    optional_default = property(get_optional_default, set_optional_default)

    def get_child(self): return self._child
    def set_child(self, value): self._child = value
    child = property(get_child, set_child)

    def get_is_modified(self): return self._is_modified
    is_modified = property(get_is_modified)

    def get_common(self): return self._common
    def set_common(self, value): self._common = value
    common = property(get_common, set_common)

    def get_name(self): return self._name
    def set_name(self, value): self._name = value
    name = property(get_name, set_name)

    def get_tooltip(self): return self._tooltip
    def set_tooltip(self, value): self._tooltip = value
    tooltip = property(get_tooltip, set_tooltip)

    def get_disabled(self): return self._disabled
    disabled = property(get_disabled)
    
    def is_visible(self, widget_class):
        if self._visibility_function:
            return self._visibility_function(widget_class)
        return True

    def get_parameters(self): return self._parameters
    parameters = property(get_parameters)

    def get_choices(self): return self._choices
    choices = property(get_choices)

    def get_packing(self): return self._packing
    def set_packing(self, value): self._packing = value
    packing = property(get_packing, set_packing)

    def get_is_translatable(self): return self._is_translatable
    def set_is_translatable(self, value): self._is_translatable = value
    is_translatable = property(get_is_translatable, set_is_translatable)
    
    def _get_type_from_spec(self, spec):
        if spec.name == 'name':
            result = None
        elif spec.name == 'user-data':
            result = None
        elif hasattr(spec, 'flags_class'):
            result = gobject.TYPE_FLAGS
        elif hasattr(spec, 'enum_class'):
            result = gobject.TYPE_ENUM
        elif spec.value_type in PropertyClass.supported_types:
            result = spec.value_type
        else:
            result = None

        return result

    def _get_default_from_spec(self, spec):
        if self._type in (gobject.TYPE_ENUM, gobject.TYPE_FLAGS,
                          gobject.TYPE_STRING, gobject.TYPE_INT,
                          gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE,
                          gobject.TYPE_BOOLEAN, gobject.TYPE_UINT):
            return spec.default_value
        else:
            return None
        
    def _get_parameters_numeric(self):
        return [] # XXX TODO

    def _str_to_enum(self, str):
        type_map = {
            tags.STRING: gobject.TYPE_STRING,
            tags.BOOLEAN: gobject.TYPE_BOOLEAN,
            tags.UNICHAR: gobject.TYPE_UINT,
            tags.UINTEGER: gobject.TYPE_UINT,
            tags.FLOAT: gobject.TYPE_FLOAT,
            tags.INTEGER: gobject.TYPE_INT,
            tags.ENUM: gobject.TYPE_ENUM,
            tags.FLAGS: gobject.TYPE_FLAGS,
            tags.OBJECT: gobject.TYPE_OBJECT
            }
        return type_map.get(str)
    
    def update_from_node(self, xmlnode, widget_class):
        from gazpacho import util, parameter, choice

        # check the id
        prop_id = xmlnode.getAttribute(tags.ID)
        if not prop_id:
            return False

        if util.xml_get_property_boolean(xmlnode, tags.DISABLED, False):
            self._disabled = True
            return True

        # if needed, update the name
        name = xmlnode.getAttribute(tags.NAME)
        if name:
            self._name = name

        # ...the type...
        child = xmlnode.getElementsByTagName(tags.TYPE)
        if child:
            oldtype = util.xml_get_text_from_node(child[0])
            newtype = self._str_to_enum(oldtype)
            if newtype is None:
                return False
            self._type = newtype

        # ...and the tooltip
        child = xmlnode.getElementsByTagName(tags.TOOLTIP)
        if child:
            tooltip = util.xml_get_text_from_node(child[0])
            if tooltip:
                self._tooltip = tooltip

        # Get the visibility function
        child = xmlnode.getElementsByTagName(tags.VISIBILITY_FUNCTION)
        if child:
            try:
                name = util.xml_get_text_from_node(child[0])
                self._visibility_function = getattr(widget_class.module, name)
            except:
                print _('Could not find visibility function %s') % name

        # Get the parameters
        child = xmlnode.getElementsByTagName(tags.PARAMETERS)
        if child:
            self._parameters = parameter.list_new_from_node(child[0])
            self._optional = parameter.get_boolean(self._parameters,
                                                   tags.OPTIONAL, False)

        # Get the choices
        if self._type == gobject.TYPE_ENUM:
            child = xmlnode.getElementsByTagName(tags.ENUMS)
            if child:
                type_name = child[0].getAttribute(tags.ENUM_TYPE)
                prop_type = gobject.type_from_name(type_name)
                self._enum_type = prop_type
                pspecs = [pspec for pspec in \
                          gobject.list_properties(widget_class.type) if \
                          pspec.value_type == prop_type]
                if len(pspecs) != 1:
                    print _('Could not find the property spec for enum type %s') % \
                          type_name
                self._choices = choice.list_new_from_node(child[0], pspecs[0])

        # If the property is an object load it
        if self._type == gobject.TYPE_OBJECT:
            child = xmlnode.getElementsByTagName(tags.GLADE_WIDGET_CLASS)
            if child:
                pass # XXX TODO

        # Get the default
        default = xmlnode.getAttribute(tags.DEFAULT)
        if default:
            if self._type == gobject.TYPE_INT:
                self._default = int(default)
            elif self._type in (gobject.TYPE_FLOAT, gobject.TYPE_DOUBLE):
                self._default = float(default)
            else:
                self._default = default

        # common, optional, etc
        self._common = util.xml_get_property_boolean(xmlnode, tags.COMMON,
                                                     False)
        self._optional = util.xml_get_property_boolean(xmlnode, tags.OPTIONAL,
                                                       False)
        if self._optional:
            self._optional_default = util.xml_get_property_boolean(xmlnode,
                                                                   tags.OPTIONAL_DEFAULT,
                                                                   False)

        self._is_translatable = util.xml_get_property_boolean(xmlnode,
                                                              tags.TRANSLATABLE,
                                                              True)
        
        child = xmlnode.getElementsByTagName(tags.SET_FUNCTION)
        if child:
            try:
                name = util.xml_get_text_from_node(child[0])
                self._set_function = getattr(widget_class.module, name)
            except:
                print _("Unable to get the 'set' function [%s] of the property [%s] of the widget's class [%s] from the module [%s]") % \
                      (name, self._name,
                       widget_class.name,
                       widget_class.module)
        
        child = xmlnode.getElementsByTagName(tags.GET_FUNCTION)
        if child:
            try:
                name = util.xml_get_text_from_node(child[0])
                self._get_function = getattr(widget_class.module, name)
            except:
                print _("Unable to get the 'get' function [%s] of the property [%s] of the widget's class [%s] from the module [%s]")  % \
                      (name,
                       self._name,
                       widget_class.name,
                       widget_class.module)
        

        editor = xmlnode.getElementsByTagName(tags.EDITOR)
        if editor:
            child = editor[0].getElementsByTagName(tags.CREATE_FUNCTION)
            if child:
                try:
                    name = util.xml_get_text_from_node(child[0])
                    self._editor_create_function = getattr(widget_class.module, name)
                except:
                    print _("Unable to get the 'editor-create' function [%s] of the property [%s] of the widget's class [%s] from the module [%s]") % \
                          (name,
                           self._name,
                           widget_class.name,
                           widget_class.module)
            child = editor[0].getElementsByTagName(tags.LOAD_FUNCTION)
            if child:
                try:
                    name = util.xml_get_text_from_node(child[0])
                    self._editor_load_function = getattr(widget_class.module, name)
                except:
                    print _("Unable to get the 'editor load function' function [%s] of the property [%s] of the widget's class [%s] from the module [%s]") % \
                          (name,
                           self._name,
                           widget_class.name,
                           widget_class.module)
                        
        # notify that we changed the property class
        self._is_modified = True
        
        return True

    def __str__(self):
        return "%s" % self.id

    def __repr__(self):
        return "%s" % self.id
    
def type2str(type):
    if type is not None:
        return gobject.type_name(type)
    else:
        return _('Error')
