# Copyright (C) 2004,2005 by SICEm S.L.
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU Lesser General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
import xml.dom.minidom
from xml.sax.saxutils import escape

import gtk
import gobject

from gazpacho import util
from gazpacho.gaction import GAction, get_gaction_group_from_gtk_action_group
from gazpacho.gaction import get_gaction_from_gtk_action
from gazpacho.loader import tags
from gazpacho.l10n import _

from os.path import basename, splitext

class Project(gobject.GObject):

    __gsignals__ = {
        'add_widget':          (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_widget':       (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'widget_name_changed': (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'selection_changed':   (gobject.SIGNAL_RUN_LAST, None, ()),
        'add_action':          (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'remove_action':       (gobject.SIGNAL_RUN_LAST, None, (object,)),
        'action_name_changed': (gobject.SIGNAL_RUN_LAST, None, (object,)),
        }
    
    project_counter = 1
    
    def __init__(self, untitled, app):
        gobject.GObject.__init__(self)

        self._app = app
        
        if untitled:
            # The name of the project like network-conf
            self.name = _('Untitled %d') % Project.project_counter
            Project.project_counter += 1

        # The full path of the xml file for this project
        self.path = None

        # The menu entry in the /Project menu
        self.entry = None
        
        # A list of GtkWidgets that make up this project. The widgets are
        # stored in no particular order.
        self.widgets = []

        # We need to keep the selection in the project because we have multiple
        # projects and when the user switchs between them, he will probably
        # not want to loose the selection. This is a list of GtkWidget items.
        self.selection = []

        #  A stack with the last executed commands
        self.undo_stack = []

        #  Points to the item previous to the redo items
        self.prev_redo_item = -1

        # widget -> old name of the widget
        self._widget_old_names = {}
        self.tooltips = gtk.Tooltips()

        # A flag that is set when a project has changes if this flag is not set
        # we don't have to query for confirmation after a close or exit is
        # requested
        self.changed = False

        # There is a UIManager in each project which holds the information
        # about menus and toolbars
        self.uimanager = None

        # This is the id for the menuitem entry on Gazpacho main menu for this
        # project
        self.uim_id = -1
        
        # list of GActionGroups
        self.action_groups = []

        # If true save the UI definition of the UIManager inside the
        # Glade file (in a CDATA section). Otherwise, save it in a
        # separate .ui file
        self.save_uimanager_in_glade_file = True

        # when saving menubar/toolbar separator they need to have a
        # unique name or otherwise UIManager won't render them
        self._separator_counter = 1
        
    def selection_changed(self):
        self.emit('selection_changed')

    def _on_widget_notify(self, widget, arg):
        if arg.name == 'name':
            old_name = self._widget_old_names[widget]
            self.widget_name_changed(widget, old_name)
            self._widget_old_names[widget] = widget.name
        elif arg.name == 'project':
            self.remove_widget(widget.widget)

    def add_widget(self, widget):
        from gazpacho.widget import Widget
        from gazpacho.placeholder import Placeholder
        # we don't list placeholders
        if isinstance(widget, Placeholder):
            return

        # If it's a container, add the children as well
        if isinstance(widget, gtk.Container):
            for child in widget.get_children():
                self.add_widget(child)

        gwidget = Widget.get_from_gtk_widget(widget)
        self.changed = True
        # The internal widgets (e.g. the label of a GtkButton) are handled
        # by gtk and don't have an associated GladeWidget: we don't want to
        # add these to our list. It would be nicer to have a flag to check
        # (as we do for placeholders) instead of checking for the associated
        # GladeWidget, so that we can assert that if a widget is _not_
        # internal, it _must_ have a corresponding GladeWidget...
        # Anyway this suffice for now.
        if not gwidget:
            return

        gwidget.project = self
        self._widget_old_names[gwidget] = gwidget.name

        gwidget.connect('notify', self._on_widget_notify)
        self.widgets.append(widget)

        self.emit('add_widget', gwidget)

    def remove_widget(self, widget):
        from gazpacho.placeholder import Placeholder
        from gazpacho.widget import Widget
        if isinstance(widget, Placeholder):
            return

        if isinstance(widget, gtk.Container):
            for child in widget.get_children():
                self.remove_widget(child)

        gwidget = Widget.get_from_gtk_widget(widget)
        if not gwidget:
            return

        if widget in self.selection:
            self.selection_remove(widget, False)
            self.selection_changed()

        self.release_widget_name(gwidget, gwidget.name)
        self.widgets.remove(widget)

        self.changed = True
        self.emit('remove_widget', gwidget)

    def add_action_group(self, action_group):
        self.action_groups.append(action_group)
        id1 = action_group.connect('add-action', self._add_action_cb)
        id2 = action_group.connect('remove-action', self._remove_action_cb)
        action_group.set_data('add_action_id', id1)
        action_group.set_data('remove_action_id', id2)

        if self.uimanager is None:
            self.uimanager = gtk.UIManager()

        action_group.create_gtk_action_group(self.uimanager)
        
        self.emit('add_action', action_group)
        self.changed = True
    
    def remove_action_group(self, action_group):
        self.action_groups.remove(action_group)
        id1 = action_group.get_data('add_action_id')
        id2 = action_group.get_data('remove_action_id')
        action_group.disconnect(id1)
        action_group.disconnect(id2)

        action_group.destroy_gtk_action_group()
        
        self.changed = True
        self.emit('remove_action', action_group)
        
    def _add_action_cb(self, action_group, action):
        self.changed = True
        self.emit('add_action', action)
    
    def _remove_action_cb(self, action_group, action):
        self.changed = True
        self.emit('remove_action', action)
    
    def change_action_name(self, action):
        self.changed = True
        self.emit('action_name_changed', action)
         
    def release_widget_name(self, gwidget, name):
        pass # XXX TODO

    def widget_name_changed(self, gwidget, old_name):
        self.release_widget_name(gwidget, old_name)
        self.emit('widget_name_changed', gwidget)
        self.changed = True
    
    def selection_clear(self, emit_signal):
        if not self.selection:
            return
        
        for widget in self.selection:
            util.remove_nodes(widget)
            
        self.selection = []
        
        if emit_signal:
            self.selection_changed()

    def selection_remove(self, widget, emit_signal):
        """ Remove the widget from the current selection """
        if not util.has_nodes(widget):
            return

        util.remove_nodes(widget)

        self.selection.remove(widget)
        if emit_signal:
            self.selection_changed()

    def selection_add(self, widget, emit_signal):
        if util.has_nodes(widget):
            return

        util.add_nodes(widget)

        self.selection.insert(0, widget)
        if emit_signal:
            self.selection_changed()
        
    def selection_set(self, widget, emit_signal):
        if util.has_nodes(widget):
            return

        self.selection_clear(False)
        self.selection_add(widget, emit_signal)

    def delete_selection(self):
        """ Delete (remove from the project) the widgets in the selection """
        from gazpacho.widget import Widget
        from gazpacho.placeholder import Placeholder

        newlist = list(self.selection)
        for widget in newlist:
            gwidget = Widget.get_from_gtk_widget(widget)
            if gwidget is not None:
                self._app.command_manager.delete(gwidget)
            elif isinstance(widget, Placeholder):
                self._app.command_manager.delete_placeholder(widget)

    def new_widget_name(self, base_name):
        i = 1
        while True:
            name = '%s%d' % (base_name, i)
            if self.get_widget_by_name(name) == None:
                return name
            i += 1

    def get_widget_by_name(self, name):
        from gazpacho.widget import Widget
        for w in self.widgets:
            gw = Widget.get_from_gtk_widget(w)
            if gw.name == name:
                return gw

    def get_separator_counter(self):
        self._separator_counter += 1
        return self._separator_counter
    
    separator_counter = property(get_separator_counter)
    
    def _write_uimanager(self, xml_doc):
        uim_node = xml_doc.createElement(tags.XML_TAG_OBJECT)
        uim_node.setAttribute(tags.XML_TAG_CLASS, "GtkUIManager")
        uim_node.setAttribute(tags.XML_TAG_ID, "uimanager")

        if self.save_uimanager_in_glade_file:
            prop_node = xml_doc.createElement(tags.XML_TAG_PROPERTY)
            prop_node.setAttribute(tags.XML_TAG_NAME, 'ui')
            ui_string = self.uimanager.get_ui()
            data = '<![CDATA[%s]]>' % ui_string
            cdata_node = xml_doc.createTextNode(data)
            prop_node.appendChild(cdata_node)
        else:
            ui_path = util.get_ui_path(path)
            prop_node = util.xml_create_string_prop_node(xml_doc, 'uifile',
                                                         ui_path)
        
        uim_node.appendChild(prop_node)
        for action_group in self.action_groups:
            child_node = xml_doc.createElement(tags.XML_TAG_CHILD)
            uim_node.appendChild(child_node)
            ag_node = action_group.write(xml_doc)
            child_node.appendChild(ag_node)

        return uim_node
            
    def write(self, xml_doc, path):
        from gazpacho.widget import Widget
        """ Returns the root node of a newly created xml representation of the
        project and its contests. """
        node = xml_doc.appendChild(xml_doc.createElement(tags.XML_TAG_PROJECT))

        # check what modules are the widgets using
        modules = []
        for w in self.widgets:
            klass = Widget.get_from_gtk_widget(w).klass
            if klass.module not in modules:
                modules.append(klass.module)

        for module in modules:
            n = xml_doc.createElement(tags.XML_TAG_REQUIRES)
            n.setAttribute(tags.XML_TAG_LIB, module.root_library)
            n.setAttribute(tags.XML_TAG_PREFIX, module.widget_prefix)
            node.appendChild(n)

        # Append uimanager and action groups
        if self.uimanager is not None:
            node.appendChild(self._write_uimanager(xml_doc))
        
        # Append toplevel widgets. Each widget then takes care of
        # appending its children
        f = Widget.get_from_gtk_widget
        [node.appendChild(f(w).write(xml_doc)) for w in self.widgets \
         if f(w).is_toplevel()]
        
        return node

    def _max_separator_counter(self, xml_node, max_value):
        v = max_value
        if xml_node.nodeType == xml_node.ELEMENT_NODE:
            name = xml_node.getAttribute('name')
            # strip the 'sep' string
            name = name[3:]
            try:
                v = int(name)
            except:
                pass
        return max([v]+[self._max_separator_counter(child, max_value) \
                        for child in xml_node.childNodes])
            
    def _load_ui(self, ui_string):
        xml_doc = xml.dom.minidom.parseString(ui_string)
        ui_node = xml_doc.documentElement
        for child_node in util.xml_filter_nodes(ui_node.childNodes,
                                                ui_node.ELEMENT_NODE):
            name = child_node.getAttribute(tags.XML_TAG_NAME)
            gwidget = self.get_widget_by_name(name)
            prop = gwidget.get_glade_property('ui')
            util.xml_clean_node(child_node, (child_node.TEXT_NODE,))
            prop.set(child_node.toxml())
        
        # update the separator counter
        max_separator_counter = self._max_separator_counter(ui_node, 0)
        self._separator_counter = max_separator_counter + 1
            
    def open(path, app):
        from gazpacho.placeholder import Placeholder
        from gazpacho.widget import Widget
        from gazpacho.loader.widgettree import WidgetTree

        wt = WidgetTree(path, Placeholder, app)

        prj = Project(False, app)
        
        # Load the widgets
        for key, widget in wt._widgets.items():
            if isinstance(widget, gtk.Widget):
                if widget.flags() & gtk.TOPLEVEL:
                    Widget.create_from_gtk_widget(widget, prj, wt._widgets.keys(),
                                                  wt._signals)
                    prj.add_widget(widget)

        # Load the UI Manager
        if wt._uimanager is not None:    
            prj._uimanager = gtk.UIManager()
            for ag in wt._uimanager.get_action_groups():
                gag = get_gaction_group_from_gtk_action_group(ag)
                prj.add_action_group(gag)
                for action in ag.list_actions():
                    gaction = get_gaction_from_gtk_action(action, gag)
                    gag.append(gaction)
            prj._load_ui(wt._uimanager.get_ui())

        if prj is not None:
            prj.path = path
            prj.name = basename(path)
            prj.changed = False

        return prj

    open = staticmethod(open)
    
    def save(self, path):
        xml_doc = xml.dom.getDOMImplementation().createDocument(None, None,
                                                                None)
        root = self.write(xml_doc, path)
        if not root:
            return False

        # save the xml tree
        f = file(path, 'w')
        f.write('<?xml version="1.0" ?>\n')
        util.write_xml(f, xml_doc.documentElement)
        f.close()

        # save the ui (if we have any)
        if self.uimanager is not None and not self.save_uimanager_in_glade_file:
            uim = xml.dom.minidom.parseString(self.uimanager.get_ui())
            uim.normalize()
            f = file(util.get_ui_path(path), 'w')
            f.write('<?xml version="1.0" ?>\n')
            util.write_xml(f, uim.documentElement)
            f.close()
            
        self.path = path
        self.name = basename(path)
        self.changed = False
        return True

gobject.type_register(Project)
