/*
 * file routines
 *
 * Author:
 *   Dietmar Maurer (dm@vlsivie.tuwien.ac.at)
 *
 *
 */

#include "vefs.h"
#include "simple.h"

EFSFile*
efs_file_open (EFSDir *dir, const char *path, gint flags)
{
	EFSDriver *d;

	g_return_val_if_fail(dir != NULL, NULL);
	g_return_val_if_fail(path != NULL, NULL);

	d = dir->efs->driver;

	if (!(flags&EFS_RDWR)) flags |= flags|EFS_READ;

	return d->fops->file_open (dir, path, flags);
}

gint         
efs_file_close (EFSFile *file)
{
	EFSDriver *d;

	g_return_val_if_fail(file != NULL, -1);
	
	d = file->efs->driver;

	return d->fops->file_close (file);
}

gint32
efs_file_seek (EFSFile *file, gint32 offset, gint whence)
{
	EFSDriver *d;

	g_return_val_if_fail(file != NULL, -1);

	d = file->efs->driver;

	return d->fops->file_seek (file, offset, whence);
}

gint32
efs_file_read (EFSFile *file, void *buf, gint32 count)
{
	EFSDriver *d;

	g_return_val_if_fail (file != NULL, -1);
	g_return_val_if_fail (buf  != NULL, -1);

	d = file->efs->driver;

	return d->fops->file_read (file, buf, count);
}

gint32
efs_file_write (EFSFile *file, void *buf, gint32 count)
{
	EFSDriver *d;

	g_return_val_if_fail (file != NULL, -1);
	g_return_val_if_fail (buf  != NULL, -1);

	if (!(file->efs->mode&EFS_WRITE)) return -1;

	d = file->efs->driver;

	return d->fops->file_write (file, buf, count);
}

gint
efs_file_trunc (EFSFile *file, guint32 size)
{
	EFSDriver *d;

	g_return_val_if_fail (file != NULL, -1);

	if (!(file->efs->mode&EFS_WRITE)) return -1;

	d = file->efs->driver;

	return d->fops->file_trunc (file, size);
}

/**
 * efs_erase:
 * @dir: reference to a #EFS directory
 * @path: file or directory name
 *
 * Description: Delete the file/directory it refers to.
 *
 * Returns: zero on success, or -1 if an error occurred.
 **/

gint
efs_erase (EFSDir *dir, const char *path)
{
	EFSDriver *d;

	g_return_val_if_fail (dir != NULL, -1);
	g_return_val_if_fail (path != NULL, -1);

	if (!(dir->efs->mode&EFS_WRITE)) return -1;

	d = dir->efs->driver;

	return d->fops->erase (dir, path);
}

/**
 * efs_rename:
 * @dir: reference to a #EFS directory
 * @old_path: file or directory name
 * @new_path: new name
 *
 * Description: Rename a file or directory.
 *
 * Returns: zero on success, or -1 if an error occurred.
 **/

gint         
efs_rename (EFSDir *dir, const char *old_path, const char *new_path)
{
	EFSDriver *d;

	g_return_val_if_fail (dir != NULL, -1);
	g_return_val_if_fail (old_path != NULL, -1);
	g_return_val_if_fail (new_path != NULL, -1);

	if (!(dir->efs->mode&EFS_WRITE)) return -1;

	d = dir->efs->driver;

	return d->fops->rename (dir, old_path, new_path);
}


