/*
 * bonobo-ui-item.c: Generic toolbar item code
 *
 * Author:
 *	Michael Meeks (michael@helixcode.com)
 *
 * Copyright 2000 Helix Code, Inc.
 */

#include <config.h>
#include <gnome.h>

#include <bonobo/bonobo-object.h>
#include <bonobo/bonobo-widget.h>
#include <bonobo/bonobo-ui-item.h>

enum {
	STATE_ALTERED,
	ACTIVATE,
	LAST_SIGNAL
};
static guint signals [LAST_SIGNAL] = { 0 };

static void
ui_item_size_request (GtkWidget      *widget,
		       GtkRequisition *requisition)
{
	GtkBin *bin = GTK_BIN (widget);

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
		gtk_widget_size_request (bin->child, requisition);
}

static void
ui_item_size_allocate (GtkWidget     *widget,
			GtkAllocation *allocation)
{
	GtkBin *bin = GTK_BIN (widget);

	widget->allocation = *allocation;
	if (bin->child && GTK_WIDGET_VISIBLE (bin->child))
		gtk_widget_size_allocate (bin->child, allocation);
}

static void
ui_item_set_style (BonoboUIItem  *item,
		    GtkReliefStyle  relief,
		    GtkToolbarStyle look)
{
	if (item->widget && GTK_IS_BUTTON (item->widget))
		gtk_button_set_relief (GTK_BUTTON (item->widget),
				       relief);

	if (look == GTK_TOOLBAR_BOTH) {
		if (item->label)
			gtk_widget_show (item->label);
		if (item->pixmap)
			gtk_widget_show (item->pixmap);
	} else {
		if (item->pixmap) {
			gtk_widget_show (item->pixmap);
			gtk_widget_hide (item->label);
		} else if (item->label)
			gtk_widget_show (item->label);
	}
}

static void
ui_item_set_state (BonoboUIItem *item,
		    const char    *state)
{
	if (item->widget) {
		if (GTK_IS_TOGGLE_BUTTON (item->widget)) {
			gtk_toggle_button_set_active (
				GTK_TOGGLE_BUTTON (item->widget),
				atoi (state));
		}
	}
}

static void
ui_item_class_init (GtkWidgetClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass *)klass;
	BonoboUIItemClass *item_class = (BonoboUIItemClass *)klass;

	klass->size_request  = ui_item_size_request;
	klass->size_allocate = ui_item_size_allocate;

	item_class->set_style = ui_item_set_style;
	item_class->set_state = ui_item_set_state;
	item_class->state_altered = NULL;
	item_class->activate = NULL;

	signals [STATE_ALTERED] = gtk_signal_new (
		"state_altered", GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (BonoboUIItemClass, state_altered),
		gtk_marshal_NONE__POINTER,
		GTK_TYPE_NONE, 1, GTK_TYPE_STRING);

	signals [ACTIVATE] = gtk_signal_new (
		"activate", GTK_RUN_FIRST,
		object_class->type,
		GTK_SIGNAL_OFFSET (BonoboUIItemClass, activate),
		gtk_marshal_NONE__NONE,
		GTK_TYPE_NONE, 0);

	gtk_object_class_add_signals (object_class, signals, LAST_SIGNAL);
}

/**
 * bonobo_ui_item_get_type:
 *
 * Returns the GtkType for the BonoboUIItem class.
 */
GtkType
bonobo_ui_item_get_type (void)
{
	static GtkType type = 0;

	if (!type) {
		GtkTypeInfo info = {
			"BonoboUIItem",
			sizeof (BonoboUIItem),
			sizeof (BonoboUIItemClass),
			(GtkClassInitFunc) ui_item_class_init,
			(GtkObjectInitFunc) NULL,
			NULL, /* reserved 1 */
			NULL, /* reserved 2 */
			(GtkClassInitFunc) NULL
		};

		type = gtk_type_unique (GTK_TYPE_BIN, &info);
	}

	return type;
}

static GtkWidget *
item_finish (BonoboUIItem *item,
	     const char    *label,
	     GtkWidget     *pixmap)
{
	GtkWidget *vbox;

	gtk_container_add (GTK_CONTAINER (item), item->widget);
	GTK_WIDGET_UNSET_FLAGS (item->widget, GTK_CAN_FOCUS);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER (item->widget), vbox);

	gtk_widget_show (vbox);

	if (label) {
		item->label = gtk_label_new (label);
		gtk_box_pack_end (GTK_BOX (vbox), item->label,
				  FALSE, FALSE, 0);
		gtk_widget_show (item->label);
	}

	if (pixmap) {
		item->pixmap = pixmap;
		gtk_box_pack_end (GTK_BOX (vbox), item->pixmap,
				  FALSE, FALSE, 0);
		gtk_widget_show (pixmap);
	}

	gtk_widget_show (item->widget);

	gtk_widget_show (GTK_WIDGET (item));

	return GTK_WIDGET (item);
}

static void
proxy_click (GtkWidget *button, BonoboUIItem *item)
{
	gtk_signal_emit (GTK_OBJECT (item), signals [ACTIVATE]);
}

GtkWidget *
bonobo_ui_item_new_item (const char *label,
			  GtkWidget  *pixmap)
{
	BonoboUIItem *item = gtk_type_new (BONOBO_UI_ITEM_TYPE);

	item->widget = gtk_button_new ();

 	gtk_signal_connect (GTK_OBJECT (item->widget), "clicked",
			    (GtkSignalFunc) proxy_click, item);

	return item_finish (item, label, pixmap);
}

static void
proxy_toggle_click (GtkWidget *button, BonoboUIItem *item)
{
	gboolean active;
	char    *new_state;

	active = gtk_toggle_button_get_active (
		GTK_TOGGLE_BUTTON (button));

	new_state = g_strdup_printf ("%d", active);

	gtk_signal_emit (GTK_OBJECT (item), signals [STATE_ALTERED],
			 new_state);

	g_free (new_state);
}

GtkWidget *
bonobo_ui_item_new_toggle (const char *label,
			    GtkWidget  *pixmap)
{
	BonoboUIItem *item = gtk_type_new (BONOBO_UI_ITEM_TYPE);

	item->widget = gtk_toggle_button_new ();

	gtk_signal_connect (GTK_OBJECT (item->widget), "clicked",
			    (GtkSignalFunc) proxy_toggle_click, item);

 	gtk_signal_connect (GTK_OBJECT (item->widget), "clicked",
			    (GtkSignalFunc) proxy_click, item);

	return item_finish (item, label, pixmap);
}

GtkWidget *
bonobo_ui_item_new_separator (void)
{
	BonoboUIItem *item = gtk_type_new (BONOBO_UI_ITEM_TYPE);

	/* FIXME: what if we flip the toolbar or do something daft ? */
	/* FIXME: relief */
	item->widget = gtk_vseparator_new ();
	item->label  = NULL;
	item->pixmap = NULL;

	gtk_widget_set_sensitive (item->widget, FALSE);
	gtk_widget_show (item->widget);

	gtk_container_add (GTK_CONTAINER (item), item->widget);
	gtk_widget_show (GTK_WIDGET (item));

	return GTK_WIDGET (item);
}

GtkWidget *
bonobo_ui_item_new_control (Bonobo_Unknown control)
{
	BonoboUIItem *item = gtk_type_new (BONOBO_UI_ITEM_TYPE);

	item->widget = bonobo_widget_new_control_from_objref
		(bonobo_object_dup_ref (control, NULL),
		 CORBA_OBJECT_NIL);
	item->label  = NULL;
	item->pixmap = NULL;

	gtk_widget_show (item->widget);

	gtk_container_add (GTK_CONTAINER (item), item->widget);
	gtk_widget_show (GTK_WIDGET (item));

	return GTK_WIDGET (item);
}

void
bonobo_ui_item_set_state (BonoboUIItem  *item,
			   const char     *state)
{
	g_return_if_fail (BONOBO_IS_UI_ITEM (item));
	
	BONOBO_UI_ITEM_CLASS (
		GTK_OBJECT (item)->klass)->set_state (item, state);
}

void
bonobo_ui_item_set_style (BonoboUIItem  *item,
			   GtkReliefStyle  relief,
			   GtkToolbarStyle look)
{
	g_return_if_fail (BONOBO_IS_UI_ITEM (item));

	BONOBO_UI_ITEM_CLASS (
		GTK_OBJECT (item)->klass)->set_style (item, relief, look);
}

void
bonobo_ui_item_set_tooltip (BonoboUIItem  *item,
			     GtkTooltips    *tooltips,
			     const char     *descr)
{
	g_return_if_fail (BONOBO_IS_UI_ITEM (item));
	g_return_if_fail (GTK_IS_TOOLTIPS (tooltips));

	if (descr && item->widget) {
		gtk_tooltips_set_tip (tooltips, item->widget,
				      descr, NULL);
	}
}
