/* dir.c - directory access wrapper functions

   Copyright (C) 2000 Maurer IT Systemlsungen KEG

   The Gnome Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The Gnome Library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the Gnome Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.

   Author: Dietmar Maurer <dietmar@maurer-it.com>

*/

#include "efs_internal.h"

/**
 * efs_dir_open:
 * @dir: return value
 * @parent: parent directory
 * @path: filesystem path
 * @flags: access flags
 *
 * Description: Opens or creates the #EFS directory @path. @path
 * is simply the name of the directory like "test_dir". "./test_dir"
 * is not valid and will always return NULL. The following @flags    
 * are supported:
 *
 * %EFS_CREATE If @path does not exist it will be created.
 *
 * %EFS_EXCL  When used with %EFS_CREATE, if @path already exists it is an 
 * error and the open will fail.
 *
 * Returns: a pointer to the subdirectory, or zero if an error occurred.
 */

EFSResult
efs_dir_open (EFSDir **dir, EFSDir *parent, const char *path, gint flags)
{
	EFSResult res;

	g_return_val_if_fail (dir != NULL, EFS_ERR_INVAL);
	*dir = NULL;
	g_return_val_if_fail (parent != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (parent->mode&EFS_DIR, EFS_ERR_INVAL);
	g_return_val_if_fail (path != NULL, EFS_ERR_INVAL);

	if (!(parent->efs->mode&EFS_WRITE)) flags &= ~EFS_CREATE; 

	res = parent->efs->driver->dops->dir_open (dir, parent, path, flags);
	if (!res) {
		if (!(*dir)) return EFS_ERR_INT;
		(*dir)->mode |= EFS_DIR;
	}

	return res;
}

/**
 * efs_dir_close
 * @dir: the directory to close.
 * 
 * Description: Closes @dir. This function 
 * recurses its opened files/subdirs. This 
 * function can be called on the root #EFS
 * after having called efs_commit.
 *
 * Returns: 0 upon success. -1 upon failure.
 */

EFSResult         
efs_dir_close (EFSDir *dir)
{
	g_return_val_if_fail (dir != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (dir->mode&EFS_DIR, EFS_ERR_INVAL);
	g_return_val_if_fail (!(dir->mode&EFS_ROOT), EFS_ERR_INVAL);

	return dir->efs->driver->dops->dir_close (dir);
}

/**
 * efs_dir_seek
 * @dir: the directory to seek
 * @offset:
 *
 * Description: This function sets the location in the  directory
 * stream  from  which  the  next  efs_dir_read() call will start.
 * efs_dir_seek() should  be  used  with  an  offset  returned  by
 * efs_dir_tell() or zero.
 *
 * Returns: 0 upon success. -1 upon failure.
 */

EFSResult 
efs_dir_seek (EFSDir *dir, guint32 offset)
{
	g_return_val_if_fail (dir != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (dir->mode&EFS_DIR, EFS_ERR_INVAL);

	return dir->efs->driver->dops->dir_seek (dir, offset);
}

/**
 * efs_dir_tell
 * @dir: directory descriptor
 * @pos: return value
 * 
 * Description: This function obtains the current value of the file
 * position indicator for @dir.
 *
 * Returns: the current position indicator, or -1 on error.
 */

EFSResult
efs_dir_tell (EFSDir *dir, guint32 *pos)
{
	g_return_val_if_fail (dir != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (dir->mode&EFS_DIR, EFS_ERR_INVAL);
	g_return_val_if_fail (pos != NULL, EFS_ERR_INVAL);

	*pos = dir->pos;

	return EFS_ERR_OK;
}

/**
 * efs_dir_read
 * @dir: the directory to close.
 * @de: a place to store the result
 *
 * Description: allows you to browse @dir content.
 *
 * Returns: it will return a positive value and store the next 
 * #EFSDirEntry in @de till all of them have been returned. It will 
 * return zero after the last #EFSDirEntry has been returned, or -1 on error.
 */

EFSResult
efs_dir_read (EFSDir *dir,  EFSDirEntry *de)
{
	g_return_val_if_fail (dir != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (dir->mode&EFS_DIR, EFS_ERR_INVAL);
	g_return_val_if_fail (de != NULL, EFS_ERR_INVAL);
	
	return dir->efs->driver->dops->dir_read (dir, de);
}

/**
 * efs_rename:
 * @dir: reference to a #EFS directory
 * @old_path: file or directory name
 * @new_path: new name
 *
 * Description: Rename a file or directory.
 *
 * Returns: zero on success, or -1 if an error occurred.
 */

EFSResult
efs_rename (EFSDir *dir, const char *old_path, const char *new_path)
{
	g_return_val_if_fail (dir != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (dir->mode&EFS_DIR, EFS_ERR_INVAL);
	g_return_val_if_fail (old_path != NULL, EFS_ERR_INVAL);
	g_return_val_if_fail (new_path != NULL, EFS_ERR_INVAL);

	if (!(dir->mode&EFS_WRITE)) return EFS_ERR_PERM;
	if (!(dir->efs->mode&EFS_WRITE)) return EFS_ERR_PERM;
	
	if (!strcmp (old_path, new_path)) return EFS_ERR_OK;

	return dir->efs->driver->dops->rename (dir, old_path, new_path);
}


