/* This is -*- C -*- */
/* $Id: guppi-seq.c,v 1.3 2000/04/13 19:45:19 trow Exp $ */

/*
 * guppi-seq.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-seq.h"
#include "guppi-seq-impl.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

static void
guppi_seq_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_seq_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_seq_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_seq_class_init(GuppiSeqClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GUPPI_TYPE_DATA);

  object_class->get_arg = guppi_seq_get_arg;
  object_class->set_arg = guppi_seq_set_arg;
  object_class->destroy = guppi_seq_destroy;
  object_class->finalize = guppi_seq_finalize;

}

static void
guppi_seq_init(GuppiSeq* obj)
{

}

GtkType
guppi_seq_get_type(void)
{
  static GtkType guppi_seq_type = 0;
  if (!guppi_seq_type) {
    static const GtkTypeInfo guppi_seq_info = {
      "GuppiSeq",
      sizeof(GuppiSeq),
      sizeof(GuppiSeqClass),
      (GtkClassInitFunc)guppi_seq_class_init,
      (GtkObjectInitFunc)guppi_seq_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_seq_type = gtk_type_unique(GUPPI_TYPE_DATA, &guppi_seq_info);
  }
  return guppi_seq_type;
}

GuppiData*
guppi_seq_new(void)
{
  return GUPPI_DATA(gtk_type_new(guppi_seq_get_type()));
}

/*****************************************************************************/

void
guppi_seq_size_hint(GuppiSeq* seq, gsize expected_size)
{
  GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);

  /* Read-only data silently ignores size hints.  It likes the size it
     is just fine, thank you. */
  if (guppi_data_is_read_only(GUPPI_DATA(seq)))
    return;

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->size_hint);
  (impl_class->size_hint)(impl, expected_size);

}

void
guppi_seq_indices(const GuppiSeq* seq, gint* min, gint* max)
{
  const GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->get_bounds);
  (impl_class->get_bounds)(impl, min, max);
}

gint
guppi_seq_min_index(const GuppiSeq* seq)
{
  gint min=0;
  g_return_val_if_fail(seq != NULL, 0);
  guppi_seq_indices(seq, &min, NULL);
  return min;
}

gint
guppi_seq_max_index(const GuppiSeq* seq)
{
  gint max=-1;
  g_return_val_if_fail(seq != NULL, -1);
  guppi_seq_indices(seq, NULL, &max);
  return max;
}

gsize
guppi_seq_size(const GuppiSeq* seq)
{
  gint min, max;
  g_return_val_if_fail(seq != NULL, 0);
  guppi_seq_indices(seq, &min, &max);
  g_assert(max+1 >= min);
  return max+1-min;
}

gboolean
guppi_seq_empty(const GuppiSeq* seq)
{
  g_return_val_if_fail(seq != NULL, TRUE);
  return guppi_seq_size(seq) == 0;
}

gboolean
guppi_seq_nonempty(const GuppiSeq* seq)
{
  g_return_val_if_fail(seq != NULL, FALSE);
  return guppi_seq_size(seq) > 0;
}

gboolean
guppi_seq_in_bounds(const GuppiSeq* seq, gint i)
{
  gint min=0, max=-1;
  g_return_val_if_fail(seq != NULL, FALSE);
  guppi_seq_indices(seq, &min, &max);
  return min <= i && i <= max;
}

gboolean
guppi_seq_contains_bounds(const GuppiSeq* a, const GuppiSeq* b)
{
  gint amin=0, amax=-1, bmin=0, bmax=-1;
  g_return_val_if_fail(a != NULL, FALSE);
  g_return_val_if_fail(b != NULL, FALSE);
  guppi_seq_indices(a, &amin, &amax);
  guppi_seq_indices(b, &bmin, &bmax);

  return amin <= bmin && bmax <= amax;
}

gboolean
guppi_seq_equal_bounds(const GuppiSeq* a, const GuppiSeq* b)
{
  gint amin=0, amax=-1, bmin=0, bmax=-1;
  g_return_val_if_fail(a != NULL, FALSE);
  g_return_val_if_fail(b != NULL, FALSE);
  guppi_seq_indices(a, &amin, &amax);
  guppi_seq_indices(b, &bmin, &bmax);

  return amin == bmin && bmax == amax;
}

void
guppi_seq_common_bounds(const GuppiSeq* a, const GuppiSeq* b,
			gint* min, gint* max)
{
  gint amin=0, amax=-1, bmin=0, bmax=-1;
  g_return_if_fail(a != NULL);
  g_return_if_fail(b != NULL);
  guppi_seq_indices(a, &amin, &amax);
  guppi_seq_indices(b, &bmin, &bmax);

  if (min)
    *min = MAX(amin, bmin);
  if (max)
    *max = MIN(amax, bmax);
}

void
guppi_seq_shift_indices(GuppiSeq* seq, gint x)
{
  GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));

  if (x == 0)
    return;

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->shift_indices);
  (impl_class->shift_indices)(impl, x);
}

void
guppi_seq_set_min_index(GuppiSeq* seq, gint x)
{
  gint min;
  g_return_if_fail(seq != NULL);
  min = guppi_seq_min_index(seq);
  guppi_seq_shift_indices(seq, x-min);
}

void
guppi_seq_set_max_index(GuppiSeq* seq, gint x)
{
  gint max;
  g_return_if_fail(seq != NULL);
  max = guppi_seq_max_index(seq);
  guppi_seq_shift_indices(seq, x-max);
}

gboolean
guppi_seq_validate(const GuppiSeq* seq, const gchar* str)
{
  return guppi_seq_validate_with_feedback(seq, str, NULL, 0);
}

gboolean
guppi_seq_validate_with_feedback(const GuppiSeq* seq,
				 const gchar* str,
				 gchar* error_buf, gsize errbuf_len)
{
  const GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_val_if_fail(seq != NULL, FALSE);
  g_return_val_if_fail(str != NULL, FALSE);

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->validate);

  return (impl_class->validate)(impl, str, error_buf, errbuf_len);
}

gchar*
guppi_seq_get_string(const GuppiSeq* seq, gint i)
{
  gchar buffer[512]; /* arbitrary upper limit */
  buffer[0] = '\0';
  guppi_seq_get_to_string(seq, i, buffer, 512);
  return g_strdup(buffer);
}

void
guppi_seq_get_to_string(const GuppiSeq* seq, gint i,
			gchar* sbuf, gsize sbuf_len)
{
  const GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(sbuf != NULL);

  if (sbuf_len == 0)
    return;

  g_return_if_fail(guppi_seq_in_bounds(seq, i));

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->get);

  (impl_class->get)(impl, i, sbuf, sbuf_len);
}

void
guppi_seq_set_from_string(GuppiSeq* seq, gint i,
			  const gchar* str)
{
  GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(str != NULL);

  g_return_if_fail(guppi_seq_in_bounds(seq, i));

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->set);

  (impl_class->set)(impl, i, str);
}

void
guppi_seq_insert_from_string(GuppiSeq* seq, gint i,
			     const gchar* str)
{
  GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(str != NULL);

  g_return_if_fail(guppi_seq_in_bounds(seq, i));

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->insert);

  (impl_class->insert)(impl, i, str);
}

void
guppi_seq_prepend_from_string(GuppiSeq* seq, const gchar* str)
{
  gint first;
  first = guppi_seq_min_index(seq);
  guppi_seq_insert_from_string(seq, first, str);
}

void
guppi_seq_append_from_string(GuppiSeq* seq, const gchar* str)
{
  gint last;
  last = guppi_seq_max_index(seq);
  guppi_seq_insert_from_string(seq, last+1, str);
}

void
guppi_seq_delete(GuppiSeq* seq, gint i)
{
  GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(guppi_seq_in_bounds(seq, i));

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->delete_many);
  (impl_class->delete_many)(impl, i, 1);
}

void
guppi_seq_delete_many(GuppiSeq* seq, gint i, gsize N)
{
  GuppiSeqImpl* impl;
  GuppiSeqImplClass* impl_class;

  g_return_if_fail(seq != NULL);
  g_return_if_fail(guppi_data_can_change(GUPPI_DATA(seq)));
  g_return_if_fail(guppi_seq_in_bounds(seq, i));

  if (N == 0)
    return;

  impl = GUPPI_SEQ_IMPL(guppi_data_impl(GUPPI_DATA(seq)));
  impl_class = GUPPI_SEQ_IMPL_CLASS(GTK_OBJECT(impl)->klass);

  g_assert(impl_class->delete_many);
  (impl_class->delete_many)(impl, i, N);
}





/* $Id: guppi-seq.c,v 1.3 2000/04/13 19:45:19 trow Exp $ */
