/* This is -*- C -*- */
/* $Id: guppi-plot-element.c,v 1.12 2000/04/25 13:57:07 trow Exp $ */

/*
 * guppi-plot-element.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-plot-plug-in.h"
#include "guppi-plot-element.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

enum {
  CHANGED_LABEL,
  SHOW,
  HIDE,
  RAISE_OR_LOWER,
  UPDATE,
  LAST_SIGNAL
};
static guint gpe_signals[LAST_SIGNAL] = { 0 };

static void
guppi_plot_element_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_plot_element_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_plot_element_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_plot_element_finalize(GtkObject* obj)
{
  GuppiPlotElement* gpe = GUPPI_PLOT_ELEMENT(obj);

  if (parent_class->finalize)
    parent_class->finalize(obj);

  if (gpe->state) {
    gtk_object_unref(GTK_OBJECT(gpe->state));
    gpe->state = NULL;
  }
  
  g_free(gpe->label);
  gpe->label = NULL;
}

static void
guppi_plot_element_class_init(GuppiPlotElementClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  gpe_signals[CHANGED_LABEL] = 
    gtk_signal_new("changed_label",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiPlotElementClass, changed_label),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gpe_signals[SHOW] = 
    gtk_signal_new("show",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiPlotElementClass, show),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gpe_signals[HIDE] = 
    gtk_signal_new("hide",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiPlotElementClass, hide),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gpe_signals[RAISE_OR_LOWER] = 
    gtk_signal_new("raise_or_lower",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiPlotElementClass, raise_or_lower),
		   gtk_marshal_NONE__INT,
		   GTK_TYPE_NONE, 1,
		   GTK_TYPE_INT);

  gpe_signals[UPDATE] = 
    gtk_signal_new("update",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiPlotElementClass, update),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);

  gtk_object_class_add_signals(object_class, gpe_signals, LAST_SIGNAL);

  object_class->get_arg = guppi_plot_element_get_arg;
  object_class->set_arg = guppi_plot_element_set_arg;
  object_class->destroy = guppi_plot_element_destroy;
  object_class->finalize = guppi_plot_element_finalize;

  klass->type_name = NULL;
  klass->make_canvas_item = NULL;
  klass->make_state_widget = NULL;
  
  guppi_plot_toolkit_set_button_tool(&klass->default_toolkit,
				     1, 0,
				     guppi_plot_tool_new_rescale(0.90));

  guppi_plot_toolkit_set_button_tool(&klass->default_toolkit,
				     2, 0,
				     guppi_plot_tool_new_drag());

  guppi_plot_toolkit_set_button_tool(&klass->default_toolkit,
				     3, 0,
				     guppi_plot_tool_new_rescale(1/0.90));
  
  guppi_plot_toolkit_set_button_tool(&klass->default_toolkit,
				     1, GDK_SHIFT_MASK,
				     guppi_plot_tool_new_reframe());

  guppi_plot_toolkit_set_button_tool(&klass->default_toolkit,
				     3, GDK_SHIFT_MASK | GDK_CONTROL_MASK,
				     guppi_plot_tool_new_recenter());
}

static void
guppi_plot_element_init(GuppiPlotElement* obj)
{
  obj->visible = TRUE;
  obj->label = NULL;
}

GtkType
guppi_plot_element_get_type(void)
{
  static GtkType guppi_plot_element_type = 0;
  if (!guppi_plot_element_type) {
    static const GtkTypeInfo guppi_plot_element_info = {
      "GuppiPlotElement",
      sizeof(GuppiPlotElement),
      sizeof(GuppiPlotElementClass),
      (GtkClassInitFunc)guppi_plot_element_class_init,
      (GtkObjectInitFunc)guppi_plot_element_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_plot_element_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_plot_element_info);
  }
  return guppi_plot_element_type;
}

GuppiPlotElement*
guppi_plot_element_new(const gchar* name, const gchar* first_arg_name, ...)
{
  GuppiPlotElement* gpe;
  GuppiItemState* state;
  GSList* arg_list;
  GSList* info_list;
  gchar* error;
  va_list var_args;

  g_return_val_if_fail(name != NULL, NULL);

  gpe = guppi_plot_element_newv(name, 0, NULL);
  g_return_val_if_fail(gpe != NULL, NULL);

  state = guppi_plot_element_state(gpe);

  va_start(var_args, first_arg_name);
  error = gtk_object_args_collect(GTK_OBJECT_TYPE(state),
				  &arg_list,
				  &info_list,
				  first_arg_name,
				  var_args);

  if (error) {
    g_warning("guppi_plot_element_new(): %s", error);
    g_free(error);
  } else {
    GSList* slist_arg;
    GSList* slist_info;

    slist_arg = arg_list;
    slist_info = info_list;
    while (slist_arg) {
      gtk_object_arg_set(GTK_OBJECT(state), slist_arg->data, slist_info->data);
      slist_arg = g_slist_next(slist_arg);
      slist_info = g_slist_next(slist_info);
    }
    gtk_args_collect_cleanup(arg_list, info_list);
  }

  return gpe;
}


GuppiPlotElement*
guppi_plot_element_newv(const gchar* name, guint nargs, GtkArg* args)
{
  GuppiPlugIn* plug_in;
  GuppiPlotPlugIn* plot_plug_in;
  GuppiPlotElement* gpe;
  GtkType st;
  
  g_return_val_if_fail(name != NULL, NULL);


  /* Find our plug-in. */
  plug_in = guppi_plug_in_lookup("plot", name);
  if (plug_in == NULL) {
    g_warning("Unknown plot plug-in: \"%s\"", name);
    return NULL;
  }
  g_return_val_if_fail(GUPPI_IS_PLOT_PLUG_IN(plug_in), NULL);
  plot_plug_in = GUPPI_PLOT_PLUG_IN(plug_in);

  /* Build our plot element. */
  g_return_val_if_fail(plot_plug_in->element_constructor != NULL, NULL);
  gpe = (plot_plug_in->element_constructor)();
  g_assert(gpe != NULL);

  /* Build the appropriate state object, passing in our args. */
  st = GUPPI_PLOT_ELEMENT_CLASS(GTK_OBJECT(gpe)->klass)->state_type;
  //    g_return_val_if_fail(st != 0, NULL);
  if (st != 0) {
    gpe->state = GUPPI_ITEM_STATE(gtk_object_newv(st, nargs, args));
    g_return_val_if_fail(gpe->state != NULL, NULL);
  } else {
    g_warning("No state type for %s", guppi_plot_element_type_name(gpe));
  }

  return gpe;
}

GuppiItemState*
guppi_plot_element_state(GuppiPlotElement* gpe)
{
  g_return_val_if_fail(gpe != NULL, NULL);

  g_assert(gpe->state != NULL);

  return gpe->state;
}

const gchar*
guppi_plot_element_type_name(GuppiPlotElement* gpe)
{
  const gchar* s;

  g_return_val_if_fail(gpe != NULL, NULL);

  s = GUPPI_PLOT_ELEMENT_CLASS(GTK_OBJECT(gpe)->klass)->type_name;
  return s ? s : _("(unknown type)");
}

const gchar*
guppi_plot_element_label(GuppiPlotElement* gpe)
{
  g_return_val_if_fail(gpe != NULL, NULL);

  return gpe->label ? gpe->label : _("(unnamed)");
}

void
guppi_plot_element_set_label(GuppiPlotElement* gpe, const gchar* label)
{
  g_return_if_fail(gpe != NULL);

  g_free(gpe->label);
  gpe->label = g_strdup(label);
  gtk_signal_emit(GTK_OBJECT(gpe), gpe_signals[CHANGED_LABEL]);
}

void
guppi_plot_element_set_gridpos(GuppiPlotElement* gpe, gint x, gint y)
{
  g_return_if_fail(gpe != NULL);

  gpe->gridpos_x = x;
  gpe->gridpos_y = y;
}

/*****************************************************************************/

static gint
tool_repeat_handler(gpointer data)
{
  GuppiPlotTool* tool = GUPPI_PLOT_TOOL(data);
  (tool->repeat)(tool, tool->canvas_item);
  return TRUE;
}

static gint
event_dispatcher(GuppiCanvasItem* gci, GdkEvent* ev, GuppiPlotElement* elem)
{
  GuppiPlotToolkit* tk = NULL;
  GuppiPlotTool* tool;
  double x, y;

  tool = GUPPI_PLOT_TOOL0(gci->active_tool);

  if (tool && tool->canvas_item != gci)
    return FALSE;

  if (ev->type == GDK_BUTTON_PRESS && tool == NULL) {

    tk = elem->local_toolkit;
    if (tk == NULL) 
      tk = &GUPPI_PLOT_ELEMENT_CLASS(GTK_OBJECT(elem)->klass)->default_toolkit;
    if (tk == NULL)
      return FALSE;

    tool = guppi_plot_toolkit_get_button_tool(tk,
					      ev->button.button,
					      ev->button.state);
    gci->active_tool = tool;
    
    if (tool) {
      gdk_pointer_grab(GTK_WIDGET(GNOME_CANVAS_ITEM(gci)->canvas)->window,
		       TRUE,
		       GDK_BUTTON_PRESS_MASK | GDK_POINTER_MOTION_HINT_MASK | GDK_BUTTON_RELEASE_MASK,
		       GTK_WIDGET(GNOME_CANVAS_ITEM(gci)->canvas)->window,
		       tool->cursor,
		       GDK_CURRENT_TIME);

      tool->canvas_item = gci;

      tool->raw_x = tool->raw_start_x = ev->button.x;
      tool->raw_y = tool->raw_start_y = ev->button.y;

      guppi_canvas_item_world2plot(gci, ev->button.x, ev->button.y, &x, &y);

      tool->start_x = tool->x = x;
      tool->start_y = tool->y = y;

      if (tool->first)
	(tool->first)(tool, gci);

      if (tool->repeating && tool->repeat)
	tool->repeater_tag = gtk_timeout_add(tool->repeat_interval,
					     tool_repeat_handler,
					     tool);

      if (tool->cue_type == GPTPC_FRAME) {
	tool->cue_item =
	  gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS_ITEM(gci)->canvas),
				gnome_canvas_rect_get_type(),
				"outline_color", "black",
				"fill_color_rgba", tool->cue_fill_color,
				"width_pixels", (guint)1,
				"x1", tool->raw_start_x,
				"y1", tool->raw_start_y,
				"x2", tool->raw_start_x,
				"y2", tool->raw_start_y,
				NULL);

      } else if (tool->cue_type == GPTPC_CIRCLE) {
	tool->cue_item =
	  gnome_canvas_item_new(gnome_canvas_root(GNOME_CANVAS_ITEM(gci)->canvas),
				gnome_canvas_ellipse_get_type(),
				"outline_color", "black",
				"fill_color_rgba", tool->cue_fill_color,
				"width_pixels", (guint)1,
				"x1", tool->raw_start_x - tool->cue_arg,
				"y1", tool->raw_start_y - tool->cue_arg,
				"x2", tool->raw_start_x + tool->cue_arg,
				"y2", tool->raw_start_y + tool->cue_arg,
				NULL);
      }

    }

    return tool != NULL;

  } else if (ev->type == GDK_BUTTON_RELEASE && tool != NULL) {
    
    if (tool->cue_item) {
      gtk_object_destroy(GTK_OBJECT(tool->cue_item));
      tool->cue_item = NULL;
    }

    if (tool->repeating)
      gtk_timeout_remove(tool->repeater_tag);

    if (tool->last) {
      tool->raw_prev_x = tool->raw_x;
      tool->raw_prev_y = tool->raw_y;
      tool->prev_x = tool->x;
      tool->prev_y = tool->y;
      guppi_canvas_item_world2plot(gci, ev->button.x, ev->button.y, &x, &y);
      tool->raw_x = ev->button.x;
      tool->raw_y = ev->button.y;
      tool->x = x;
      tool->y = y;
      (tool->last)(tool, gci);
    }

    gdk_pointer_ungrab(GDK_CURRENT_TIME);

    tool->canvas_item = NULL;
    gci->active_tool = NULL;
    

    return TRUE;

  } else if (ev->type == GDK_MOTION_NOTIFY && tool != NULL &&
	     (tool->tracks_motion || tool->middle || tool->cue_item != NULL)) {

    tool->raw_prev_x = tool->raw_x;
    tool->raw_prev_y = tool->raw_y;
    tool->prev_x = tool->x;
    tool->prev_y = tool->y;
    guppi_canvas_item_world2plot(gci, ev->motion.x, ev->motion.y, &x, &y);
    tool->raw_x = ev->motion.x;
    tool->raw_y = ev->motion.y;

    if (tool->cue_item) {
      if (tool->cue_type == GPTPC_FRAME) {

	gnome_canvas_item_set(tool->cue_item,
			      "x1", MIN(tool->raw_start_x, tool->raw_x),
			      "y1", MIN(tool->raw_start_y, tool->raw_y),
			      "x2", MAX(tool->raw_start_x, tool->raw_x),
			      "y2", MAX(tool->raw_start_y, tool->raw_y),
			      NULL);
      } else if (tool->cue_type == GPTPC_CIRCLE) {

	gnome_canvas_item_set(tool->cue_item,
			      "x1", tool->raw_x - tool->cue_arg,
			      "y1", tool->raw_y - tool->cue_arg,
			      "x2", tool->raw_x + tool->cue_arg,
			      "y2", tool->raw_y + tool->cue_arg,
			      NULL);
      } else {
	g_assert_not_reached();
      }
    }
			    

    tool->x = x;
    tool->y = y;
    if (tool->middle) 
      (tool->middle)(tool, gci);
    if (ev->motion.is_hint)
      gdk_window_get_pointer(ev->motion.window, NULL, NULL, NULL);
    
    return TRUE;
  }

  return FALSE;
}

#if 0
static void
resize_cb(GtkWidget* w, GtkAllocation* al, GuppiCanvasItem* gci)
{
  double x0, y0, x1, y1;

  /* Get the bounding world coordinates of the entire visible piece of
     the canvas. */
  gnome_canvas_window_to_world(GNOME_CANVAS_ITEM(gci)->canvas,
			       0, 0, &x0, &y0);
  gnome_canvas_window_to_world(GNOME_CANVAS_ITEM(gci)->canvas,
			       al->width, al->height, &x1, &y1);

  /* Set our item's world so as to fill up that world.  This is a hack
     in place of real geometry management. */
  guppi_canvas_item_set_world(gci, x0, y0, x1, y1);
}
#endif

GuppiCanvasItem*
guppi_plot_element_make_canvas_item(GuppiPlotElement* gpe,
				    GnomeCanvas* canv,
				    double x1, double y1,
				    double x2, double y2)
{
  GuppiCanvasItem* gci;
  GuppiPlotElementClass* klass;
  GuppiItemState* state;

  g_return_val_if_fail(gpe != NULL, NULL);
  g_return_val_if_fail(canv != NULL, NULL);

  klass = GUPPI_PLOT_ELEMENT_CLASS(GTK_OBJECT(gpe)->klass);
  if (klass->make_canvas_item) {

    gci = (klass->make_canvas_item)(gpe, canv, x1, y1, x2, y2);
    g_assert(gci);

  } else {
    g_assert(klass->item_type);
    
    state = guppi_plot_element_state(gpe);
    g_assert(state);
    
    gci = GUPPI_CANVAS_ITEM(gnome_canvas_item_new(gnome_canvas_root(canv),
						  klass->item_type,
						  "x1", x1, "y1", y1,
						  "x2", x2, "y2", y2,
						  NULL));
    g_assert(gci);

    guppi_canvas_item_set_state(gci, guppi_plot_element_state(gpe));
  }

  gtk_signal_connect(GTK_OBJECT(gci),
		     "event",
		     GTK_SIGNAL_FUNC(event_dispatcher),
		     gpe);

  return gci;
}

GtkWidget*
guppi_plot_element_make_state_editor(GuppiPlotElement* gpe)
{
  GtkWidget* (*fn)(GuppiPlotElement*);

  g_return_val_if_fail(gpe != NULL, NULL);

  fn = GUPPI_PLOT_ELEMENT_CLASS(GTK_OBJECT(gpe)->klass)->make_state_widget;
  if (fn == NULL) {
    g_warning("The virtual function make_state_editor() is not defined for type %s", guppi_plot_element_type_name(gpe));
    return NULL;
  } 

  return fn(gpe);
}
				     

void
guppi_plot_element_show(GuppiPlotElement* gpe)
{
  if (!gpe->visible) {
    gpe->visible = TRUE;
    gtk_signal_emit(GTK_OBJECT(gpe), gpe_signals[SHOW]);
  }
}

void
guppi_plot_element_hide(GuppiPlotElement* gpe)
{
  if (gpe->visible) {
    gpe->visible = FALSE;
    gtk_signal_emit(GTK_OBJECT(gpe), gpe_signals[HIDE]);
  }
}

void
guppi_plot_element_set_visible(GuppiPlotElement* gpe, gboolean v)
{
  if (v)
    guppi_plot_element_show(gpe);
  else
    guppi_plot_element_hide(gpe);
}

void
guppi_plot_element_reverse_visibility(GuppiPlotElement* gpe)
{
  g_return_if_fail(gpe != NULL);
  guppi_plot_element_set_visible(gpe, !gpe->visible);
}

void
guppi_plot_element_raise(GuppiPlotElement* gpe, gint n)
{
  g_return_if_fail(gpe != NULL);

  if (n != 0)
    gtk_signal_emit(GTK_OBJECT(gpe), gpe_signals[RAISE_OR_LOWER], n);
}

void
guppi_plot_element_raise_to_top(GuppiPlotElement* gpe)
{
  g_return_if_fail(gpe != NULL);

  /* This will, of course, fail when there are more than 10^6 elements
     in a single plot. :-) */
  guppi_plot_element_raise(gpe, 1000000);
}

void
guppi_plot_element_lower(GuppiPlotElement* gpe, gint n)
{
  g_return_if_fail(gpe != NULL);
  if (n != 0)
    guppi_plot_element_raise(gpe, -n);
}

void
guppi_plot_element_lower_to_bottom(GuppiPlotElement* gpe)
{
  g_return_if_fail(gpe != NULL);
  guppi_plot_element_raise(gpe, -1000000);
}

void
guppi_plot_element_request_update(GuppiPlotElement* gpe)
{
  g_return_if_fail(gpe != NULL);
  g_assert_not_reached();
  gtk_signal_emit(GTK_OBJECT(gpe), gpe_signals[UPDATE]);
}
  
/***********************************************************/

void
guppi_plot_element_edit(GuppiPlotElement* gpe)
{
  GtkWidget* win;
  GtkWidget* ed;

  win = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  ed = guppi_plot_element_make_state_editor(gpe);
  gtk_container_add(GTK_CONTAINER(win), ed);
  gtk_widget_show_all(win);
}




/* $Id: guppi-plot-element.c,v 1.12 2000/04/25 13:57:07 trow Exp $ */
