/* This is -*- C -*- */
/* $Id: guppi-layout-grid-cell.c,v 1.4 2000/05/03 17:06:51 trow Exp $ */

/*
 * guppi-layout-grid-cell.c
 *
 * Copyright (C) 2000 EMC Capital Management, Inc.
 *
 * Developed by Jon Trowbridge <trow@gnu.org> and
 * Havoc Pennington <hp@pobox.com>.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "guppi-layout-grid-cell.h"

static GtkObjectClass* parent_class = NULL;

enum {
  ARG_0
};

enum {
  CHANGED,
  LAST_SIGNAL
};

guint glgc_signals[LAST_SIGNAL] = { 0 };

static void
guppi_layout_grid_cell_get_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_layout_grid_cell_set_arg(GtkObject* obj, GtkArg* arg, guint arg_id)
{
  switch (arg_id) {

  default:
    break;
  };
}

static void
guppi_layout_grid_cell_destroy(GtkObject* obj)
{
  if (parent_class->destroy)
    parent_class->destroy(obj);
}

static void
guppi_layout_grid_cell_finalize(GtkObject* obj)
{
  if (parent_class->finalize)
    parent_class->finalize(obj);
}

static void
guppi_layout_grid_cell_class_init(GuppiLayoutGridCellClass* klass)
{
  GtkObjectClass* object_class = (GtkObjectClass*)klass;

  parent_class = gtk_type_class(GTK_TYPE_OBJECT);

  object_class->get_arg = guppi_layout_grid_cell_get_arg;
  object_class->set_arg = guppi_layout_grid_cell_set_arg;
  object_class->destroy = guppi_layout_grid_cell_destroy;
  object_class->finalize = guppi_layout_grid_cell_finalize;

  glgc_signals[CHANGED] = 
    gtk_signal_new("changed",
		   GTK_RUN_FIRST,
		   object_class->type,
		   GTK_SIGNAL_OFFSET(GuppiLayoutGridCellClass, changed),
		   gtk_marshal_NONE__NONE,
		   GTK_TYPE_NONE, 0);
		   
  gtk_object_class_add_signals(object_class, glgc_signals, LAST_SIGNAL);

}

static void
guppi_layout_grid_cell_init(GuppiLayoutGridCell* obj)
{
  obj->align_horizontal = GUPPI_CELL_ALIGN_FILL;
  obj->align_vertical = GUPPI_CELL_ALIGN_FILL;
}

GtkType
guppi_layout_grid_cell_get_type(void)
{
  static GtkType guppi_layout_grid_cell_type = 0;
  if (!guppi_layout_grid_cell_type) {
    static const GtkTypeInfo guppi_layout_grid_cell_info = {
      "GuppiLayoutGridCell",
      sizeof(GuppiLayoutGridCell),
      sizeof(GuppiLayoutGridCellClass),
      (GtkClassInitFunc)guppi_layout_grid_cell_class_init,
      (GtkObjectInitFunc)guppi_layout_grid_cell_init,
      NULL, NULL, (GtkClassInitFunc)NULL
    };
    guppi_layout_grid_cell_type = gtk_type_unique(GTK_TYPE_OBJECT, &guppi_layout_grid_cell_info);
  }
  return guppi_layout_grid_cell_type;
}

GuppiLayoutGridCell*
guppi_layout_grid_cell_new(void)
{
  return GUPPI_LAYOUT_GRID_CELL(gtk_type_new(guppi_layout_grid_cell_get_type()));
}

void
guppi_layout_grid_cell_calc_properties(GuppiLayoutGridCell* cell)
{
  GList* iter;
  double w0=0, w=0, w1=0, whung=0, h0=0, h=0, h1=0, hhung=0;
  GuppiCanvasItem* gci;
  gboolean block_flag;
  gboolean any_changed = FALSE;

  g_return_if_fail(cell != NULL);

  iter = cell->items;

  /* First, check and see if any of our items are flagged as being changed.
     If not, we can return without making any changes. */

  while (iter) {
    gci = GUPPI_CANVAS_ITEM(iter->data);

    if (gci->changed_geometry)
      any_changed = TRUE;

    block_flag = gci->block_changed_geometry_signals;
    gci->block_changed_geometry_signals = TRUE;
    guppi_canvas_item_calc_geometry(gci);
    gci->block_changed_geometry_signals = block_flag;

    if (gci->changed_geometry)
      any_changed = TRUE;

    iter = g_list_next(iter);
  }

  if (!any_changed) {
    /* If none of our items changed, we didn't change either. */
    return;
  }

  /* Now iter across our items, doing the necessary calculations. */

  iter = cell->items;

  while (iter) {
    gci = GUPPI_CANVAS_ITEM(iter->data);

    w0 = MAX(w0, gci->min_width);
    w = MAX(w, gci->optimal_width);
    w1 = w1 > 0 ? MIN(w1, gci->max_width) : gci->max_width;
    whung = MAX(whung, gci->width_hunger);

    h0 = MAX(h0, gci->min_height);
    h = MAX(h, gci->optimal_height);
    h1 = h1 > 0 ? MIN(h1, gci->max_height) : gci->max_height;
    hhung = MAX(hhung, gci->height_hunger);

    gci->changed_geometry = FALSE; /* Reset item's "changed" flag */

    iter = g_list_next(iter);
  }


  /* We only want to issue the "changed" signal if something actually
     did change. */

  if (cell->min_width != w0 ||
      cell->req_width != w ||
      cell->max_width != w1 ||
      cell->hunger_width != whung ||
      cell->min_height != h0 ||
      cell->req_height != h ||
      cell->max_height != h1 ||
      cell->hunger_height != hhung) {

    cell->changed_geometry = TRUE;
    
    cell->min_width = w0;
    cell->req_width = w;
    cell->max_width = w1;
    cell->hunger_width = whung;
    cell->min_height = h0;
    cell->req_height = h;
    cell->max_height = h1;
    cell->hunger_height = hhung;

    gtk_signal_emit(GTK_OBJECT(cell), glgc_signals[CHANGED]);
  }
}

static void
touch_and_calc(GuppiCanvasItem* gci, GuppiLayoutGridCell* cell)
{
  guppi_layout_grid_cell_calc_properties(cell);
}

void
guppi_layout_grid_cell_add_item(GuppiLayoutGridCell* cell,
				GuppiCanvasItem* item)
{
  g_return_if_fail(cell != NULL);
  g_return_if_fail(item != NULL);

  cell->items = g_list_prepend(cell->items, item);
  item->need_geometry_recalc = TRUE;
  
  gtk_signal_connect(GTK_OBJECT(item),
		     "changed_geometry",
		     GTK_SIGNAL_FUNC(touch_and_calc),
		     cell);

  cell->changed_geometry = TRUE;
  gtk_signal_emit(GTK_OBJECT(cell), glgc_signals[CHANGED]);
}

void
guppi_layout_grid_cell_set_allocation(GuppiLayoutGridCell* cell,
				      double x, double y,
				      double width, double height)
{
  GList* iter;
  double x0, y0, x1, y1;

  g_return_if_fail(cell != NULL);
  g_return_if_fail(width > 0);
  g_return_if_fail(height > 0);

  cell->x = x;
  cell->y = y;
  cell->width = width;
  cell->height = height;

  g_assert(cell->align_horizontal != GUPPI_CELL_ALIGN_TOP &&
	   cell->align_horizontal != GUPPI_CELL_ALIGN_BOTTOM);
  g_assert(cell->align_vertical != GUPPI_CELL_ALIGN_LEFT &&
	   cell->align_vertical != GUPPI_CELL_ALIGN_RIGHT);

  /* This is the default behavior: to just fill up the space that we
     were given. */

  x0 = x;
  y0 = y;
  x1 = x+width;
  y1 = y+height;

  /* If we were given more than we asked for, the items can optionally
     only occupy the requested space and align themselves in the allocated
     space in a specified manner. */

  /* Should we be able to align on a per-canvas-item basis?  We might
     need to change this later to allow for that... */

  if (width > cell->req_width) {
    if (cell->align_horizontal == GUPPI_CELL_ALIGN_LEFT) {
      x0 = x;
      x1 = x + cell->req_width;
    } else if (cell->align_horizontal == GUPPI_CELL_ALIGN_RIGHT) {
      x0 = x + width - cell->req_width;
      x1 = x + width;
    } else if (cell->align_horizontal == GUPPI_CELL_ALIGN_CENTER) {
      x0 = x + (width - cell->req_width)/2;
      x1 = x + cell->req_width;
    }
  }

  if (height > cell->req_height) {
    if (cell->align_vertical == GUPPI_CELL_ALIGN_TOP) {
      y0 = y;
      y1 = y + cell->req_height;
    } else if (cell->align_vertical == GUPPI_CELL_ALIGN_BOTTOM) {
      y0 = y + height - cell->req_height;
      y1 = y + height;
    } else if (cell->align_vertical == GUPPI_CELL_ALIGN_CENTER) {
      y0 = y + (height - cell->req_height)/2;
      y1 = y0 + cell->req_height;
    }
  }

  /* Reposition our canvas items inside the new bounds. */

  iter = cell->items;
  while (iter) {
    guppi_canvas_item_set_world(GUPPI_CANVAS_ITEM(iter->data), x0, y0, x1, y1);
    iter = g_list_next(iter);
  }
}

/* $Id: guppi-layout-grid-cell.c,v 1.4 2000/05/03 17:06:51 trow Exp $ */
