/*
 * NotFoundHandler.java
 *
 * Brazil project web application toolkit,
 * export version: 2.1 
 * Copyright (c) 1999-2003 Sun Microsystems, Inc.
 *
 * Sun Public License Notice
 *
 * The contents of this file are subject to the Sun Public License Version 
 * 1.0 (the "License"). You may not use this file except in compliance with 
 * the License. A copy of the License is included as the file "license.terms",
 * and also available at http://www.sun.com/
 * 
 * The Original Code is from:
 *    Brazil project web application toolkit release 2.1.
 * The Initial Developer of the Original Code is: suhler.
 * Portions created by suhler are Copyright (C) Sun Microsystems, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s): cstevens, suhler.
 *
 * Version:  2.2
 * Created by suhler on 99/03/29
 * Last modified by suhler on 03/08/01 16:18:23
 */

package sunlabs.brazil.handler;

import sunlabs.brazil.server.Handler;
import sunlabs.brazil.server.Request;
import sunlabs.brazil.server.Server;
import sunlabs.brazil.server.FileHandler;
import java.io.File;
import java.io.IOException;

/**
 * Handler for returning "file not found" errors back to the client.
 * Look for the file "NotFound.html" in the current directory, and return it
 * if it exists.  Otherwise, return the "NotFound.html" file in the document
 * root directory.  If neither can be found, then punt, and let someone else
 * deal with it.
 * <p>
 * If more sophisticated processing is desired, then the
 * {@link UrlMapperHandler}
 * may be used in combination with the
 * {@link sunlabs.brazil.template.TemplateHandler}.
 *
 * <p>
 * Configuration parameters understood by this handler
 * <dl class=props>
 * <dt>root	<dd>The location of the document root for locating the
 *		default "not found" file (also looks using prefix of "").
 * <dt>prefix, suffix, glob, match
 * <dd>Specify the URL that triggers this handler.
 * (See {@link MatchString}).
 * <dt>fileName	<dd>The name of the file to send for missing files.
 *		Defaults to "notfound.html"
 * <dt>type	<dd>The file type, defaults to text/html
 * </dl>
 *
 * @author		Stephen Uhler
 * @version		2.2, 03/08/01
 */

public class NotFoundHandler implements Handler {
    static final String NAME =   "fileName";  // name of the not-found file
    static final String TYPE =   "type";  // type of the not-found file (should be looked up from the suffix)

    MatchString isMine;            // check for matching url
    String fileName;		// name of "not found" file
    String type;		// file type
    File rootFile;		// root not-found file

    /**
     * Extract the handler properties.
     * Get the URL prefix and default "missing" file name.
     */

    public boolean
    init(Server server, String prefix) {
	isMine = new MatchString(prefix, server.props);
	fileName = server.props.getProperty(prefix + NAME, "notfound.html");
	type = server.props.getProperty(prefix + TYPE, "text/html");
	String root = server.props.getProperty(prefix + FileHandler.ROOT,
		server.props.getProperty(FileHandler.ROOT, "."));
	rootFile = new File(root, fileName);
	server.log(Server.LOG_DIAGNOSTIC, prefix,
	    "looking for: " + rootFile);
	if (!rootFile.isFile()) {
	    server.log(Server.LOG_WARNING, prefix, "Can't find file: " + rootFile);
	}
	return true;
    }

    /**
     * Look for and deliver the "not found" file
     * Look in the current directory first, then in the doc root.
     * Only files whose suffixes have valid mime types are delivered.
     */

    public boolean
    respond(Request request) throws IOException {
	if (!isMine.match(request.url)) {
	    return false;
	}
	String missing = request.props.getProperty("fileName");
	if (missing == null) {
	    request.log(Server.LOG_DIAGNOSTIC, "No missing file found!!");
	    return false;
	}
	File name = new File((new File(missing)).getParent(), fileName);
	if (name.canRead() && name.isFile()) {
	    FileHandler.sendFile(request, name, 404, type);
	    request.log(Server.LOG_DIAGNOSTIC, "sending not-found file");
	} else if (rootFile.canRead() && rootFile.isFile())  {
	    FileHandler.sendFile(request, rootFile, 404, type);
	    request.log(Server.LOG_DIAGNOSTIC, "sending Root not-found file");
	} else {
	    return false;
	}
	return true;
    }
}
