/*
 * Decompiled with CFR 0.152.
 */
package org.apache.tez.analyzer.plugins;

import java.io.File;
import java.io.FilenameFilter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.apache.commons.io.FileUtils;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.conf.Configured;
import org.apache.hadoop.util.Tool;
import org.apache.tez.analyzer.Analyzer;
import org.apache.tez.analyzer.CSVResult;
import org.apache.tez.analyzer.Result;
import org.apache.tez.common.Preconditions;
import org.apache.tez.dag.api.TezException;
import org.apache.tez.history.ATSImportTool;
import org.apache.tez.history.parser.ATSFileParser;
import org.apache.tez.history.parser.ProtoHistoryParser;
import org.apache.tez.history.parser.SimpleHistoryParser;
import org.apache.tez.history.parser.datamodel.DagInfo;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public abstract class TezAnalyzerBase
extends Configured
implements Tool,
Analyzer {
    private static final Logger LOG = LoggerFactory.getLogger(TezAnalyzerBase.class);
    private static final String EVENT_FILE_NAME = "eventFileName";
    private static final String OUTPUT_DIR = "outputDir";
    private static final String SAVE_RESULTS = "saveResults";
    private static final String DAG_ID = "dagId";
    private static final String FROM_SIMPLE_HISTORY = "fromSimpleHistory";
    private static final String FROM_PROTO_HISTORY = "fromProtoHistory";
    private static final String HELP = "help";
    private static final int SEPARATOR_WIDTH = 80;
    private static final int MIN_COL_WIDTH = 12;
    private String outputDir;
    private boolean saveResults = false;

    public TezAnalyzerBase(Configuration config) {
        this.setConf(config);
    }

    private static Options buildOptions() {
        Option dagIdOption = Option.builder().argName(DAG_ID).longOpt(DAG_ID).desc("DagId that needs to be analyzed").hasArg().required(true).build();
        Option outputDirOption = Option.builder().argName(OUTPUT_DIR).longOpt(OUTPUT_DIR).desc("Directory to write outputs to.").hasArg().required(false).build();
        Option saveResults = Option.builder().argName(SAVE_RESULTS).longOpt(SAVE_RESULTS).desc("Saves results to output directory (optional)").hasArg(false).required(false).build();
        Option eventFileNameOption = Option.builder().argName(EVENT_FILE_NAME).longOpt(EVENT_FILE_NAME).desc("File with event data for the DAG").hasArg().required(false).build();
        Option fromSimpleHistoryOption = Option.builder().argName(FROM_SIMPLE_HISTORY).longOpt(FROM_SIMPLE_HISTORY).desc("Event data from Simple History logging. Must also specify event file").required(false).build();
        Option fromProtoHistoryOption = Option.builder().argName(FROM_PROTO_HISTORY).longOpt(FROM_PROTO_HISTORY).desc("Event data from Proto History logging. Must also specify event file").required(false).build();
        Option help = Option.builder().argName(HELP).longOpt(HELP).desc("print help").required(false).build();
        Options opts = new Options();
        opts.addOption(dagIdOption);
        opts.addOption(outputDirOption);
        opts.addOption(saveResults);
        opts.addOption(eventFileNameOption);
        opts.addOption(fromSimpleHistoryOption);
        opts.addOption(fromProtoHistoryOption);
        opts.addOption(help);
        return opts;
    }

    protected String getOutputDir() {
        return this.outputDir;
    }

    private void printUsage() {
        System.err.println("Analyzer base options are");
        Options options = TezAnalyzerBase.buildOptions();
        for (Option option : options.getOptions()) {
            System.err.println(option.getArgName() + " : " + option.getDescription());
        }
    }

    public int run(String[] args) throws Exception {
        SimpleHistoryParser parser;
        CommandLine cmdLine = null;
        try {
            cmdLine = new DefaultParser().parse(TezAnalyzerBase.buildOptions(), args);
        }
        catch (ParseException e) {
            System.err.println("Invalid options on command line");
            this.printUsage();
            return -1;
        }
        this.saveResults = cmdLine.hasOption(SAVE_RESULTS);
        if (cmdLine.hasOption(HELP)) {
            this.printUsage();
            return 0;
        }
        this.outputDir = cmdLine.getOptionValue(OUTPUT_DIR);
        if (this.outputDir == null) {
            this.outputDir = System.getProperty("user.dir");
        }
        String dagId = cmdLine.getOptionValue(DAG_ID);
        ArrayList<File> files = new ArrayList<File>();
        if (cmdLine.hasOption(EVENT_FILE_NAME)) {
            for (String file : cmdLine.getOptionValue(EVENT_FILE_NAME).split(",")) {
                File fileOrDir = new File(file);
                if (!fileOrDir.exists()) continue;
                if (fileOrDir.isFile()) {
                    files.add(fileOrDir);
                    continue;
                }
                files.addAll(this.collectFilesForDagId(fileOrDir, dagId));
            }
        }
        DagInfo dagInfo = null;
        if (files.isEmpty()) {
            if (cmdLine.hasOption(FROM_SIMPLE_HISTORY)) {
                System.err.println("Event file name must be specified when using simple history");
                this.printUsage();
                return -2;
            }
            if (cmdLine.hasOption(FROM_PROTO_HISTORY)) {
                System.err.println("Proto file name must be specified when using proto history");
                this.printUsage();
                return -2;
            }
            String[] importArgs = new String[]{"--dagId=" + dagId, "--downloadDir=" + this.outputDir};
            int result = ATSImportTool.process((String[])importArgs);
            if (result != 0) {
                System.err.println("Error downloading data from ATS");
                return -3;
            }
            files.add(new File(this.outputDir + "/" + dagId + ".zip"));
        }
        Preconditions.checkState((!files.isEmpty() ? 1 : 0) != 0);
        if (cmdLine.hasOption(FROM_SIMPLE_HISTORY)) {
            parser = new SimpleHistoryParser(files);
            dagInfo = parser.getDAGData(dagId);
        } else if (cmdLine.hasOption(FROM_PROTO_HISTORY)) {
            parser = new ProtoHistoryParser(files);
            dagInfo = parser.getDAGData(dagId);
        } else {
            parser = new ATSFileParser(files);
            dagInfo = parser.getDAGData(dagId);
        }
        Preconditions.checkState((boolean)dagInfo.getDagId().equals(dagId));
        this.analyze(dagInfo);
        Result result = this.getResult();
        if (this.saveResults) {
            String dagInfoFileName = this.outputDir + File.separator + this.getClass().getName() + "_" + dagInfo.getDagId() + ".dag";
            FileUtils.writeStringToFile((File)new File(dagInfoFileName), (String)dagInfo.toExtendedString());
            LOG.info("Saved dag info in " + dagInfoFileName);
            if (result instanceof CSVResult) {
                String fileName = this.outputDir + File.separator + this.getClass().getName() + "_" + dagInfo.getDagId() + ".csv";
                ((CSVResult)result).dumpToFile(fileName);
                LOG.info("Saved results in " + fileName);
            }
        }
        return 0;
    }

    private List<File> collectFilesForDagId(File parentDir, final String dagId) {
        File[] arrFiles = parentDir.listFiles(new FilenameFilter(){

            @Override
            public boolean accept(File dir, String name) {
                return name.contains(dagId);
            }
        });
        if (arrFiles == null || arrFiles.length == 0) {
            throw new RuntimeException(String.format("cannot find relevant files for dag: '%s' in dir: %s", dagId, parentDir));
        }
        List<File> files = Arrays.asList(arrFiles);
        LOG.info("collected files for dag: \n" + files.stream().map(f -> "\n" + f.getAbsolutePath()).collect(Collectors.toList()));
        return files;
    }

    public void printResults() throws TezException {
        Result result = this.getResult();
        if (result instanceof CSVResult) {
            Object[] headers = ((CSVResult)result).getHeaders();
            StringBuilder formatBuilder = new StringBuilder();
            int size = Math.max(12, 80 / headers.length);
            for (int i = 0; i < headers.length; ++i) {
                formatBuilder.append("%-").append(size).append("s ");
            }
            String format = formatBuilder.toString();
            StringBuilder sep = new StringBuilder();
            for (int i = 0; i < 80; ++i) {
                sep.append("-");
            }
            String separator = sep.toString();
            LOG.debug(separator);
            LOG.debug(String.format(format.toString(), headers));
            LOG.debug(separator);
            Iterator<String[]> recordsIterator = ((CSVResult)result).getRecordsIterator();
            while (recordsIterator.hasNext()) {
                String line = String.format(format, recordsIterator.next());
                LOG.debug(line);
            }
            LOG.debug(separator);
        }
    }
}

