/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdds.scm.safemode;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import org.apache.hadoop.hdds.client.RatisReplicationConfig;
import org.apache.hadoop.hdds.client.ReplicationConfig;
import org.apache.hadoop.hdds.conf.ConfigurationSource;
import org.apache.hadoop.hdds.protocol.DatanodeDetails;
import org.apache.hadoop.hdds.protocol.proto.HddsProtos;
import org.apache.hadoop.hdds.scm.events.SCMEvents;
import org.apache.hadoop.hdds.scm.ha.SCMContext;
import org.apache.hadoop.hdds.scm.node.NodeManager;
import org.apache.hadoop.hdds.scm.node.NodeStatus;
import org.apache.hadoop.hdds.scm.node.states.NodeNotFoundException;
import org.apache.hadoop.hdds.scm.pipeline.Pipeline;
import org.apache.hadoop.hdds.scm.pipeline.PipelineID;
import org.apache.hadoop.hdds.scm.pipeline.PipelineManager;
import org.apache.hadoop.hdds.scm.safemode.SCMSafeModeManager;
import org.apache.hadoop.hdds.scm.safemode.SafeModeExitRule;
import org.apache.hadoop.hdds.scm.server.upgrade.FinalizationManager;
import org.apache.hadoop.hdds.server.events.EventQueue;
import org.apache.hadoop.hdds.server.events.TypedEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HealthyPipelineSafeModeRule
extends SafeModeExitRule<Pipeline> {
    private static final Logger LOG = LoggerFactory.getLogger(HealthyPipelineSafeModeRule.class);
    private int healthyPipelineThresholdCount;
    private int currentHealthyPipelineCount = 0;
    private final double healthyPipelinesPercent;
    private final Set<PipelineID> processedPipelineIDs = new HashSet<PipelineID>();
    private final PipelineManager pipelineManager;
    private final int minHealthyPipelines;
    private final SCMContext scmContext;
    private final Set<PipelineID> unProcessedPipelineSet = new HashSet<PipelineID>();
    private final NodeManager nodeManager;

    HealthyPipelineSafeModeRule(EventQueue eventQueue, PipelineManager pipelineManager, SCMSafeModeManager manager, ConfigurationSource configuration, SCMContext scmContext, NodeManager nodeManager) {
        super(manager, eventQueue);
        this.pipelineManager = pipelineManager;
        this.scmContext = scmContext;
        this.nodeManager = nodeManager;
        this.healthyPipelinesPercent = configuration.getDouble("hdds.scm.safemode.healthy.pipeline.pct", 0.1);
        this.minHealthyPipelines = this.getMinHealthyPipelines(configuration);
        Preconditions.checkArgument((this.healthyPipelinesPercent >= 0.0 && this.healthyPipelinesPercent <= 1.0 ? 1 : 0) != 0, (Object)"hdds.scm.safemode.healthy.pipeline.pct value should be >= 0.0 and <= 1.0");
        this.initializeRule(false);
    }

    private int getMinHealthyPipelines(ConfigurationSource config) {
        int minDatanodes = config.getInt("hdds.scm.safemode.min.datanode", 3);
        return minDatanodes / 3;
    }

    @VisibleForTesting
    public synchronized void setHealthyPipelineThresholdCount(int actualPipelineCount) {
        this.healthyPipelineThresholdCount = (int)Math.ceil(this.healthyPipelinesPercent * (double)actualPipelineCount);
    }

    @Override
    protected TypedEvent<Pipeline> getEventType() {
        return SCMEvents.OPEN_PIPELINE;
    }

    @Override
    protected synchronized boolean validate() {
        boolean shouldRunSafemodeCheck = FinalizationManager.shouldCreateNewPipelines(this.scmContext.getFinalizationCheckpoint());
        if (!shouldRunSafemodeCheck) {
            LOG.info("All SCM pipelines are closed due to ongoing upgrade finalization. Bypassing healthy pipeline safemode rule.");
            return true;
        }
        return this.currentHealthyPipelineCount >= this.healthyPipelineThresholdCount;
    }

    @Override
    protected synchronized void process(Pipeline pipeline) {
        Preconditions.checkNotNull((Object)pipeline);
        if (pipeline.getType() != HddsProtos.ReplicationType.RATIS || ((RatisReplicationConfig)pipeline.getReplicationConfig()).getReplicationFactor() != HddsProtos.ReplicationFactor.THREE) {
            Logger safeModeManagerLog = SCMSafeModeManager.getLogger();
            if (safeModeManagerLog.isDebugEnabled()) {
                safeModeManagerLog.debug("Skipping pipeline safemode report processing as Replication type isn't RATIS or replication factor isn't 3.");
            }
            return;
        }
        if (this.processedPipelineIDs.contains(pipeline.getId())) {
            LOG.info("Skipping pipeline safemode report processing check as pipeline: {} is already recorded.", (Object)pipeline.getId());
            return;
        }
        List pipelineDns = pipeline.getNodes();
        if (pipelineDns.size() != 3) {
            LOG.warn("Only {} DNs reported this pipeline: {}, all 3 DNs should report the pipeline", (Object)pipelineDns.size(), (Object)pipeline.getId());
            return;
        }
        LinkedHashMap<DatanodeDetails, String> badDnsWithReasons = new LinkedHashMap<DatanodeDetails, String>();
        for (DatanodeDetails dn : pipelineDns) {
            try {
                NodeStatus status = this.nodeManager.getNodeStatus(dn);
                if (status.equals(NodeStatus.inServiceHealthy())) continue;
                String reason = String.format("Health: %s, Operational State: %s", status.getHealth(), status.getOperationalState());
                badDnsWithReasons.put(dn, reason);
            }
            catch (NodeNotFoundException e) {
                badDnsWithReasons.put(dn, "DN not registered with SCM");
            }
        }
        if (!badDnsWithReasons.isEmpty()) {
            String badDnSummary = badDnsWithReasons.entrySet().stream().map(entry -> String.format("DN %s: %s", ((DatanodeDetails)entry.getKey()).getID(), entry.getValue())).collect(Collectors.joining("; "));
            LOG.warn("Below DNs reported by Pipeline: {} are either in bad health or un-registered with SCMs. Details: {}", (Object)pipeline.getId(), (Object)badDnSummary);
            return;
        }
        this.getSafeModeMetrics().incCurrentHealthyPipelinesCount();
        ++this.currentHealthyPipelineCount;
        this.processedPipelineIDs.add(pipeline.getId());
        this.unProcessedPipelineSet.remove(pipeline.getId());
        if (this.scmInSafeMode()) {
            SCMSafeModeManager.getLogger().info("SCM in safe mode. Healthy pipelines reported count is {}, required healthy pipeline reported count is {}", (Object)this.currentHealthyPipelineCount, (Object)this.getHealthyPipelineThresholdCount());
        }
    }

    @Override
    public synchronized void refresh(boolean forceRefresh) {
        if (forceRefresh) {
            this.initializeRule(true);
        } else if (!this.validate()) {
            this.initializeRule(true);
        }
    }

    private synchronized void initializeRule(boolean refresh) {
        this.unProcessedPipelineSet.addAll(this.pipelineManager.getPipelines((ReplicationConfig)RatisReplicationConfig.getInstance((HddsProtos.ReplicationFactor)HddsProtos.ReplicationFactor.THREE), Pipeline.PipelineState.OPEN).stream().map(Pipeline::getId).collect(Collectors.toSet()));
        int pipelineCount = this.unProcessedPipelineSet.size();
        this.healthyPipelineThresholdCount = Math.max(this.minHealthyPipelines, (int)Math.ceil(this.healthyPipelinesPercent * (double)pipelineCount));
        if (refresh) {
            LOG.info("Refreshed total pipeline count is {}, healthy pipeline threshold count is {}", (Object)pipelineCount, (Object)this.healthyPipelineThresholdCount);
        } else {
            LOG.info("Total pipeline count is {}, healthy pipeline threshold count is {}", (Object)pipelineCount, (Object)this.healthyPipelineThresholdCount);
        }
        this.getSafeModeMetrics().setNumHealthyPipelinesThreshold(this.healthyPipelineThresholdCount);
    }

    @Override
    protected synchronized void cleanup() {
        this.processedPipelineIDs.clear();
        this.unProcessedPipelineSet.clear();
    }

    @VisibleForTesting
    public synchronized int getCurrentHealthyPipelineCount() {
        return this.currentHealthyPipelineCount;
    }

    @VisibleForTesting
    public synchronized int getHealthyPipelineThresholdCount() {
        return this.healthyPipelineThresholdCount;
    }

    @Override
    public String getStatusText() {
        String status = String.format("healthy Ratis/THREE pipelines (=%d) >= healthyPipelineThresholdCount (=%d)", this.getCurrentHealthyPipelineCount(), this.getHealthyPipelineThresholdCount());
        status = this.updateStatusTextWithSamplePipelines(status);
        return status;
    }

    private synchronized String updateStatusTextWithSamplePipelines(String status) {
        Set samplePipelines = this.unProcessedPipelineSet.stream().limit(5L).collect(Collectors.toSet());
        if (!samplePipelines.isEmpty()) {
            String samplePipelineText = "Sample pipelines not satisfying the criteria : " + samplePipelines;
            status = status.concat("\n").concat(samplePipelineText);
        }
        return status;
    }
}

