/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.ozone.recon;

import com.google.common.util.concurrent.ThreadFactoryBuilder;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.commons.compress.archivers.tar.TarArchiveEntry;
import org.apache.commons.compress.archivers.tar.TarArchiveInputStream;
import org.apache.commons.io.FileUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TarExtractor {
    private static final Logger LOG = LoggerFactory.getLogger(TarExtractor.class);
    private final AtomicBoolean executorServiceStarted = new AtomicBoolean(false);
    private int threadPoolSize;
    private ExecutorService executor;
    private ThreadFactory threadFactory;

    public TarExtractor(int threadPoolSize, String threadNamePrefix) {
        this.threadPoolSize = threadPoolSize;
        this.threadFactory = new ThreadFactoryBuilder().setNameFormat("FetchOMDBTar-%d" + threadNamePrefix).build();
    }

    public void extractTar(InputStream tarStream, Path outputDir) throws IOException, InterruptedException, ExecutionException {
        String stagingDirName = ".staging_" + UUID.randomUUID();
        Path parentDir = outputDir.getParent();
        if (parentDir == null) {
            parentDir = outputDir;
        }
        Path stagingDir = parentDir.resolve(stagingDirName);
        Files.createDirectories(stagingDir, new FileAttribute[0]);
        ArrayList<Future<Void>> futures = new ArrayList<Future<Void>>();
        try (TarArchiveInputStream tarInput = new TarArchiveInputStream(tarStream);){
            TarArchiveEntry tarArchiveEntry;
            while ((tarArchiveEntry = tarInput.getNextTarEntry()) != null) {
                if (!tarArchiveEntry.isDirectory()) {
                    byte[] fileData = this.readEntryData(tarInput, tarArchiveEntry.getSize());
                    TarArchiveEntry finalEntry = tarArchiveEntry;
                    futures.add(this.executor.submit(() -> {
                        this.writeFile(stagingDir, finalEntry.getName(), fileData);
                        return null;
                    }));
                    continue;
                }
                File dir = new File(stagingDir.toFile(), tarArchiveEntry.getName());
                if (dir.exists() || dir.mkdirs()) continue;
                throw new IOException("Failed to create directory: " + dir);
            }
        }
        for (Future future : futures) {
            future.get();
        }
        if (Files.exists(outputDir, new LinkOption[0])) {
            FileUtils.deleteDirectory((File)outputDir.toFile());
        }
        try {
            Files.move(stagingDir, outputDir, StandardCopyOption.ATOMIC_MOVE, StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            LOG.warn("Atomic move of staging dir : {} to {} failed.", new Object[]{stagingDir, outputDir, e});
        }
        LOG.info("Tar extraction completed and moved from staging to: {}", (Object)outputDir);
    }

    private byte[] readEntryData(TarArchiveInputStream tarInput, long size) throws IOException {
        int bytesRead;
        ByteArrayOutputStream buffer = new ByteArrayOutputStream();
        byte[] fileData = new byte[(int)size];
        while (size > 0L && (bytesRead = tarInput.read(fileData, 0, (int)Math.min((long)fileData.length, size))) != -1) {
            buffer.write(fileData, 0, bytesRead);
            size -= (long)bytesRead;
        }
        return buffer.toByteArray();
    }

    private void writeFile(Path outputDir, String fileName, byte[] fileData) {
        try {
            File outputFile = new File(outputDir.toFile(), fileName);
            Path parentDir = outputFile.toPath().getParent();
            if (parentDir != null && !Files.exists(parentDir, new LinkOption[0])) {
                Files.createDirectories(parentDir, new FileAttribute[0]);
            }
            try (ByteArrayInputStream fis = new ByteArrayInputStream(fileData);
                 OutputStream fos = Files.newOutputStream(outputFile.toPath(), new OpenOption[0]);){
                int bytesRead;
                byte[] buffer = new byte[8192];
                while ((bytesRead = fis.read(buffer)) != -1) {
                    fos.write(buffer, 0, bytesRead);
                }
            }
        }
        catch (IOException e) {
            throw new RuntimeException("Error writing file: " + fileName, e);
        }
    }

    public void start() {
        if (this.executorServiceStarted.compareAndSet(false, true)) {
            this.executor = new ThreadPoolExecutor(0, this.threadPoolSize, 60L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(), this.threadFactory);
        }
    }

    public void stop() {
        if (this.executorServiceStarted.compareAndSet(true, false)) {
            this.executor.shutdown();
            try {
                if (!this.executor.awaitTermination(60L, TimeUnit.SECONDS)) {
                    LOG.warn("Tar Extractor Executor Service did not terminate in time.");
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted during shutdown. Forcing shutdown of Tar Extractor Executor Service...");
                this.executor.shutdownNow();
                Thread.currentThread().interrupt();
            }
        }
    }
}

