/*
 * Decompiled with CFR 0.152.
 */
package org.apache.nifi.remote.util;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.net.Authenticator;
import java.net.InetAddress;
import java.net.InetSocketAddress;
import java.net.PasswordAuthentication;
import java.net.ProxySelector;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.http.HttpClient;
import java.net.http.HttpRequest;
import java.net.http.HttpResponse;
import java.nio.charset.StandardCharsets;
import java.security.cert.Certificate;
import java.security.cert.X509Certificate;
import java.time.Duration;
import java.util.Collection;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.regex.Pattern;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLPeerUnverifiedException;
import javax.net.ssl.SSLSession;
import org.apache.commons.lang3.StringUtils;
import org.apache.nifi.remote.Peer;
import org.apache.nifi.remote.SiteToSiteEventReporter;
import org.apache.nifi.remote.TransferDirection;
import org.apache.nifi.remote.client.http.TransportProtocolVersionNegotiator;
import org.apache.nifi.remote.exception.HandshakeException;
import org.apache.nifi.remote.exception.PortNotRunningException;
import org.apache.nifi.remote.exception.ProtocolException;
import org.apache.nifi.remote.exception.UnknownPortException;
import org.apache.nifi.remote.io.http.HttpCommunicationsSession;
import org.apache.nifi.remote.io.http.HttpInput;
import org.apache.nifi.remote.io.http.HttpOutput;
import org.apache.nifi.remote.protocol.CommunicationsSession;
import org.apache.nifi.remote.protocol.ResponseCode;
import org.apache.nifi.remote.protocol.http.HttpProxy;
import org.apache.nifi.remote.util.ClusterUrlParser;
import org.apache.nifi.remote.util.ExtendTransactionCommand;
import org.apache.nifi.security.cert.StandardPrincipalFormatter;
import org.apache.nifi.stream.io.StreamUtils;
import org.apache.nifi.web.api.dto.ControllerDTO;
import org.apache.nifi.web.api.dto.remote.PeerDTO;
import org.apache.nifi.web.api.entity.ControllerEntity;
import org.apache.nifi.web.api.entity.PeersEntity;
import org.apache.nifi.web.api.entity.TransactionResultEntity;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SiteToSiteRestApiClient
implements Closeable {
    private static final String ACCEPT_HEADER = "Accept";
    private static final String APPLICATION_JSON = "application/json";
    private static final int DATA_PACKET_CHANNEL_READ_BUFFER_SIZE = 16384;
    private static final Logger logger = LoggerFactory.getLogger(SiteToSiteRestApiClient.class);
    private static final ObjectMapper objectMapper = new ObjectMapper().disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
    private String baseUrl;
    protected final SSLContext sslContext;
    protected final HttpProxy proxy;
    private final SiteToSiteEventReporter eventReporter;
    private final HttpClient.Builder httpClientBuilder;
    private HttpClient httpClient;
    private boolean compress = false;
    private long requestExpirationMillis = 0L;
    private int serverTransactionTtl = 0;
    private int batchCount = 0;
    private long batchSize = 0L;
    private long batchDurationMillis = 0L;
    private final TransportProtocolVersionNegotiator transportProtocolVersionNegotiator = new TransportProtocolVersionNegotiator(1);
    private String trustedPeerDn;
    private final ScheduledExecutorService ttlExtendTaskExecutor;
    private ScheduledFuture<?> ttlExtendingFuture;
    private int readTimeoutMillis;
    private long cacheExpirationMillis = 30000L;
    private static final Pattern HTTP_ABS_URL = Pattern.compile("^https?://.+$");
    private CompletableFuture<HttpResponse<String>> transactionFuture;
    private static final ConcurrentMap<String, RemoteGroupContents> contentsMap = new ConcurrentHashMap<String, RemoteGroupContents>();
    private final long lastPruneTimestamp = System.currentTimeMillis();

    public SiteToSiteRestApiClient(SSLContext sslContext, HttpProxy proxy, SiteToSiteEventReporter eventReporter) {
        this.sslContext = sslContext;
        this.proxy = proxy;
        this.eventReporter = eventReporter;
        ThreadFactory threadFactory = Thread.ofVirtual().name(SiteToSiteRestApiClient.class.getSimpleName(), 1L).factory();
        this.ttlExtendTaskExecutor = Executors.newScheduledThreadPool(1, threadFactory);
        this.httpClientBuilder = HttpClient.newBuilder();
        if (sslContext != null) {
            this.httpClientBuilder.sslContext(sslContext);
        }
        if (proxy != null && proxy.getPort() != null) {
            InetSocketAddress proxyAddress = new InetSocketAddress(proxy.getHost(), (int)proxy.getPort());
            ProxySelector proxySelector = ProxySelector.of(proxyAddress);
            this.httpClientBuilder.proxy(proxySelector);
            this.httpClientBuilder.authenticator(this.getProxyAuthenticator());
        }
        this.httpClient = this.httpClientBuilder.build();
    }

    @Override
    public void close() throws IOException {
        this.stopExtendingTransaction();
        this.httpClient.shutdown();
        this.httpClient.close();
    }

    private Authenticator getProxyAuthenticator() {
        final String username = this.proxy.getUsername();
        final char[] password = this.proxy.getPassword().toCharArray();
        return new Authenticator(this){
            private final PasswordAuthentication authentication;
            {
                this.authentication = new PasswordAuthentication(username, password);
            }

            @Override
            protected PasswordAuthentication getPasswordAuthentication() {
                PasswordAuthentication passwordAuthentication = Authenticator.RequestorType.PROXY == this.getRequestorType() ? this.authentication : null;
                return passwordAuthentication;
            }
        };
    }

    public ControllerDTO getController(String clusterUrls) throws IOException {
        return this.getController(ClusterUrlParser.parseClusterUrls(clusterUrls));
    }

    public ControllerDTO getController(Set<String> clusterUrls) throws IOException {
        IOException lastException = new IOException("Get Controller failed");
        for (String clusterUrl : clusterUrls) {
            this.setBaseUrl(ClusterUrlParser.resolveBaseUrl(clusterUrl));
            try {
                return this.getController();
            }
            catch (IOException e) {
                lastException = e;
                logger.warn("Failed to get controller from {}", (Object)clusterUrl, (Object)e);
            }
        }
        if (clusterUrls.size() > 1) {
            throw new IOException("Tried all cluster URLs but none of those was accessible. Last Exception was " + String.valueOf(lastException), lastException);
        }
        throw lastException;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private ControllerDTO getController() throws IOException {
        String internedUrl;
        if (System.currentTimeMillis() > this.lastPruneTimestamp + TimeUnit.MINUTES.toMillis(5L)) {
            this.pruneCache();
        }
        String string = internedUrl = this.baseUrl.intern();
        synchronized (string) {
            RemoteGroupContents groupContents = (RemoteGroupContents)contentsMap.get(internedUrl);
            if (groupContents == null || groupContents.getContents() == null || groupContents.isOlderThan(this.cacheExpirationMillis)) {
                ControllerDTO refreshedContents;
                logger.debug("No Contents for remote group at URL {} or contents have expired; will refresh contents", (Object)internedUrl);
                try {
                    HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(this.getUri("/site-to-site"));
                    refreshedContents = this.send(requestBuilder, ControllerEntity.class).getController();
                }
                catch (Exception e) {
                    ControllerDTO existingController = groupContents == null ? null : groupContents.getContents();
                    RemoteGroupContents updatedContents = new RemoteGroupContents(existingController);
                    contentsMap.put(internedUrl, updatedContents);
                    throw e;
                }
                logger.debug("Successfully retrieved contents for remote group at URL {}", (Object)internedUrl);
                RemoteGroupContents updatedContents = new RemoteGroupContents(refreshedContents);
                contentsMap.put(internedUrl, updatedContents);
                return refreshedContents;
            }
            logger.debug("Contents for remote group at URL {} have already been fetched and have not yet expired. Will return the cached value.", (Object)internedUrl);
            return groupContents.getContents();
        }
    }

    private void pruneCache() {
        for (Map.Entry entry : contentsMap.entrySet()) {
            String url = (String)entry.getKey();
            RemoteGroupContents contents = (RemoteGroupContents)entry.getValue();
            if (!contents.isOlderThan(TimeUnit.MINUTES.toMillis(5L))) continue;
            contentsMap.remove(url, contents);
        }
    }

    public Collection<PeerDTO> getPeers() throws IOException {
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(this.getUri("/site-to-site/peers"));
        return this.send(requestBuilder, PeersEntity.class).getPeers();
    }

    public String initiateTransaction(TransferDirection direction, String portId) throws IOException {
        Optional<String> serverTransactionTtlHeader;
        String transactionUrl;
        HttpResponse<InputStream> response;
        String portType = TransferDirection.RECEIVE.equals((Object)direction) ? "output-ports" : "input-ports";
        logger.debug("initiateTransaction handshaking portType={}, portId={}", (Object)portType, (Object)portId);
        URI uri = this.getUri("/data-transfer/" + portType + "/" + portId + "/transactions");
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(uri).POST(HttpRequest.BodyPublishers.noBody());
        requestBuilder.setHeader(ACCEPT_HEADER, APPLICATION_JSON);
        if (TransferDirection.RECEIVE.equals((Object)direction)) {
            response = this.sendRequest(requestBuilder);
        } else {
            if (this.shouldCheckProxyAuth()) {
                this.getController();
            }
            response = this.sendRequest(requestBuilder);
        }
        int responseCode = response.statusCode();
        logger.debug("initiateTransaction responseCode={}", (Object)responseCode);
        if (responseCode == 201) {
            response.body().close();
            transactionUrl = this.readTransactionUrl(response);
            if (StringUtils.isEmpty((CharSequence)transactionUrl)) {
                throw new ProtocolException("Server returned RESPONSE_CODE_CREATED without Location header");
            }
            Optional<String> transportProtocolVersionHeader = response.headers().firstValue("x-nifi-site-to-site-protocol-version");
            if (transportProtocolVersionHeader.isEmpty()) {
                throw new ProtocolException("Transport Protocol Response Header not found");
            }
            int protocolVersionConfirmedByServer = Integer.parseInt(transportProtocolVersionHeader.get());
            logger.debug("Finished version negotiation, protocolVersionConfirmedByServer={}", (Object)protocolVersionConfirmedByServer);
            this.transportProtocolVersionNegotiator.setVersion(protocolVersionConfirmedByServer);
            serverTransactionTtlHeader = response.headers().firstValue("x-nifi-site-to-site-server-transaction-ttl");
            if (serverTransactionTtlHeader.isEmpty()) {
                throw new ProtocolException("Transaction TTL Response Header not found");
            }
        } else {
            InputStream content = response.body();
            try {
                throw this.handleErrResponse(responseCode, content);
            }
            catch (Throwable throwable) {
                if (content != null) {
                    try {
                        content.close();
                    }
                    catch (Throwable throwable2) {
                        throwable.addSuppressed(throwable2);
                    }
                }
                throw throwable;
            }
        }
        this.serverTransactionTtl = Integer.parseInt(serverTransactionTtlHeader.get());
        logger.debug("initiateTransaction handshaking finished, transactionUrl={}", (Object)transactionUrl);
        return transactionUrl;
    }

    private IOException toIOException(ExecutionException e) {
        Throwable cause = e.getCause();
        if (cause instanceof IOException) {
            return (IOException)cause;
        }
        return new IOException(cause);
    }

    private boolean shouldCheckProxyAuth() {
        return this.proxy != null && StringUtils.isNotEmpty((CharSequence)this.proxy.getUsername());
    }

    public boolean openConnectionForReceive(String transactionUrl, Peer peer) throws IOException {
        URI uri = this.getUri(transactionUrl + "/flow-files");
        HttpCommunicationsSession session = (HttpCommunicationsSession)peer.getCommunicationsSession();
        session.setDataTransferUrl(uri.toString());
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(uri).GET();
        requestBuilder.setHeader(ACCEPT_HEADER, APPLICATION_JSON);
        HttpResponse<InputStream> response = this.sendRequest(requestBuilder);
        int responseCode = response.statusCode();
        switch (responseCode) {
            case 200: {
                logger.debug("Server returned RESPONSE_CODE_OK, indicating there was no data.");
                response.body().close();
                return false;
            }
            case 202: {
                final InputStream httpIn = response.body();
                InputStream streamCapture = new InputStream(){
                    boolean closed = false;

                    @Override
                    public int read() throws IOException {
                        if (this.closed) {
                            return -1;
                        }
                        int r = httpIn.read();
                        if (r < 0) {
                            this.closed = true;
                            logger.debug("Reached to end of input stream. Closing resources...");
                            SiteToSiteRestApiClient.this.stopExtendingTransaction();
                            SiteToSiteRestApiClient.this.closeSilently(httpIn);
                        }
                        return r;
                    }
                };
                ((HttpInput)session.getInput()).setInputStream(streamCapture);
                this.startExtendingTransaction(transactionUrl);
                return true;
            }
        }
        InputStream content = response.body();
        try {
            throw this.handleErrResponse(responseCode, content);
        }
        catch (Throwable throwable) {
            if (content != null) {
                try {
                    content.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
            }
            throw throwable;
        }
    }

    public void openConnectionForSend(String transactionUrl, Peer peer) throws IOException {
        String flowFilesPath = transactionUrl + "/flow-files";
        URI uri = this.getUri(flowFilesPath);
        HttpCommunicationsSession session = (HttpCommunicationsSession)peer.getCommunicationsSession();
        session.setDataTransferUrl(uri.toString());
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(uri);
        requestBuilder.setHeader("Content-Type", "application/octet-stream");
        requestBuilder.setHeader(ACCEPT_HEADER, "text/plain");
        requestBuilder.setHeader("x-nifi-site-to-site-protocol-version", String.valueOf(this.transportProtocolVersionNegotiator.getVersion()));
        this.setRequestHeaders(requestBuilder);
        PipedOutputStream outputStream = new PipedOutputStream();
        PipedInputStream inputStream = new PipedInputStream(outputStream, 16384);
        HttpOutput httpOutput = (HttpOutput)session.getOutput();
        httpOutput.setOutputStream(outputStream);
        requestBuilder.POST(HttpRequest.BodyPublishers.ofInputStream(() -> inputStream));
        HttpRequest request = requestBuilder.build();
        this.transactionFuture = this.httpClient.sendAsync(request, HttpResponse.BodyHandlers.ofString());
        this.startExtendingTransaction(transactionUrl);
    }

    public void finishTransferFlowFiles(CommunicationsSession commSession) throws IOException {
        HttpResponse<String> response;
        if (this.transactionFuture == null) {
            throw new IllegalStateException("Data Transfer not started");
        }
        commSession.getOutput().getOutputStream().close();
        this.stopExtendingTransaction();
        try {
            response = this.transactionFuture.get(this.readTimeoutMillis, TimeUnit.MILLISECONDS);
        }
        catch (ExecutionException e) {
            throw this.toIOException(e);
        }
        catch (InterruptedException | TimeoutException e) {
            throw new IOException(e);
        }
        int responseCode = response.statusCode();
        if (responseCode != 202) {
            ByteArrayInputStream content = new ByteArrayInputStream(response.body().getBytes(StandardCharsets.UTF_8));
            try {
                throw this.handleErrResponse(responseCode, content);
            }
            catch (Throwable throwable) {
                try {
                    ((InputStream)content).close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
        }
        String receivedChecksum = response.body();
        ((HttpInput)commSession.getInput()).setInputStream(new ByteArrayInputStream(receivedChecksum.getBytes()));
        ((HttpCommunicationsSession)commSession).setChecksum(receivedChecksum);
        logger.debug("receivedChecksum={}", (Object)receivedChecksum);
    }

    private void startExtendingTransaction(String transactionUrl) {
        if (this.ttlExtendingFuture == null) {
            int extendFrequency = this.serverTransactionTtl / 2;
            logger.debug("Extend Transaction Started [{}] Frequency [{} seconds]", (Object)transactionUrl, (Object)extendFrequency);
            ExtendTransactionCommand command = new ExtendTransactionCommand(this, transactionUrl, this.eventReporter);
            this.ttlExtendingFuture = this.ttlExtendTaskExecutor.scheduleWithFixedDelay(command, extendFrequency, extendFrequency, TimeUnit.SECONDS);
        }
    }

    private void closeSilently(Closeable closeable) {
        try {
            if (closeable != null) {
                closeable.close();
            }
        }
        catch (IOException e) {
            logger.debug("Failed close [{}]", (Object)closeable, (Object)e);
        }
    }

    public TransactionResultEntity extendTransaction(String transactionUrl) throws IOException {
        logger.debug("Sending extendTransaction request to transactionUrl: {}", (Object)transactionUrl);
        URI uri = this.getUri(transactionUrl);
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(uri).PUT(HttpRequest.BodyPublishers.noBody());
        return this.send(requestBuilder, TransactionResultEntity.class);
    }

    private void stopExtendingTransaction() {
        if (!this.ttlExtendTaskExecutor.isShutdown()) {
            this.ttlExtendTaskExecutor.shutdown();
        }
        if (this.ttlExtendingFuture != null && !this.ttlExtendingFuture.isCancelled()) {
            boolean cancelled = this.ttlExtendingFuture.cancel(true);
            logger.debug("Extend Transaction Cancelled [{}]", (Object)cancelled);
        }
    }

    private IOException handleErrResponse(int responseCode, InputStream in) throws IOException {
        TransactionResultEntity errEntity = this.readResponse(in);
        ResponseCode errCode = ResponseCode.fromCode(errEntity.getResponseCode());
        return switch (errCode) {
            case ResponseCode.UNKNOWN_PORT -> new UnknownPortException(errEntity.getMessage());
            case ResponseCode.PORT_NOT_IN_VALID_STATE -> new PortNotRunningException(errEntity.getMessage());
            default -> responseCode == 403 ? new HandshakeException(errEntity.getMessage()) : new IOException("Unexpected response code: " + responseCode + " errCode:" + String.valueOf((Object)errCode) + " errMessage:" + errEntity.getMessage());
        };
    }

    private TransactionResultEntity readResponse(InputStream inputStream) throws IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        StreamUtils.copy((InputStream)inputStream, (OutputStream)bos);
        String responseMessage = null;
        try {
            responseMessage = bos.toString(StandardCharsets.UTF_8);
            return (TransactionResultEntity)objectMapper.readValue(responseMessage, TransactionResultEntity.class);
        }
        catch (JsonParseException | JsonMappingException e) {
            TransactionResultEntity entity = new TransactionResultEntity();
            entity.setResponseCode(ResponseCode.ABORT.getCode());
            entity.setMessage(responseMessage);
            return entity;
        }
    }

    private String readTransactionUrl(HttpResponse<InputStream> response) {
        Optional<String> locationUriIntentHeader = response.headers().firstValue("x-location-uri-intent");
        logger.debug("locationUriIntentHeader={}", locationUriIntentHeader);
        if (locationUriIntentHeader.isPresent() && "transaction-url".equals(locationUriIntentHeader.get())) {
            Optional<String> transactionUrl = response.headers().firstValue("Location");
            logger.debug("transactionUrl={}", transactionUrl);
            if (transactionUrl.isPresent()) {
                return transactionUrl.get();
            }
        }
        return null;
    }

    private void setRequestHeaders(HttpRequest.Builder requestBuilder) {
        if (this.compress) {
            requestBuilder.setHeader("x-nifi-site-to-site-use-compression", Boolean.TRUE.toString());
        }
        if (this.requestExpirationMillis > 0L) {
            requestBuilder.setHeader("x-nifi-site-to-site-request-expiration", String.valueOf(this.requestExpirationMillis));
        }
        if (this.batchCount > 0) {
            requestBuilder.setHeader("x-nifi-site-to-site-batch-count", String.valueOf(this.batchCount));
        }
        if (this.batchSize > 0L) {
            requestBuilder.setHeader("x-nifi-site-to-site-batch-size", String.valueOf(this.batchSize));
        }
        if (this.batchDurationMillis > 0L) {
            requestBuilder.setHeader("x-nifi-site-to-site-batch-duration", String.valueOf(this.batchDurationMillis));
        }
    }

    private URI getUri(String path) {
        URI url;
        try {
            if (HTTP_ABS_URL.matcher(path).find()) {
                url = new URI(path);
            } else {
                if (StringUtils.isEmpty((CharSequence)this.getBaseUrl())) {
                    throw new IllegalStateException("API baseUrl is not resolved yet, call setBaseUrl or resolveBaseUrl before sending requests with relative path.");
                }
                url = new URI(this.baseUrl + path);
            }
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException(e.getMessage());
        }
        return url;
    }

    public String getBaseUrl() {
        return this.baseUrl;
    }

    public void setBaseUrl(String baseUrl) {
        this.baseUrl = baseUrl;
    }

    public void setConnectTimeoutMillis(int connectTimeoutMillis) {
        this.httpClientBuilder.connectTimeout(Duration.ofMillis(connectTimeoutMillis));
        this.httpClient.close();
        this.httpClient = this.httpClientBuilder.build();
    }

    public void setReadTimeoutMillis(int readTimeoutMillis) {
        this.readTimeoutMillis = readTimeoutMillis;
    }

    public void setCacheExpirationMillis(long expirationMillis) {
        this.cacheExpirationMillis = expirationMillis;
    }

    public void setBaseUrl(String scheme, String host, int port) {
        String baseUri;
        try {
            URI uri = new URI(scheme, null, host, port, "/nifi-api", null, null);
            baseUri = uri.toString();
        }
        catch (URISyntaxException e) {
            throw new IllegalArgumentException(e);
        }
        this.setBaseUrl(baseUri);
    }

    public void setCompress(boolean compress) {
        this.compress = compress;
    }

    public void setLocalAddress(InetAddress localAddress) {
        this.httpClientBuilder.localAddress(localAddress);
        this.httpClient.close();
        this.httpClient = this.httpClientBuilder.build();
    }

    public void setRequestExpirationMillis(long requestExpirationMillis) {
        if (requestExpirationMillis < 0L) {
            throw new IllegalArgumentException("requestExpirationMillis can't be a negative value.");
        }
        this.requestExpirationMillis = requestExpirationMillis;
    }

    public void setBatchCount(int batchCount) {
        if (batchCount < 0) {
            throw new IllegalArgumentException("batchCount can't be a negative value.");
        }
        this.batchCount = batchCount;
    }

    public void setBatchSize(long batchSize) {
        if (batchSize < 0L) {
            throw new IllegalArgumentException("batchSize can't be a negative value.");
        }
        this.batchSize = batchSize;
    }

    public void setBatchDurationMillis(long batchDurationMillis) {
        if (batchDurationMillis < 0L) {
            throw new IllegalArgumentException("batchDurationMillis can't be a negative value.");
        }
        this.batchDurationMillis = batchDurationMillis;
    }

    public Integer getTransactionProtocolVersion() {
        return this.transportProtocolVersionNegotiator.getTransactionProtocolVersion();
    }

    public String getTrustedPeerDn() {
        return this.trustedPeerDn;
    }

    public TransactionResultEntity commitReceivingFlowFiles(String transactionUrl, ResponseCode clientResponse, String checksum) throws IOException {
        logger.debug("Sending commitReceivingFlowFiles request to transactionUrl: {}, clientResponse={}, checksum={}", new Object[]{transactionUrl, clientResponse, checksum});
        this.stopExtendingTransaction();
        StringBuilder urlBuilder = new StringBuilder(transactionUrl).append("?responseCode=").append(clientResponse.getCode());
        if (ResponseCode.CONFIRM_TRANSACTION.equals((Object)clientResponse)) {
            urlBuilder.append("&checksum=").append(checksum);
        }
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(this.getUri(urlBuilder.toString())).DELETE();
        requestBuilder.setHeader(ACCEPT_HEADER, APPLICATION_JSON);
        HttpResponse<InputStream> response = this.sendRequest(requestBuilder);
        int responseCode = response.statusCode();
        try (InputStream content = response.body();){
            switch (responseCode) {
                case 200: 
                case 400: {
                    break;
                }
                default: {
                    throw this.handleErrResponse(responseCode, content);
                }
            }
            TransactionResultEntity transactionResultEntity = this.readResponse(content);
            return transactionResultEntity;
        }
    }

    public TransactionResultEntity commitTransferFlowFiles(String transactionUrl, ResponseCode clientResponse) throws IOException {
        String requestUrl = transactionUrl + "?responseCode=" + clientResponse.getCode();
        logger.debug("Sending commitTransferFlowFiles request to transactionUrl: {}", (Object)requestUrl);
        HttpRequest.Builder requestBuilder = HttpRequest.newBuilder(this.getUri(requestUrl)).DELETE();
        requestBuilder.setHeader(ACCEPT_HEADER, APPLICATION_JSON);
        HttpResponse<InputStream> response = this.sendRequest(requestBuilder);
        int responseCode = response.statusCode();
        try (InputStream content = response.body();){
            switch (responseCode) {
                case 200: 
                case 400: {
                    break;
                }
                default: {
                    throw this.handleErrResponse(responseCode, content);
                }
            }
            TransactionResultEntity transactionResultEntity = this.readResponse(content);
            return transactionResultEntity;
        }
    }

    private <T> T send(HttpRequest.Builder requestBuilder, Class<T> responseClass) throws IOException {
        requestBuilder.setHeader(ACCEPT_HEADER, APPLICATION_JSON);
        HttpResponse<InputStream> response = this.sendRequest(requestBuilder);
        int statusCode = response.statusCode();
        try (InputStream inputStream = response.body();){
            if (200 == statusCode) {
                Object object = objectMapper.readValue(inputStream, responseClass);
                return (T)object;
            }
            throw new IOException("Request URI [%s] HTTP %d".formatted(response.uri(), statusCode));
        }
    }

    private HttpResponse<InputStream> sendRequest(HttpRequest.Builder requestBuilder) throws IOException {
        this.setRequestHeaders(requestBuilder);
        requestBuilder.setHeader("x-nifi-site-to-site-protocol-version", String.valueOf(this.transportProtocolVersionNegotiator.getVersion()));
        requestBuilder.timeout(Duration.ofMillis(this.readTimeoutMillis));
        HttpRequest request = requestBuilder.build();
        try {
            HttpResponse<InputStream> response = this.httpClient.send(request, HttpResponse.BodyHandlers.ofInputStream());
            Optional<SSLSession> sslSessionFound = response.sslSession();
            sslSessionFound.ifPresent(this::setTrustedPeerDn);
            return response;
        }
        catch (InterruptedException e) {
            throw new IOException("Request URI [%s] interrupted".formatted(request.uri()), e);
        }
    }

    private void setTrustedPeerDn(SSLSession sslSession) {
        try {
            Certificate[] peerCertificates = sslSession.getPeerCertificates();
            if (peerCertificates.length == 0) {
                logger.info("Peer Certificates not found");
            } else {
                X509Certificate peerCertificate = (X509Certificate)peerCertificates[0];
                this.trustedPeerDn = StandardPrincipalFormatter.getInstance().getSubject(peerCertificate);
            }
        }
        catch (SSLPeerUnverifiedException e) {
            logger.warn("Peer Certificate verification failed", (Throwable)e);
        }
    }

    private static class RemoteGroupContents {
        private final ControllerDTO contents;
        private final long timestamp;

        public RemoteGroupContents(ControllerDTO contents) {
            this.contents = contents;
            this.timestamp = System.currentTimeMillis();
        }

        public ControllerDTO getContents() {
            return this.contents;
        }

        public boolean isOlderThan(long millis) {
            long millisSinceRefresh = System.currentTimeMillis() - this.timestamp;
            return millisSinceRefresh > millis;
        }
    }
}

