/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.streams.processor.internals.tasks;

import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import org.apache.kafka.common.KafkaFuture;
import org.apache.kafka.common.internals.KafkaFutureImpl;
import org.apache.kafka.common.utils.LogContext;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.streams.errors.StreamsException;
import org.apache.kafka.streams.processor.TaskId;
import org.apache.kafka.streams.processor.internals.ReadOnlyTask;
import org.apache.kafka.streams.processor.internals.StreamTask;
import org.apache.kafka.streams.processor.internals.Task;
import org.apache.kafka.streams.processor.internals.TaskExecutionMetadata;
import org.apache.kafka.streams.processor.internals.TasksRegistry;
import org.apache.kafka.streams.processor.internals.tasks.DefaultTaskExecutor;
import org.apache.kafka.streams.processor.internals.tasks.TaskExecutor;
import org.apache.kafka.streams.processor.internals.tasks.TaskExecutorCreator;
import org.apache.kafka.streams.processor.internals.tasks.TaskManager;
import org.slf4j.Logger;

public class DefaultTaskManager
implements TaskManager {
    private final Time time;
    private final Logger log;
    private final TasksRegistry tasks;
    private final Lock tasksLock = new ReentrantLock();
    private final Condition tasksCondition = this.tasksLock.newCondition();
    private final List<TaskId> lockedTasks = new ArrayList<TaskId>();
    private final Map<TaskId, StreamsException> uncaughtExceptions = new HashMap<TaskId, StreamsException>();
    private final Map<TaskId, TaskExecutor> assignedTasks = new HashMap<TaskId, TaskExecutor>();
    private final TaskExecutionMetadata taskExecutionMetadata;
    private final List<TaskExecutor> taskExecutors;

    public DefaultTaskManager(Time time, String clientId, TasksRegistry tasks, TaskExecutorCreator executorCreator, TaskExecutionMetadata taskExecutionMetadata, int numExecutors) {
        String logPrefix = String.format("%s ", clientId);
        LogContext logContext = new LogContext(logPrefix);
        this.log = logContext.logger(DefaultTaskManager.class);
        this.time = time;
        this.tasks = tasks;
        this.taskExecutionMetadata = taskExecutionMetadata;
        this.taskExecutors = new ArrayList<TaskExecutor>(numExecutors);
        for (int i = 1; i <= numExecutors; ++i) {
            String name = clientId + "-TaskExecutor-" + i;
            this.taskExecutors.add(executorCreator.create(this, name, time, taskExecutionMetadata));
        }
    }

    @Override
    public StreamTask assignNextTask(TaskExecutor executor) {
        return this.returnWithTasksLocked(() -> {
            if (!this.taskExecutors.contains(executor)) {
                throw new IllegalArgumentException("The requested executor for getting next task to assign is unrecognized");
            }
            for (Task task : this.tasks.activeTasks()) {
                if (this.assignedTasks.containsKey(task.id()) || this.lockedTasks.contains(task.id()) || !this.canProgress((StreamTask)task, this.time.milliseconds()) || this.hasUncaughtException(task.id())) continue;
                this.assignedTasks.put(task.id(), executor);
                this.log.debug("Assigned task {} to executor {}", (Object)task.id(), (Object)executor.name());
                return (StreamTask)task;
            }
            this.log.debug("Found no assignable task for executor {}", (Object)executor.name());
            return null;
        });
    }

    @Override
    public void awaitProcessableTasks() throws InterruptedException {
        boolean interrupted = this.returnWithTasksLocked(() -> {
            for (Task task : this.tasks.activeTasks()) {
                if (this.assignedTasks.containsKey(task.id()) || this.lockedTasks.contains(task.id()) || !this.canProgress((StreamTask)task, this.time.milliseconds()) || this.hasUncaughtException(task.id())) continue;
                this.log.debug("Await unblocked: returning early from await since a processable task {} was found", (Object)task.id());
                return false;
            }
            try {
                this.log.debug("Await blocking");
                this.tasksCondition.await();
            }
            catch (InterruptedException ignored) {
                this.log.debug("Await unblocked: Interrupted while waiting for processable tasks");
                return true;
            }
            this.log.debug("Await unblocked: Woken up to check for processable tasks");
            return false;
        });
        if (interrupted) {
            throw new InterruptedException();
        }
    }

    @Override
    public void signalTaskExecutors() {
        this.log.debug("Waking up task executors");
        this.executeWithTasksLocked(this.tasksCondition::signalAll);
    }

    @Override
    public void unassignTask(StreamTask task, TaskExecutor executor) {
        this.executeWithTasksLocked(() -> {
            if (!this.taskExecutors.contains(executor)) {
                throw new IllegalArgumentException("The requested executor for unassign task is unrecognized");
            }
            TaskExecutor lockedExecutor = this.assignedTasks.get(task.id());
            if (lockedExecutor == null || lockedExecutor != executor) {
                throw new IllegalArgumentException("Task " + task.id() + " is not locked by the executor");
            }
            this.assignedTasks.remove(task.id());
            this.log.debug("Unassigned {} from executor {}", (Object)task.id(), (Object)executor.name());
            this.tasksCondition.signalAll();
        });
    }

    @Override
    public KafkaFuture<Void> lockTasks(Set<TaskId> taskIds) {
        KafkaFutureImpl result = new KafkaFutureImpl();
        if (taskIds.isEmpty()) {
            result.complete(null);
            return result;
        }
        return (KafkaFuture)this.returnWithTasksLocked(() -> {
            this.lockedTasks.addAll(taskIds);
            ConcurrentSkipListSet remainingTaskIds = new ConcurrentSkipListSet(taskIds);
            for (TaskId taskId : taskIds) {
                Task task = this.tasks.task(taskId);
                if (task == null) {
                    throw new IllegalArgumentException("Trying to lock task " + taskId + " but it's not owned");
                }
                if (!task.isActive()) {
                    throw new IllegalArgumentException("The locking task " + taskId + " is not an active task");
                }
                if (this.assignedTasks.containsKey(taskId)) {
                    TaskExecutor executor = this.assignedTasks.get(taskId);
                    this.log.debug("Requesting release of task {} from {}", (Object)taskId, (Object)executor.name());
                    KafkaFuture<StreamTask> future = executor.unassign();
                    future.whenComplete((streamTask, throwable) -> {
                        if (throwable != null) {
                            result.completeExceptionally(throwable);
                        } else {
                            assert (!this.assignedTasks.containsKey(taskId));
                            assert (streamTask == null || streamTask.id() == taskId);
                            remainingTaskIds.remove(taskId);
                            if (remainingTaskIds.isEmpty()) {
                                result.complete(null);
                            }
                        }
                    });
                    continue;
                }
                remainingTaskIds.remove(taskId);
                if (!remainingTaskIds.isEmpty()) continue;
                result.complete(null);
            }
            return result;
        });
    }

    @Override
    public KafkaFuture<Void> lockAllTasks() {
        return this.returnWithTasksLocked(() -> this.lockTasks(this.tasks.activeTasks().stream().map(Task::id).collect(Collectors.toSet())));
    }

    @Override
    public void unlockTasks(Set<TaskId> taskIds) {
        if (taskIds.isEmpty()) {
            return;
        }
        this.executeWithTasksLocked(() -> {
            this.lockedTasks.removeAll(taskIds);
            this.log.debug("Waking up task executors");
            this.tasksCondition.signalAll();
        });
    }

    @Override
    public void unlockAllTasks() {
        this.executeWithTasksLocked(() -> this.unlockTasks(this.tasks.activeTasks().stream().map(Task::id).collect(Collectors.toSet())));
    }

    @Override
    public void add(Set<StreamTask> tasksToAdd) {
        this.executeWithTasksLocked(() -> {
            for (StreamTask task : tasksToAdd) {
                this.tasks.addTask(task);
            }
            this.log.debug("Waking up task executors");
            this.tasksCondition.signalAll();
        });
        this.log.info("Added tasks {} to the task manager to process", tasksToAdd);
    }

    @Override
    public void remove(TaskId taskId) {
        this.executeWithTasksLocked(() -> {
            if (this.assignedTasks.containsKey(taskId)) {
                throw new IllegalArgumentException("The task to remove is still assigned to executors");
            }
            if (!this.lockedTasks.contains(taskId)) {
                throw new IllegalArgumentException("The task to remove is not locked yet by the task manager");
            }
            if (!this.tasks.contains(taskId)) {
                throw new IllegalArgumentException("The task to remove is not owned by the task manager");
            }
            this.tasks.removeTask(this.tasks.task(taskId));
        });
        this.log.info("Removed task {} from the task manager", (Object)taskId);
    }

    @Override
    public Set<ReadOnlyTask> getTasks() {
        return this.returnWithTasksLocked(() -> this.tasks.activeTasks().stream().map(ReadOnlyTask::new).collect(Collectors.toSet()));
    }

    @Override
    public void setUncaughtException(StreamsException exception, TaskId taskId) {
        this.executeWithTasksLocked(() -> {
            if (!this.assignedTasks.containsKey(taskId)) {
                throw new IllegalArgumentException("An uncaught exception can only be set as long as the task is still assigned");
            }
            if (this.uncaughtExceptions.containsKey(taskId)) {
                throw new IllegalArgumentException("The uncaught exception must be cleared before restarting processing");
            }
            this.uncaughtExceptions.put(taskId, exception);
        });
        this.log.info("Set an uncaught exception of type {} for task {}, with error message: {}", new Object[]{((Object)((Object)exception)).getClass().getName(), taskId, exception.getMessage()});
    }

    @Override
    public Map<TaskId, RuntimeException> drainUncaughtExceptions() {
        Map returnValue = this.returnWithTasksLocked(() -> {
            HashMap<TaskId, StreamsException> result = new HashMap<TaskId, StreamsException>(this.uncaughtExceptions);
            this.uncaughtExceptions.clear();
            return result;
        });
        if (!returnValue.isEmpty()) {
            this.log.debug("Drained {} uncaught exceptions", (Object)returnValue.size());
        }
        return returnValue;
    }

    @Override
    public boolean hasUncaughtException(TaskId taskId) {
        return this.returnWithTasksLocked(() -> this.uncaughtExceptions.containsKey(taskId));
    }

    private void executeWithTasksLocked(Runnable action) {
        this.tasksLock.lock();
        try {
            action.run();
        }
        finally {
            this.tasksLock.unlock();
        }
    }

    private <T> T returnWithTasksLocked(Supplier<T> action) {
        this.tasksLock.lock();
        try {
            T t = action.get();
            return t;
        }
        finally {
            this.tasksLock.unlock();
        }
    }

    private boolean canProgress(StreamTask task, long nowMs) {
        return this.taskExecutionMetadata.canProcessTask(task, nowMs) && task.isProcessable(nowMs) || this.taskExecutionMetadata.canPunctuateTask(task) && (task.canPunctuateStreamTime() || task.canPunctuateSystemTime());
    }

    @Override
    public void startTaskExecutors() {
        for (TaskExecutor t : this.taskExecutors) {
            t.start();
        }
    }

    @Override
    public void shutdown(Duration duration) {
        for (TaskExecutor t : this.taskExecutors) {
            t.requestShutdown();
        }
        this.signalTaskExecutors();
        for (TaskExecutor t : this.taskExecutors) {
            t.awaitShutdown(duration);
        }
    }

    public static class DefaultTaskExecutorCreator
    implements TaskExecutorCreator {
        @Override
        public TaskExecutor create(TaskManager taskManager, String name, Time time, TaskExecutionMetadata taskExecutionMetadata) {
            return new DefaultTaskExecutor(taskManager, name, time, taskExecutionMetadata);
        }
    }
}

