/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.storage.internals.log;

import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Policy;
import java.io.ByteArrayInputStream;
import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.nio.file.attribute.FileAttribute;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.function.Function;
import java.util.stream.Stream;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.Uuid;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.common.utils.Utils;
import org.apache.kafka.server.log.remote.storage.RemoteLogSegmentMetadata;
import org.apache.kafka.server.log.remote.storage.RemoteResourceNotFoundException;
import org.apache.kafka.server.log.remote.storage.RemoteStorageException;
import org.apache.kafka.server.log.remote.storage.RemoteStorageManager;
import org.apache.kafka.server.util.ShutdownableThread;
import org.apache.kafka.storage.internals.log.CorruptIndexException;
import org.apache.kafka.storage.internals.log.OffsetIndex;
import org.apache.kafka.storage.internals.log.OffsetPosition;
import org.apache.kafka.storage.internals.log.StorageAction;
import org.apache.kafka.storage.internals.log.TimeIndex;
import org.apache.kafka.storage.internals.log.TimestampOffset;
import org.apache.kafka.storage.internals.log.TransactionIndex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RemoteIndexCache
implements Closeable {
    private static final Logger log = LoggerFactory.getLogger(RemoteIndexCache.class);
    private static final String TMP_FILE_SUFFIX = ".tmp";
    public static final String REMOTE_LOG_INDEX_CACHE_CLEANER_THREAD = "remote-log-index-cleaner";
    public static final String DIR_NAME = "remote-log-index-cache";
    private final File cacheDir;
    private final AtomicBoolean isRemoteIndexCacheClosed = new AtomicBoolean(false);
    private final LinkedBlockingQueue<Entry> expiredIndexes = new LinkedBlockingQueue();
    private final ReentrantReadWriteLock lock = new ReentrantReadWriteLock();
    private final RemoteStorageManager remoteStorageManager;
    private final ShutdownableThread cleanerThread;
    private Cache<Uuid, Entry> internalCache;

    public RemoteIndexCache(long maxSize, RemoteStorageManager remoteStorageManager, String logDir) throws IOException {
        this.remoteStorageManager = remoteStorageManager;
        this.cacheDir = new File(logDir, DIR_NAME);
        this.internalCache = this.initEmptyCache(maxSize);
        this.init();
        this.cleanerThread = this.createCleanerThread();
        this.cleanerThread.start();
    }

    public void resizeCacheSize(long remoteLogIndexFileCacheSize) {
        this.lock.writeLock().lock();
        try {
            ((Policy.Eviction)this.internalCache.policy().eviction().orElseThrow(() -> new NoSuchElementException("No eviction policy is set for the remote index cache."))).setMaximum(remoteLogIndexFileCacheSize);
        }
        finally {
            this.lock.writeLock().unlock();
        }
    }

    private Cache<Uuid, Entry> initEmptyCache(long maxSize) {
        return Caffeine.newBuilder().maximumWeight(maxSize).weigher((key, entry) -> (int)((Entry)entry).entrySizeBytes).evictionListener((key, entry, cause) -> {
            if (entry != null) {
                this.enqueueEntryForCleanup((Entry)entry, (Uuid)key);
            } else {
                log.error("Received entry as null for key {} when the it is removed from the cache.", key);
            }
        }).build();
    }

    public Collection<Entry> expiredIndexes() {
        return Collections.unmodifiableCollection(this.expiredIndexes);
    }

    public Cache<Uuid, Entry> internalCache() {
        return this.internalCache;
    }

    public File cacheDir() {
        return this.cacheDir;
    }

    public void remove(Uuid key) {
        this.lock.readLock().lock();
        try {
            this.internalCache.asMap().computeIfPresent(key, (k, v) -> {
                this.enqueueEntryForCleanup((Entry)v, (Uuid)k);
                return null;
            });
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    public void removeAll(Collection<Uuid> keys) {
        this.lock.readLock().lock();
        try {
            keys.forEach(key -> this.internalCache.asMap().computeIfPresent(key, (k, v) -> {
                this.enqueueEntryForCleanup((Entry)v, (Uuid)k);
                return null;
            }));
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private void enqueueEntryForCleanup(Entry entry, Uuid key) {
        try {
            entry.markForCleanup();
            if (!this.expiredIndexes.offer(entry)) {
                log.error("Error while inserting entry {} for key {} into the cleaner queue because queue is full.", (Object)entry, (Object)key);
            }
        }
        catch (IOException e) {
            throw new KafkaException((Throwable)e);
        }
    }

    public ShutdownableThread cleanerThread() {
        return this.cleanerThread;
    }

    private ShutdownableThread createCleanerThread() {
        ShutdownableThread thread = new ShutdownableThread(REMOTE_LOG_INDEX_CACHE_CLEANER_THREAD){

            public void doWork() {
                try {
                    Entry entry = (Entry)RemoteIndexCache.this.expiredIndexes.take();
                    this.log.debug("Cleaning up index entry {}", (Object)entry);
                    entry.cleanup();
                }
                catch (InterruptedException ie) {
                    if (!RemoteIndexCache.this.isRemoteIndexCacheClosed.get()) {
                        this.log.error("Cleaner thread received interruption but remote index cache is not closed", (Throwable)ie);
                        throw new KafkaException((Throwable)ie);
                    }
                    this.log.debug("Cleaner thread was interrupted on cache shutdown");
                }
                catch (Exception ex) {
                    this.log.error("Error occurred while cleaning up expired entry", (Throwable)ex);
                }
            }
        };
        thread.setDaemon(true);
        return thread;
    }

    private void init() throws IOException {
        long start = Time.SYSTEM.hiResClockMs();
        try {
            Files.createDirectory(this.cacheDir.toPath(), new FileAttribute[0]);
            log.info("Created new file {} for RemoteIndexCache", (Object)this.cacheDir);
        }
        catch (FileAlreadyExistsException e) {
            log.info("RemoteIndexCache directory {} already exists. Re-using the same directory.", (Object)this.cacheDir);
        }
        catch (Exception e) {
            log.error("Unable to create directory {} for RemoteIndexCache.", (Object)this.cacheDir, (Object)e);
            throw new KafkaException((Throwable)e);
        }
        try (Stream<Path> paths = Files.list(this.cacheDir.toPath());){
            paths.forEach(path -> {
                if (path.endsWith(".deleted") || path.endsWith(TMP_FILE_SUFFIX)) {
                    try {
                        if (Files.deleteIfExists(path)) {
                            log.debug("Deleted file path {} on cache initialization", path);
                        }
                    }
                    catch (IOException e) {
                        throw new KafkaException((Throwable)e);
                    }
                }
            });
        }
        paths = Files.list(this.cacheDir.toPath());
        var4_5 = null;
        try {
            Iterator iterator = paths.iterator();
            while (iterator.hasNext()) {
                Path path2 = (Path)iterator.next();
                Path fileNamePath = path2.getFileName();
                if (fileNamePath == null) {
                    throw new KafkaException("Empty file name in remote index cache directory: " + this.cacheDir);
                }
                String indexFileName = fileNamePath.toString();
                Uuid uuid = RemoteIndexCache.remoteLogSegmentIdFromRemoteIndexFileName(indexFileName);
                if (this.internalCache.asMap().containsKey(uuid)) continue;
                String fileNameWithoutSuffix = indexFileName.substring(0, indexFileName.indexOf("."));
                File offsetIndexFile = new File(this.cacheDir, fileNameWithoutSuffix + ".index");
                File timestampIndexFile = new File(this.cacheDir, fileNameWithoutSuffix + ".timeindex");
                File txnIndexFile = new File(this.cacheDir, fileNameWithoutSuffix + ".txnindex");
                if (Files.exists(offsetIndexFile.toPath(), new LinkOption[0]) && Files.exists(timestampIndexFile.toPath(), new LinkOption[0]) && Files.exists(txnIndexFile.toPath(), new LinkOption[0])) {
                    long offset = RemoteIndexCache.offsetFromRemoteIndexFileName(indexFileName);
                    OffsetIndex offsetIndex = new OffsetIndex(offsetIndexFile, offset, Integer.MAX_VALUE, false);
                    offsetIndex.sanityCheck();
                    TimeIndex timeIndex = new TimeIndex(timestampIndexFile, offset, Integer.MAX_VALUE, false);
                    timeIndex.sanityCheck();
                    TransactionIndex txnIndex = new TransactionIndex(offset, txnIndexFile);
                    txnIndex.sanityCheck();
                    Entry entry = new Entry(offsetIndex, timeIndex, txnIndex);
                    this.internalCache.put((Object)uuid, (Object)entry);
                    continue;
                }
                RemoteIndexCache.tryAll(Arrays.asList(() -> {
                    Files.deleteIfExists(offsetIndexFile.toPath());
                    return null;
                }, () -> {
                    Files.deleteIfExists(timestampIndexFile.toPath());
                    return null;
                }, () -> {
                    Files.deleteIfExists(txnIndexFile.toPath());
                    return null;
                }));
            }
        }
        catch (Throwable throwable) {
            var4_5 = throwable;
            throw throwable;
        }
        finally {
            if (paths != null) {
                if (var4_5 != null) {
                    try {
                        paths.close();
                    }
                    catch (Throwable throwable) {
                        var4_5.addSuppressed(throwable);
                    }
                } else {
                    paths.close();
                }
            }
        }
        log.info("RemoteIndexCache starts up in {} ms.", (Object)(Time.SYSTEM.hiResClockMs() - start));
    }

    private <T> T loadIndexFile(File file, RemoteLogSegmentMetadata remoteLogSegmentMetadata, Function<RemoteLogSegmentMetadata, InputStream> fetchRemoteIndex, Function<File, T> readIndex) throws IOException {
        File indexFile = new File(this.cacheDir, file.getName());
        T index = null;
        if (Files.exists(indexFile.toPath(), new LinkOption[0])) {
            try {
                index = readIndex.apply(indexFile);
            }
            catch (CorruptIndexException ex) {
                log.info("Error occurred while loading the stored index file {}", (Object)indexFile.getPath(), (Object)ex);
            }
        }
        if (index == null) {
            File tmpIndexFile = new File(indexFile.getParentFile(), indexFile.getName() + TMP_FILE_SUFFIX);
            try (InputStream inputStream = fetchRemoteIndex.apply(remoteLogSegmentMetadata);){
                Files.copy(inputStream, tmpIndexFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
            }
            Utils.atomicMoveWithFallback((Path)tmpIndexFile.toPath(), (Path)indexFile.toPath(), (boolean)false);
            index = readIndex.apply(indexFile);
        }
        return index;
    }

    public Entry getIndexEntry(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        if (this.isRemoteIndexCacheClosed.get()) {
            throw new IllegalStateException("Unable to fetch index for segment id=" + remoteLogSegmentMetadata.remoteLogSegmentId().id() + ". Instance is already closed.");
        }
        this.lock.readLock().lock();
        try {
            if (this.isRemoteIndexCacheClosed.get()) {
                throw new IllegalStateException("Unable to fetch index for segment-id = " + remoteLogSegmentMetadata.remoteLogSegmentId().id() + ". Index instance is already closed.");
            }
            Entry entry = (Entry)this.internalCache.get((Object)remoteLogSegmentMetadata.remoteLogSegmentId().id(), uuid -> this.createCacheEntry(remoteLogSegmentMetadata));
            return entry;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    private Entry createCacheEntry(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        long startOffset = remoteLogSegmentMetadata.startOffset();
        try {
            File offsetIndexFile = RemoteIndexCache.remoteOffsetIndexFile(this.cacheDir, remoteLogSegmentMetadata);
            OffsetIndex offsetIndex = this.loadIndexFile(offsetIndexFile, remoteLogSegmentMetadata, rlsMetadata -> {
                try {
                    return this.remoteStorageManager.fetchIndex(rlsMetadata, RemoteStorageManager.IndexType.OFFSET);
                }
                catch (RemoteStorageException e) {
                    throw new KafkaException((Throwable)e);
                }
            }, file -> {
                try {
                    OffsetIndex index = new OffsetIndex((File)file, startOffset, Integer.MAX_VALUE, false);
                    index.sanityCheck();
                    return index;
                }
                catch (IOException e) {
                    throw new KafkaException((Throwable)e);
                }
            });
            File timeIndexFile = RemoteIndexCache.remoteTimeIndexFile(this.cacheDir, remoteLogSegmentMetadata);
            TimeIndex timeIndex = this.loadIndexFile(timeIndexFile, remoteLogSegmentMetadata, rlsMetadata -> {
                try {
                    return this.remoteStorageManager.fetchIndex(rlsMetadata, RemoteStorageManager.IndexType.TIMESTAMP);
                }
                catch (RemoteStorageException e) {
                    throw new KafkaException((Throwable)e);
                }
            }, file -> {
                try {
                    TimeIndex index = new TimeIndex((File)file, startOffset, Integer.MAX_VALUE, false);
                    index.sanityCheck();
                    return index;
                }
                catch (IOException e) {
                    throw new KafkaException((Throwable)e);
                }
            });
            File txnIndexFile = RemoteIndexCache.remoteTransactionIndexFile(this.cacheDir, remoteLogSegmentMetadata);
            TransactionIndex txnIndex = this.loadIndexFile(txnIndexFile, remoteLogSegmentMetadata, rlsMetadata -> {
                try {
                    return this.remoteStorageManager.fetchIndex(rlsMetadata, RemoteStorageManager.IndexType.TRANSACTION);
                }
                catch (RemoteResourceNotFoundException e) {
                    return new ByteArrayInputStream(new byte[0]);
                }
                catch (RemoteStorageException e) {
                    throw new KafkaException((Throwable)e);
                }
            }, file -> {
                try {
                    TransactionIndex index = new TransactionIndex(startOffset, (File)file);
                    index.sanityCheck();
                    return index;
                }
                catch (IOException e) {
                    throw new KafkaException((Throwable)e);
                }
            });
            return new Entry(offsetIndex, timeIndex, txnIndex);
        }
        catch (IOException e) {
            throw new KafkaException((Throwable)e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int lookupOffset(RemoteLogSegmentMetadata remoteLogSegmentMetadata, long offset) {
        this.lock.readLock().lock();
        try {
            int n = this.getIndexEntry((RemoteLogSegmentMetadata)remoteLogSegmentMetadata).lookupOffset((long)offset).position;
            return n;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public int lookupTimestamp(RemoteLogSegmentMetadata remoteLogSegmentMetadata, long timestamp, long startingOffset) throws IOException {
        this.lock.readLock().lock();
        try {
            int n = this.getIndexEntry((RemoteLogSegmentMetadata)remoteLogSegmentMetadata).lookupTimestamp((long)timestamp, (long)startingOffset).position;
            return n;
        }
        finally {
            this.lock.readLock().unlock();
        }
    }

    @Override
    public void close() {
        if (!this.isRemoteIndexCacheClosed.getAndSet(true)) {
            this.lock.writeLock().lock();
            try {
                log.info("Close initiated for RemoteIndexCache. Cache stats={}. Cache entries pending delete={}", (Object)this.internalCache.stats(), (Object)this.expiredIndexes.size());
                boolean shutdownRequired = this.cleanerThread.initiateShutdown();
                this.internalCache.asMap().forEach((uuid, entry) -> entry.close());
                if (shutdownRequired) {
                    this.cleanerThread.awaitShutdown();
                }
                log.info("Close completed for RemoteIndexCache");
            }
            catch (InterruptedException e) {
                throw new KafkaException((Throwable)e);
            }
            finally {
                this.lock.writeLock().unlock();
            }
        }
    }

    private static void tryAll(List<StorageAction<Void, Exception>> actions) throws IOException {
        IOException firstIOException = null;
        ArrayList<Exception> exceptions = new ArrayList<Exception>();
        for (StorageAction<Void, Exception> action : actions) {
            try {
                action.execute();
            }
            catch (IOException e) {
                if (firstIOException == null) {
                    firstIOException = e;
                    continue;
                }
                exceptions.add(e);
            }
            catch (Exception e) {
                exceptions.add(e);
            }
        }
        if (firstIOException != null) {
            exceptions.forEach(firstIOException::addSuppressed);
            throw firstIOException;
        }
        if (!exceptions.isEmpty()) {
            Iterator iterator = exceptions.iterator();
            KafkaException kafkaException = new KafkaException((Throwable)iterator.next());
            while (iterator.hasNext()) {
                kafkaException.addSuppressed((Throwable)iterator.next());
            }
            throw kafkaException;
        }
    }

    private static Uuid remoteLogSegmentIdFromRemoteIndexFileName(String fileName) {
        int underscoreIndex = fileName.indexOf("_");
        int dotIndex = fileName.indexOf(".");
        return Uuid.fromString((String)fileName.substring(underscoreIndex + 1, dotIndex));
    }

    private static long offsetFromRemoteIndexFileName(String fileName) {
        return Long.parseLong(fileName.substring(0, fileName.indexOf("_")));
    }

    private static String generateFileNamePrefixForIndex(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        long startOffset = remoteLogSegmentMetadata.startOffset();
        Uuid segmentId = remoteLogSegmentMetadata.remoteLogSegmentId().id();
        return startOffset + "_" + segmentId.toString();
    }

    public static File remoteOffsetIndexFile(File dir, RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        return new File(dir, RemoteIndexCache.remoteOffsetIndexFileName(remoteLogSegmentMetadata));
    }

    public static String remoteOffsetIndexFileName(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        String prefix = RemoteIndexCache.generateFileNamePrefixForIndex(remoteLogSegmentMetadata);
        return prefix + ".index";
    }

    public static File remoteTimeIndexFile(File dir, RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        return new File(dir, RemoteIndexCache.remoteTimeIndexFileName(remoteLogSegmentMetadata));
    }

    public static String remoteTimeIndexFileName(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        return RemoteIndexCache.generateFileNamePrefixForIndex(remoteLogSegmentMetadata) + ".timeindex";
    }

    public static File remoteTransactionIndexFile(File dir, RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        return new File(dir, RemoteIndexCache.remoteTransactionIndexFileName(remoteLogSegmentMetadata));
    }

    public static String remoteTransactionIndexFileName(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        return RemoteIndexCache.generateFileNamePrefixForIndex(remoteLogSegmentMetadata) + ".txnindex";
    }

    public static String remoteDeletedSuffixIndexFileName(RemoteLogSegmentMetadata remoteLogSegmentMetadata) {
        return RemoteIndexCache.generateFileNamePrefixForIndex(remoteLogSegmentMetadata) + ".deleted";
    }

    public static class Entry
    implements AutoCloseable {
        private final OffsetIndex offsetIndex;
        private final TimeIndex timeIndex;
        private final TransactionIndex txnIndex;
        private final ReentrantReadWriteLock entryLock = new ReentrantReadWriteLock();
        private boolean cleanStarted = false;
        private boolean markedForCleanup = false;
        private final long entrySizeBytes;

        public Entry(OffsetIndex offsetIndex, TimeIndex timeIndex, TransactionIndex txnIndex) {
            this.offsetIndex = offsetIndex;
            this.timeIndex = timeIndex;
            this.txnIndex = txnIndex;
            this.entrySizeBytes = this.estimatedEntrySize();
        }

        public OffsetIndex offsetIndex() {
            return this.offsetIndex;
        }

        public TimeIndex timeIndex() {
            return this.timeIndex;
        }

        public TransactionIndex txnIndex() {
            return this.txnIndex;
        }

        public boolean isCleanStarted() {
            return this.cleanStarted;
        }

        public boolean isMarkedForCleanup() {
            return this.markedForCleanup;
        }

        public long entrySizeBytes() {
            return this.entrySizeBytes;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private long estimatedEntrySize() {
            this.entryLock.readLock().lock();
            try {
                long l = (long)(this.offsetIndex.sizeInBytes() + this.timeIndex.sizeInBytes()) + Files.size(this.txnIndex.file().toPath());
                return l;
            }
            catch (IOException e) {
                log.warn("Error occurred when estimating remote index cache entry bytes size, just set 0 firstly.", (Throwable)e);
                long l = 0L;
                return l;
            }
            finally {
                this.entryLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public OffsetPosition lookupOffset(long targetOffset) {
            this.entryLock.readLock().lock();
            try {
                if (this.markedForCleanup) {
                    throw new IllegalStateException("This entry is marked for cleanup");
                }
                OffsetPosition offsetPosition = this.offsetIndex.lookup(targetOffset);
                return offsetPosition;
            }
            finally {
                this.entryLock.readLock().unlock();
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        public OffsetPosition lookupTimestamp(long timestamp, long startingOffset) throws IOException {
            this.entryLock.readLock().lock();
            try {
                if (this.markedForCleanup) {
                    throw new IllegalStateException("This entry is marked for cleanup");
                }
                TimestampOffset timestampOffset = this.timeIndex.lookup(timestamp);
                OffsetPosition offsetPosition = this.offsetIndex.lookup(Math.max(startingOffset, timestampOffset.offset));
                return offsetPosition;
            }
            finally {
                this.entryLock.readLock().unlock();
            }
        }

        public void markForCleanup() throws IOException {
            this.entryLock.writeLock().lock();
            try {
                if (!this.markedForCleanup) {
                    this.markedForCleanup = true;
                    this.offsetIndex.renameTo(new File(Utils.replaceSuffix((String)this.offsetIndex.file().getPath(), (String)"", (String)".deleted")));
                    this.timeIndex.renameTo(new File(Utils.replaceSuffix((String)this.timeIndex.file().getPath(), (String)"", (String)".deleted")));
                    this.txnIndex.renameTo(new File(Utils.replaceSuffix((String)this.txnIndex.file().getPath(), (String)"", (String)".deleted")));
                }
            }
            finally {
                this.entryLock.writeLock().unlock();
            }
        }

        public void cleanup() throws IOException {
            this.entryLock.writeLock().lock();
            try {
                this.markForCleanup();
                if (!this.cleanStarted) {
                    this.cleanStarted = true;
                    List<StorageAction> actions = Arrays.asList(() -> {
                        this.offsetIndex.deleteIfExists();
                        return null;
                    }, () -> {
                        this.timeIndex.deleteIfExists();
                        return null;
                    }, () -> {
                        this.txnIndex.deleteIfExists();
                        return null;
                    });
                    RemoteIndexCache.tryAll(actions);
                }
            }
            finally {
                this.entryLock.writeLock().unlock();
            }
        }

        @Override
        public void close() {
            this.entryLock.writeLock().lock();
            try {
                Utils.closeQuietly((AutoCloseable)this.offsetIndex, (String)"OffsetIndex");
                Utils.closeQuietly((AutoCloseable)this.timeIndex, (String)"TimeIndex");
                Utils.closeQuietly((AutoCloseable)this.txnIndex, (String)"TransactionIndex");
            }
            finally {
                this.entryLock.writeLock().unlock();
            }
        }

        public String toString() {
            return "Entry{offsetIndex=" + this.offsetIndex.file().getName() + ", timeIndex=" + this.timeIndex.file().getName() + ", txnIndex=" + this.txnIndex.file().getName() + '}';
        }
    }
}

