/*
 * Decompiled with CFR 0.152.
 */
package org.apache.ratis.server.raftlog.segmented;

import java.io.BufferedInputStream;
import java.io.Closeable;
import java.io.DataInputStream;
import java.io.EOFException;
import java.io.File;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.ByteBuffer;
import java.nio.file.OpenOption;
import java.util.Optional;
import java.util.zip.Checksum;
import org.apache.ratis.io.CorruptedFileException;
import org.apache.ratis.metrics.Timekeeper;
import org.apache.ratis.proto.RaftProtos;
import org.apache.ratis.protocol.exceptions.ChecksumException;
import org.apache.ratis.server.metrics.SegmentedRaftLogMetrics;
import org.apache.ratis.server.raftlog.segmented.SegmentedRaftLogFormat;
import org.apache.ratis.thirdparty.com.google.protobuf.CodedInputStream;
import org.apache.ratis.thirdparty.com.google.protobuf.CodedOutputStream;
import org.apache.ratis.util.FileUtils;
import org.apache.ratis.util.IOUtils;
import org.apache.ratis.util.Preconditions;
import org.apache.ratis.util.PureJavaCrc32C;
import org.apache.ratis.util.SizeInBytes;
import org.apache.ratis.util.StringUtils;
import org.apache.ratis.util.UncheckedAutoCloseable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SegmentedRaftLogReader
implements Closeable {
    static final Logger LOG = LoggerFactory.getLogger(SegmentedRaftLogReader.class);
    private final File file;
    private final LimitedInputStream limiter;
    private final DataInputStream in;
    private byte[] temp = new byte[4096];
    private final Checksum checksum;
    private final SegmentedRaftLogMetrics raftLogMetrics;
    private final SizeInBytes maxOpSize;

    SegmentedRaftLogReader(File file, SizeInBytes maxOpSize, SegmentedRaftLogMetrics raftLogMetrics) throws IOException {
        this.file = file;
        this.limiter = new LimitedInputStream(new BufferedInputStream(FileUtils.newInputStream((File)file, (OpenOption[])new OpenOption[0])));
        this.in = new DataInputStream(this.limiter);
        this.checksum = new PureJavaCrc32C();
        this.maxOpSize = maxOpSize;
        this.raftLogMetrics = raftLogMetrics;
    }

    boolean verifyHeader() throws IOException {
        int headerLength = SegmentedRaftLogFormat.getHeaderLength();
        int readLength = this.in.read(this.temp, 0, headerLength);
        Preconditions.assertTrue((readLength <= headerLength ? 1 : 0) != 0);
        int matchLength = SegmentedRaftLogFormat.matchHeader(this.temp, 0, readLength);
        Preconditions.assertTrue((matchLength <= readLength ? 1 : 0) != 0);
        if (readLength == headerLength && matchLength == readLength) {
            return true;
        }
        if (SegmentedRaftLogFormat.isTerminator(this.temp, matchLength, readLength - matchLength)) {
            return false;
        }
        throw new CorruptedFileException(this.file, "Log header mismatched: expected header length=" + SegmentedRaftLogFormat.getHeaderLength() + ", read length=" + readLength + ", match length=" + matchLength + ", header in file=" + StringUtils.bytes2HexString((byte[])this.temp, (int)0, (int)readLength) + ", expected header=" + StringUtils.bytes2HexString((ByteBuffer)SegmentedRaftLogFormat.getHeaderBytebuffer()));
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    RaftProtos.LogEntryProto readEntry() throws IOException {
        Timekeeper timekeeper = Optional.ofNullable(this.raftLogMetrics).map(SegmentedRaftLogMetrics::getReadEntryTimer).orElse(null);
        try (UncheckedAutoCloseable ignored = Timekeeper.start((Timekeeper)timekeeper);){
            RaftProtos.LogEntryProto logEntryProto = this.decodeEntry();
            return logEntryProto;
        }
        catch (EOFException eof) {
            this.in.reset();
            if (LOG.isWarnEnabled()) {
                LOG.warn("Ignoring the last partial written log entry in " + this.file + ": " + eof);
                return null;
            }
            if (!LOG.isTraceEnabled()) return null;
            LOG.trace("Ignoring the last partial written log entry in " + this.file, (Throwable)eof);
            return null;
        }
        catch (IOException e) {
            this.in.reset();
            throw e;
        }
        catch (Exception e) {
            this.in.reset();
            throw new IOException("got unexpected exception " + e.getMessage(), e);
        }
    }

    long scanEntry() throws IOException {
        return Optional.ofNullable(this.decodeEntry()).map(RaftProtos.LogEntryProto::getIndex).orElse(-1L);
    }

    void verifyTerminator() throws IOException {
        this.limiter.clearLimit();
        int numRead = -1;
        int idx = 0;
        block4: while (true) {
            try {
                numRead = this.in.read(this.temp);
                if (numRead == -1) {
                    return;
                }
                idx = 0;
                while (true) {
                    if (idx >= numRead) continue block4;
                    if (!SegmentedRaftLogFormat.isTerminator(this.temp[idx])) {
                        throw new IOException("Read extra bytes after the terminator at position " + (this.limiter.getPos() - (long)numRead + (long)idx) + " in " + this.file);
                    }
                    ++idx;
                }
            }
            finally {
                if (numRead == -1) continue;
                this.in.reset();
                IOUtils.skipFully((InputStream)this.in, (long)idx);
                this.in.mark(this.temp.length + 1);
                IOUtils.skipFully((InputStream)this.in, (long)1L);
                continue;
            }
            break;
        }
    }

    private RaftProtos.LogEntryProto decodeEntry() throws IOException {
        byte nextByte;
        int max = this.maxOpSize.getSizeInt();
        this.limiter.setLimit(max);
        this.in.mark(max);
        try {
            nextByte = this.in.readByte();
        }
        catch (EOFException eof) {
            return null;
        }
        if (SegmentedRaftLogFormat.isTerminator(nextByte)) {
            this.verifyTerminator();
            return null;
        }
        int entryLength = CodedInputStream.readRawVarint32((int)nextByte, (InputStream)this.in);
        if (entryLength > max) {
            throw new IOException("Entry has size " + entryLength + ", but MAX_OP_SIZE = " + this.maxOpSize);
        }
        int varintLength = CodedOutputStream.computeUInt32SizeNoTag((int)entryLength);
        int totalLength = varintLength + entryLength;
        this.checkBufferSize(totalLength, max);
        this.in.reset();
        this.in.mark(max);
        IOUtils.readFully((InputStream)this.in, (byte[])this.temp, (int)0, (int)totalLength);
        this.checksum.reset();
        this.checksum.update(this.temp, 0, totalLength);
        int expectedChecksum = this.in.readInt();
        int calculatedChecksum = (int)this.checksum.getValue();
        if (expectedChecksum != calculatedChecksum) {
            String s = StringUtils.format((String)"Log entry corrupted: Calculated checksum is %08X but read checksum is %08X.", (Object[])new Object[]{calculatedChecksum, expectedChecksum});
            throw new ChecksumException(s, this.limiter.markPos);
        }
        return RaftProtos.LogEntryProto.parseFrom((CodedInputStream)CodedInputStream.newInstance((byte[])this.temp, (int)varintLength, (int)entryLength));
    }

    private void checkBufferSize(int entryLength, int max) {
        Preconditions.assertTrue((entryLength <= max ? 1 : 0) != 0);
        int length = this.temp.length;
        if (length < entryLength) {
            while (length < entryLength) {
                length = Math.min(length * 2, max);
            }
            this.temp = new byte[length];
        }
    }

    long getPos() {
        return this.limiter.getPos();
    }

    void skipFully(long length) throws IOException {
        this.limiter.clearLimit();
        IOUtils.skipFully((InputStream)this.limiter, (long)length);
    }

    @Override
    public void close() {
        IOUtils.cleanup((Logger)LOG, (Closeable[])new Closeable[]{this.in});
    }

    static class LimitedInputStream
    extends FilterInputStream {
        private long curPos = 0L;
        private volatile long markPos = -1L;
        private long limitPos = Long.MAX_VALUE;

        LimitedInputStream(InputStream is) {
            super(is);
        }

        private void checkLimit(long amt) throws IOException {
            long extra = this.curPos + amt - this.limitPos;
            if (extra > 0L) {
                throw new IOException("Tried to read " + amt + " byte(s) past the limit at offset " + this.limitPos);
            }
        }

        @Override
        public int read() throws IOException {
            this.checkLimit(1L);
            int ret = super.read();
            if (ret != -1) {
                ++this.curPos;
            }
            return ret;
        }

        @Override
        public int read(byte[] data) throws IOException {
            this.checkLimit(data.length);
            int ret = super.read(data);
            if (ret > 0) {
                this.curPos += (long)ret;
            }
            return ret;
        }

        @Override
        public int read(byte[] data, int offset, int length) throws IOException {
            this.checkLimit(length);
            int ret = super.read(data, offset, length);
            if (ret > 0) {
                this.curPos += (long)ret;
            }
            return ret;
        }

        public void setLimit(long limit) {
            this.limitPos = this.curPos + limit;
        }

        public void clearLimit() {
            this.limitPos = Long.MAX_VALUE;
        }

        @Override
        public synchronized void mark(int limit) {
            super.mark(limit);
            this.markPos = this.curPos;
        }

        @Override
        public synchronized void reset() throws IOException {
            if (this.markPos == -1L) {
                throw new IOException("Not marked!");
            }
            super.reset();
            this.curPos = this.markPos;
            this.markPos = -1L;
        }

        public long getPos() {
            return this.curPos;
        }

        @Override
        public long skip(long amt) throws IOException {
            long extra = this.curPos + amt - this.limitPos;
            if (extra > 0L) {
                throw new IOException("Tried to skip " + extra + " bytes past the limit at offset " + this.limitPos);
            }
            long ret = super.skip(amt);
            this.curPos += ret;
            return ret;
        }
    }
}

