/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bcel.classfile;

import java.io.DataInput;
import java.io.DataOutputStream;
import java.io.IOException;

import org.apache.bcel.Const;
import org.apache.bcel.util.Args;

/**
 * This attribute exists for local or anonymous classes and ... there can be only one.
 *
 * @since 6.0
 */
public class EnclosingMethod extends Attribute {

    // Pointer to the CONSTANT_Class_info structure representing the
    // innermost class that encloses the declaration of the current class.
    private int classIndex;

    // If the current class is not immediately enclosed by a method or
    // constructor, then the value of the method_index item must be zero.
    // Otherwise, the value of the method_index item must point to a
    // CONSTANT_NameAndType_info structure representing the name and the
    // type of a method in the class referenced by the class we point
    // to in the class_index. *It is the compiler responsibility* to
    // ensure that the method identified by this index is the closest
    // lexically enclosing method that includes the local/anonymous class.
    private int methodIndex;

    // Ctors - and code to read an attribute in.
    EnclosingMethod(final int nameIndex, final int len, final DataInput input, final ConstantPool cpool) throws IOException {
        this(nameIndex, len, input.readUnsignedShort(), input.readUnsignedShort(), cpool);
    }

    private EnclosingMethod(final int nameIndex, final int len, final int classIndex, final int methodIndex, final ConstantPool cpool) {
        super(Const.ATTR_ENCLOSING_METHOD, nameIndex, Args.require(len, 4, "EnclosingMethod attribute length"), cpool);
        this.classIndex = Args.requireU2(classIndex, 0, cpool.getLength(), "EnclosingMethod class index");
        this.methodIndex = Args.requireU2(methodIndex, "EnclosingMethod method index");
    }

    @Override
    public void accept(final Visitor v) {
        v.visitEnclosingMethod(this);
    }

    @Override
    public Attribute copy(final ConstantPool constantPool) {
        return (Attribute) clone();
    }

    @Override
    public final void dump(final DataOutputStream file) throws IOException {
        super.dump(file);
        file.writeShort(classIndex);
        file.writeShort(methodIndex);
    }

    /**
     * Gets the enclosing class.
     *
     * @return the enclosing class.
     */
    public final ConstantClass getEnclosingClass() {
        return super.getConstantPool().getConstant(classIndex, Const.CONSTANT_Class, ConstantClass.class);
    }

    /**
     * Gets the enclosing class index.
     *
     * @return the enclosing class index.
     */
    public final int getEnclosingClassIndex() {
        return classIndex;
    }

    /**
     * Gets the enclosing method.
     *
     * @return the enclosing method.
     */
    public final ConstantNameAndType getEnclosingMethod() {
        if (methodIndex == 0) {
            return null;
        }
        return super.getConstantPool().getConstant(methodIndex, Const.CONSTANT_NameAndType, ConstantNameAndType.class);
    }

    /**
     * Gets the enclosing method index.
     *
     * @return the enclosing method index.
     */
    public final int getEnclosingMethodIndex() {
        return methodIndex;
    }

    /**
     * Sets the enclosing class index.
     *
     * @param idx the index.
     */
    public final void setEnclosingClassIndex(final int idx) {
        classIndex = idx;
    }

    /**
     * Sets the enclosing method index.
     *
     * @param idx the index.
     */
    public final void setEnclosingMethodIndex(final int idx) {
        methodIndex = idx;
    }
}
