/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.hints;

import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.UUID;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import org.apache.cassandra.cql3.QueryProcessor;
import org.apache.cassandra.cql3.UntypedResultSet;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.Keyspace;
import org.apache.cassandra.db.Mutation;
import org.apache.cassandra.db.SystemKeyspace;
import org.apache.cassandra.db.compaction.CompactionManager;
import org.apache.cassandra.db.marshal.UUIDType;
import org.apache.cassandra.db.partitions.PartitionUpdate;
import org.apache.cassandra.hints.Hint;
import org.apache.cassandra.hints.HintsDescriptor;
import org.apache.cassandra.hints.HintsWriter;
import org.apache.cassandra.io.FSWriteError;
import org.apache.cassandra.io.sstable.Descriptor;
import org.apache.cassandra.io.util.DataInputBuffer;
import org.apache.cassandra.io.util.FileUtils;
import org.apache.cassandra.serializers.MarshalException;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class LegacyHintsMigrator {
    private static final Logger logger = LoggerFactory.getLogger(LegacyHintsMigrator.class);
    private final File hintsDirectory;
    private final long maxHintsFileSize;
    private final ColumnFamilyStore legacyHintsTable;
    private final int pageSize;

    public LegacyHintsMigrator(File hintsDirectory, long maxHintsFileSize) {
        this.hintsDirectory = hintsDirectory;
        this.maxHintsFileSize = maxHintsFileSize;
        this.legacyHintsTable = Keyspace.open("system").getColumnFamilyStore("hints");
        this.pageSize = LegacyHintsMigrator.calculatePageSize(this.legacyHintsTable);
    }

    private static int calculatePageSize(ColumnFamilyStore legacyHintsTable) {
        int size = 128;
        int meanCellCount = legacyHintsTable.getMeanColumns();
        double meanPartitionSize = legacyHintsTable.getMeanPartitionSize();
        if (meanCellCount != 0 && meanPartitionSize != 0.0) {
            int avgHintSize = (int)meanPartitionSize / meanCellCount;
            size = Math.max(2, Math.min(size, 524288 / avgHintSize));
        }
        return size;
    }

    public void migrate() {
        if (this.legacyHintsTable.isEmpty()) {
            return;
        }
        logger.info("Migrating legacy hints to new storage");
        logger.info("Forcing a major compaction of {}.{} table", (Object)"system", (Object)"hints");
        this.compactLegacyHints();
        logger.info("Writing legacy hints to the new storage");
        this.migrateLegacyHints();
        logger.info("Truncating {}.{} table", (Object)"system", (Object)"hints");
        this.legacyHintsTable.truncateBlocking();
    }

    private void compactLegacyHints() {
        ArrayList<Descriptor> descriptors = new ArrayList<Descriptor>();
        this.legacyHintsTable.getTracker().getUncompacting().forEach(sstable -> descriptors.add(sstable.descriptor));
        if (!descriptors.isEmpty()) {
            this.forceCompaction(descriptors);
        }
    }

    private void forceCompaction(Collection<Descriptor> descriptors) {
        try {
            CompactionManager.instance.submitUserDefined(this.legacyHintsTable, descriptors, FBUtilities.nowInSeconds()).get();
        }
        catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private void migrateLegacyHints() {
        ByteBuffer buffer = ByteBuffer.allocateDirect(262144);
        String query = String.format("SELECT DISTINCT target_id FROM %s.%s", "system", "hints");
        QueryProcessor.executeInternal(query, new Object[0]).forEach(row -> this.migrateLegacyHints(row.getUUID("target_id"), buffer));
        FileUtils.clean(buffer);
    }

    private void migrateLegacyHints(UUID hostId, ByteBuffer buffer) {
        String query = String.format("SELECT target_id, hint_id, message_version, mutation, ttl(mutation) AS ttl, writeTime(mutation) AS write_time FROM %s.%s WHERE target_id = ?", "system", "hints");
        UntypedResultSet rows = QueryProcessor.executeInternalWithPaging(query, this.pageSize, hostId);
        this.migrateLegacyHints(hostId, rows, buffer);
        LegacyHintsMigrator.deleteLegacyHintsPartition(hostId);
    }

    private void migrateLegacyHints(UUID hostId, UntypedResultSet rows, ByteBuffer buffer) {
        this.migrateLegacyHints(hostId, rows.iterator(), buffer);
    }

    private void migrateLegacyHints(UUID hostId, Iterator<UntypedResultSet.Row> iterator, ByteBuffer buffer) {
        do {
            this.migrateLegacyHintsInternal(hostId, iterator, buffer);
        } while (iterator.hasNext());
    }

    private void migrateLegacyHintsInternal(UUID hostId, Iterator<UntypedResultSet.Row> iterator, ByteBuffer buffer) {
        HintsDescriptor descriptor = new HintsDescriptor(hostId, System.currentTimeMillis());
        try (HintsWriter writer = HintsWriter.create(this.hintsDirectory, descriptor);
             HintsWriter.Session session = writer.newSession(buffer);){
            while (iterator.hasNext()) {
                Hint hint = LegacyHintsMigrator.convertLegacyHint(iterator.next());
                if (hint != null) {
                    session.append(hint);
                }
                if (session.position() < this.maxHintsFileSize) continue;
                break;
            }
        }
        catch (IOException e) {
            throw new FSWriteError((Throwable)e, descriptor.fileName());
        }
    }

    private static Hint convertLegacyHint(UntypedResultSet.Row row) {
        Mutation mutation = LegacyHintsMigrator.deserializeLegacyMutation(row);
        if (mutation == null) {
            return null;
        }
        long creationTime = row.getLong("write_time");
        int expirationTime = FBUtilities.nowInSeconds() + row.getInt("ttl");
        int originalGCGS = expirationTime - (int)TimeUnit.MILLISECONDS.toSeconds(creationTime);
        int gcgs = Math.min(originalGCGS, mutation.smallestGCGS());
        return Hint.create(mutation, creationTime, gcgs);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static Mutation deserializeLegacyMutation(UntypedResultSet.Row row) {
        try (DataInputBuffer dib = new DataInputBuffer(row.getBlob("mutation"), true);){
            Mutation mutation2 = Mutation.serializer.deserialize(dib, row.getInt("message_version"));
            mutation2.getPartitionUpdates().forEach(PartitionUpdate::validate);
            Mutation mutation = mutation2;
            return mutation;
        }
        catch (IOException e) {
            logger.error("Failed to migrate a hint for {} from legacy {}.{} table: {}", new Object[]{row.getUUID("target_id"), "system", "hints", e});
            return null;
        }
        catch (MarshalException e) {
            logger.warn("Failed to validate a hint for {} (table id {}) from legacy {}.{} table - skipping: {})", new Object[]{row.getUUID("target_id"), "system", "hints", e});
            return null;
        }
    }

    private static void deleteLegacyHintsPartition(UUID hostId) {
        Mutation mutation = new Mutation(PartitionUpdate.fullPartitionDelete(SystemKeyspace.LegacyHints, UUIDType.instance.decompose(hostId), System.currentTimeMillis(), FBUtilities.nowInSeconds()));
        mutation.applyUnsafe();
    }
}

