/**
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.atlas.notification;

import org.apache.atlas.AtlasClient;
import org.apache.atlas.AtlasException;
import org.apache.atlas.AtlasServiceException;
import org.apache.atlas.exception.AtlasBaseException;
import org.apache.atlas.kafka.AtlasKafkaMessage;
import org.apache.atlas.kafka.KafkaNotification;
import org.apache.atlas.model.instance.AtlasEntity;
import org.apache.atlas.notification.hook.HookNotification;
import org.apache.atlas.kafka.*;
import org.apache.atlas.repository.converters.AtlasInstanceConverter;
import org.apache.atlas.repository.store.graph.AtlasEntityStore;
import org.apache.atlas.repository.store.graph.v1.EntityStream;
import org.apache.atlas.type.AtlasType;
import org.apache.atlas.type.AtlasTypeRegistry;
import org.apache.atlas.typesystem.Referenceable;
import org.apache.atlas.web.service.ServiceState;
import org.apache.commons.lang.RandomStringUtils;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.testng.Assert;
import org.testng.annotations.AfterTest;
import org.testng.annotations.BeforeTest;
import org.testng.annotations.Test;
import static org.apache.atlas.notification.hook.HookNotification.HookNotificationMessage;
import java.util.List;
import java.util.Properties;

import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyBoolean;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.*;

import org.apache.commons.configuration.Configuration;
import org.apache.atlas.ApplicationProperties;
import static org.testng.Assert.*;



public class NotificationHookConsumerKafkaTest {

    public static final String NAME = "name";
    public static final String DESCRIPTION = "description";
    public static final String QUALIFIED_NAME = "qualifiedName";
    private NotificationInterface notificationInterface = null;
    private EmbeddedKafkaServer   kafkaServer           = null;
    private KafkaNotification     kafkaNotification     = null;


    @Mock
    private AtlasEntityStore atlasEntityStore;

    @Mock
    private ServiceState serviceState;

    @Mock
    private AtlasInstanceConverter instanceConverter;

    @Mock
    private AtlasTypeRegistry typeRegistry;

    @BeforeTest
    public void setup() throws AtlasException, InterruptedException, AtlasBaseException {
        MockitoAnnotations.initMocks(this);
        AtlasType mockType = mock(AtlasType.class);
        when(typeRegistry.getType(anyString())).thenReturn(mockType);
        AtlasEntity.AtlasEntitiesWithExtInfo mockEntity = mock(AtlasEntity.AtlasEntitiesWithExtInfo.class);
        when(instanceConverter.toAtlasEntities(anyList())).thenReturn(mockEntity);

        initNotificationService();
    }

    @AfterTest
    public void shutdown() {
        cleanUpNotificationService();
    }

    @Test
    public void testConsumerConsumesNewMessageWithAutoCommitDisabled() throws AtlasException, InterruptedException, AtlasBaseException {
        produceMessage(new HookNotification.EntityCreateRequest("test_user1", createEntity()));

        NotificationConsumer<HookNotificationMessage> consumer = createNewConsumer(kafkaNotification, false);
        NotificationHookConsumer notificationHookConsumer =
                new NotificationHookConsumer(notificationInterface, atlasEntityStore, serviceState, instanceConverter, typeRegistry);
        NotificationHookConsumer.HookConsumer hookConsumer = notificationHookConsumer.new HookConsumer(consumer);

        consumeOneMessage(consumer, hookConsumer);
        verify(atlasEntityStore).createOrUpdate(any(EntityStream.class), anyBoolean());

        // produce another message, and make sure it moves ahead. If commit succeeded, this would work.
        produceMessage(new HookNotification.EntityCreateRequest("test_user2", createEntity()));
        consumeOneMessage(consumer, hookConsumer);
        verify(atlasEntityStore,times(2)).createOrUpdate(any(EntityStream.class), anyBoolean());
        reset(atlasEntityStore);
    }

    @Test
    public void consumerConsumesNewMessageButCommitThrowsAnException_MessageOffsetIsRecorded() throws AtlasException, InterruptedException, AtlasBaseException {

        ExceptionThrowingCommitConsumer        consumer                 = createNewConsumerThatThrowsExceptionInCommit(kafkaNotification, true);
        NotificationHookConsumer               notificationHookConsumer = new NotificationHookConsumer(notificationInterface, atlasEntityStore, serviceState, instanceConverter, typeRegistry);
        NotificationHookConsumer.HookConsumer  hookConsumer             = notificationHookConsumer.new HookConsumer(consumer);
        NotificationHookConsumer.FailedCommitOffsetRecorder failedCommitOffsetRecorder = hookConsumer.failedCommitOffsetRecorder;

        produceMessage(new HookNotification.EntityCreateRequest("test_user2", createEntity()));

        try {
            produceMessage(new HookNotification.EntityCreateRequest("test_user1", createEntity()));
            consumeOneMessage(consumer, hookConsumer);
            consumeOneMessage(consumer, hookConsumer);
        }
        catch(KafkaException ex) {
            assertTrue(true, "ExceptionThrowing consumer throws an excepion.");
        }

        assertTrue(failedCommitOffsetRecorder.getCurrentOffset() > -1);

        consumer.disableCommitExpcetion();

        produceMessage(new HookNotification.EntityCreateRequest("test_user1", createEntity()));
        consumeOneMessage(consumer, hookConsumer);
        consumeOneMessage(consumer, hookConsumer);

        assertNull(failedCommitOffsetRecorder.getCurrentOffset());

        reset(atlasEntityStore);
    }

    @Test(dependsOnMethods = "testConsumerConsumesNewMessageWithAutoCommitDisabled")
    public void testConsumerRemainsAtSameMessageWithAutoCommitEnabled() throws Exception {
        produceMessage(new HookNotification.EntityCreateRequest("test_user3", createEntity()));

        NotificationConsumer<HookNotificationMessage> consumer = createNewConsumer(kafkaNotification, true);

        assertNotNull (consumer);

        NotificationHookConsumer notificationHookConsumer =
                new NotificationHookConsumer(notificationInterface, atlasEntityStore, serviceState, instanceConverter, typeRegistry);
        NotificationHookConsumer.HookConsumer hookConsumer = notificationHookConsumer.new HookConsumer(consumer);

        consumeOneMessage(consumer, hookConsumer);
        verify(atlasEntityStore).createOrUpdate(any(EntityStream.class), anyBoolean());

        // produce another message, but this will not be consumed, as commit code is not executed in hook consumer.
        produceMessage(new HookNotification.EntityCreateRequest("test_user4", createEntity()));

        consumeOneMessage(consumer, hookConsumer);
        verify(atlasEntityStore,times(2)).createOrUpdate(any(EntityStream.class), anyBoolean());
    }

    AtlasKafkaConsumer<HookNotificationMessage> createNewConsumer(KafkaNotification kafkaNotification, boolean autoCommitEnabled) {
        return (AtlasKafkaConsumer) kafkaNotification.createConsumers(NotificationInterface.NotificationType.HOOK, 1, autoCommitEnabled).get(0);
    }

    ExceptionThrowingCommitConsumer createNewConsumerThatThrowsExceptionInCommit(KafkaNotification kafkaNotification, boolean autoCommitEnabled) {
        Properties prop = kafkaNotification.getConsumerProperties(NotificationInterface.NotificationType.HOOK);
        KafkaConsumer consumer = kafkaNotification.getKafkaConsumer(prop, NotificationInterface.NotificationType.HOOK, true);
        return new ExceptionThrowingCommitConsumer(NotificationInterface.NotificationType.HOOK, consumer, autoCommitEnabled, 1000);
    }

    void consumeOneMessage(NotificationConsumer<HookNotificationMessage> consumer,
                           NotificationHookConsumer.HookConsumer hookConsumer) throws InterruptedException {
        try {
            long startTime = System.currentTimeMillis(); //fetch starting time
            while ((System.currentTimeMillis() - startTime) < 10000) {
                List<AtlasKafkaMessage<HookNotificationMessage>> messages = consumer.receive();

                for (AtlasKafkaMessage<HookNotificationMessage> msg : messages) {
                    hookConsumer.handleMessage(msg);
                }

                if (messages.size() > 0) {
                    break;
                }
            }
        } catch (AtlasServiceException | AtlasException e) {
            Assert.fail("Consumer failed with exception ", e);
        }
    }

    Referenceable createEntity() {
        final Referenceable entity = new Referenceable(AtlasClient.DATA_SET_SUPER_TYPE);
        entity.set(NAME, "db" + randomString());
        entity.set(DESCRIPTION, randomString());
        entity.set(QUALIFIED_NAME, randomString());
        return entity;
    }

    protected String randomString() {
        return RandomStringUtils.randomAlphanumeric(10);
    }

    private void produceMessage(HookNotification message) throws NotificationException {
        kafkaNotification.send(NotificationInterface.NotificationType.HOOK, message);
    }

    void initNotificationService() throws AtlasException, InterruptedException {
        Configuration applicationProperties = ApplicationProperties.get();
        applicationProperties.setProperty("atlas.kafka.data", "target/" + RandomStringUtils.randomAlphanumeric(5));

        kafkaServer           = new EmbeddedKafkaServer(applicationProperties);
        kafkaNotification     = new KafkaNotification(applicationProperties);
        notificationInterface = kafkaNotification;

        kafkaServer.start();
        kafkaNotification.start();
        Thread.sleep(2000);
    }

    void cleanUpNotificationService() {
        if (kafkaNotification != null) {
            kafkaNotification.close();
            kafkaNotification.stop();
        }
    }

    private void produceMessage(HookNotificationMessage message) throws NotificationException {
        kafkaNotification.send(NotificationInterface.NotificationType.HOOK, message);
    }

    private static class ExceptionThrowingCommitConsumer extends AtlasKafkaConsumer {

        private boolean exceptionThrowingEnabled;

        public ExceptionThrowingCommitConsumer(NotificationInterface.NotificationType notificationType,
                                               KafkaConsumer kafkaConsumer, boolean autoCommitEnabled, long pollTimeoutMilliSeconds) {
            super(notificationType.getDeserializer(), kafkaConsumer, autoCommitEnabled, pollTimeoutMilliSeconds);
            exceptionThrowingEnabled = true;
        }

        @Override
        public void commit(TopicPartition partition, long offset) {
            if(exceptionThrowingEnabled) {
                throw new KafkaException("test case verifying exception");
            }
            else {
                super.commit(partition, offset);
            }
        }

        public void disableCommitExpcetion() {
            exceptionThrowingEnabled = false;
        }
    }
}
