/* texdecc.c

   Copyright 2000, 2015 Akira Kakuto.

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this library; if not, see <http://www.gnu.org/licenses/>.

 *
 * texdocc.c for W32
 * (2007/07/01 --ak) Public domain.
 *
 * Usage: texdocc [-h | -V | -v -s -l] docname [SubStringInPath]
 *        -l list matching files without using viewers
 *        -s use substring for the docname
 *        -v verbose mode
 *        -h print usage and exit
 *        -V print version and exit
 *
 * examples:
 * texdocc powerdot
 *         displays powerdot.pdf if it exists under $TEXMF/doc/
 * texdocc README pdftex
 *         displays README under $TEXMF/doc/pdftex/
 * texdocc -l README dvipdfmx
 *         list the full path of README under $TEXMF/doc/dvipdfmx/
 * texdocc -l -s read pdftex
 *         list files containing "read" as a substring in the file name
 *         and containing "pdftex" in the string of the full path

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include <malloc.h>

/*
 * kpathsea functions
 */

#include <kpathsea/kpathsea.h>
#include <kpathsea/dirent.h>
#include <windows.h>
#define  TRANSFORM(x) ((unsigned)CharLower((LPTSTR)(BYTE)(x)))
#define  PATHSEP  ';'

#define  VERSION  "0.2"
/*
 Global variables
*/
char Substr[512];
char Foundname[512];
int  Listonly = 0;
int  Usesubname = 0;
int  Viewer = -1;
int  Verbose = 0;

/*
 * The function doc_search is used only if Usesubname != 0,
 * when Substr[0] != '\0'.
 */
void doc_search(char *name, char *dirsubstr)
{
  DIR *dp;
  struct dirent *de;
  char   buff[512];
  char   *p;
  int    len;

  if(Listonly == 0 && Viewer != -1)
    return;

  for(p = name; *p; p++) {
    if(IS_KANJI(p)) {
      p++;
      continue;
    }
    if(*p == '\\') *p = '/';
  }

  if(dp = opendir(name)) {
    for(p=name; *p; p++) {
      if(IS_KANJI(p)) {
        p++;
        continue;
      }
      *p = TRANSFORM(*p);
    }

    while(de = readdir(dp)) {
      if((de->d_name[0] != '.')) {
        for(p=de->d_name; *p; p++) {
          if(IS_KANJI(p)) {
            p++;
            continue;
          }
          *p = TRANSFORM(*p);
        }

        if(!de->d_isdir && strstr(de->d_name, Substr) != 0) {
          sprintf(Foundname, "%s/%s", name, de->d_name);
          p = strrchr (de->d_name, '.');
            if (!p) {
              Viewer = 0;
            } else if (!stricmp (p, ".dvi")) {
              Viewer = 1;
            } else if (!stricmp (p, ".htm")) {
              Viewer = 2;
            } else if (!stricmp (p, ".html")) {
              Viewer = 2;
            } else if (!stricmp (p, ".pdf")) {
              Viewer = 3;
            } else if (!stricmp (p, ".ps")) {
              Viewer = 4;
            } else if (!stricmp (p, ".eps")) {
              Viewer = 4;
            } else {
              Viewer = 0;
            }

            if(Listonly == 0) {
              if(dirsubstr[0] == '\0')
                return;
              else {
                if(strstr(Foundname, dirsubstr) != 0)
                  return;
                else {
                  Viewer = -1;
                  Foundname[0] = '\0';
                }
              }
            } else {
            if(dirsubstr[0] == '\0')
              printf("%s\n", Foundname);
            else if(strstr(Foundname, dirsubstr) != 0)
              printf("%s\n", Foundname);
          }
        }
      }
    }
    closedir(dp);
    dp = opendir(name);
    while(de = readdir(dp)) {
      if((de->d_name[0] != '.') && de->d_isdir) {
        len = (int)strlen(name);
        strcpy(buff, name);
        if(name[len-1] != '/')
          strcat(buff, "/");
        strcat(buff, de->d_name);
        doc_search(buff, dirsubstr);
      }
    }
    closedir(dp);
    return;
  }
  else
    return;
}

/*
 * DVIVIEW:
 * "dviout "
 *
 */
#define DVIVIEW "dviout "
#define GSVIEW "gsview32 "


/*
 * search by using Kpathsea
 */
char *k_find_file (char *fname, char *dirsubstr)
{
  char *p, *q;
  char **list, **lp;

  if(dirsubstr[0] == '\0' && Listonly == 0) {
    p = kpse_find_file (fname, kpse_texdoc_format, 0);
    return p;
  } else {
    p = kpse_var_value("TEXDOCS");
    if (!p) {
      fprintf (stderr, "Please define TEXDOCS in texmf.cnf.\n");
        exit (1);
    }
    q = kpse_brace_expand (p);
    free (p);
    list = kpse_all_path_search (q, fname);
    if (q) free (q);
    if (!list) return NULL;
    q = NULL;
    for (lp = list ; *lp; lp++) {
      p = *lp;
      (void)_strlwr (p);
      if(dirsubstr[0] == '\0') {
        if(Listonly) {
          printf("%s\n", p);
        } else {
          q = (char *)malloc(strlen(p) + 2);
          strcpy(q, p);
          break;
        }
      } else if (strstr (p, dirsubstr)) {
        if(Listonly) {
          printf("%s\n", p);
        } else {
          q = (char *)malloc (strlen (p) + 2);
          strcpy (q, p);
          break;
        }
      }
    }
    for(lp = list; *lp; lp++) {
       if(*lp) free(*lp);
    }
    if(list) free(list);
    return q;
  }
  return NULL;
}

int is_nt (void)
{
  OSVERSIONINFO info;

  info.dwOSVersionInfoSize = sizeof(OSVERSIONINFO);

  if (!GetVersionEx (&info)) {
    fprintf (stderr, "Failed to get OSVersionInfo.\n");
    exit (1);
  }
  if (info.dwPlatformId == VER_PLATFORM_WIN32_NT) {
    return 1;
  } else {
    return 0;
  }
}

int run (char *cmd)
{
  int ret;
  ret = win32_system (cmd);
  return ret;
}

void usage(char *ss)
{
  fprintf(stderr, "Usage(1): %s -h\n", ss);
  fprintf(stderr, "          display this message and exit\n");
  fprintf(stderr, "          %s -V\n", ss);
  fprintf(stderr, "          print version and exit\n");
  fprintf(stderr, "Usage(2): %s [Option(s)] FileName [SubStringInPath]\n",
          ss);
  fprintf(stderr, "Options :\n");
  fprintf(stderr, "          -l list matching files without using viewers\n");
  fprintf(stderr, "          -s use substring for the FileName\n");
  fprintf(stderr, "          -v verbose mode\n");
  fprintf(stderr, "Example :\n");
  fprintf(stderr, "texdocc -l -s readme pdftex\n");
  fprintf(stderr, "          list all files containing \"readme\" as a substring\n");
  fprintf(stderr, "          in the file name, and containing a string \"pdftex\"\n");
  fprintf(stderr, "          in the full path name.\n");
  exit (1);
}

int main (int argc, char **argv)
{
  char *p, *q, *pa, *pb, *pc;
  char dirsubstr[256];
  char cmd[256];
  char fname[256];
  char infname[256];
  char fullname[512];
  int  i, viewer;
  char **pathbuff = NULL;
  int  numtree;
  char *texmfdocs, *saveptr = NULL;
  char *progname;
/*
  viewer =
  0 : more (text file)
  1 : DVIVIEW
  2 : html
  3 : pdf
  4 : ps
*/

  kpse_set_program_name (argv[0], NULL);
  progname = kpse_program_name;

  if (argc == 1)
    usage(progname);

  fullname[0] = '\0';
  dirsubstr[0] = '\0';
  fname[0] = '\0';
  infname[0] = '\0';
  Foundname[0] = '\0';
  Substr[0] = '\0';

  argc--;
  argv++;
  while(argc > 0) {
    if(strncmp(*argv, "-V", 2) == 0 || strncmp(*argv, "--V", 3) == 0) {
      fprintf(stderr, "texdocc Version " VERSION "\n");
      return 0;
    }
    if(strncmp(*argv, "-h", 2) == 0 || strncmp(*argv, "--h", 3) == 0) {
      usage(progname);
    }
    if(strncmp(*argv, "-v", 2) == 0 || strncmp(*argv, "--v", 3) == 0) {
      Verbose = 1;
      argc--;
      argv++;
      continue;
    }
    if(strncmp(*argv, "-l", 2) == 0 || strncmp(*argv, "--l", 3) == 0) {
      Listonly = 1;
      argc--;
      argv++;
      continue;
    }
    if(strncmp(*argv, "-s", 2) == 0 || strncmp(*argv, "--s", 3) == 0) {
      Usesubname = 1;
      argc--;
      argv++;
      continue;
    }
    if(!fname[0]) {
      strcpy(fname, *argv);
      strcpy(infname, *argv);
      _strlwr(fname);
      _strlwr(infname);
      if(Usesubname) {
        strcpy(Substr, *argv);
        _strlwr(Substr);
      }
      argc--;
      argv++;
      continue;
    }
    if(fname[0] && !dirsubstr[0]) {
      strcpy(dirsubstr, *argv);
      _strlwr(dirsubstr);
      argc--;
      argv++;
      continue;
    }
    if(dirsubstr[0]) {
      fprintf(stderr, "Wrong argument.\n");
      exit (1);
    }
  }

  if(!infname[0]) {
    fprintf(stderr, "Wrong argument.\n");
    exit (1);
  }

  if(Usesubname) {
    if(!(p = kpse_var_value("TEXDOCS"))) {
      fprintf(stderr, "TEXDOCS is not defined in texmf.cnf.\n");
      exit (100);
    }

    texmfdocs = saveptr = kpse_brace_expand(p);
    free(p);

    i = 0;
    p = texmfdocs;

    while(*p) {
      if(*p == '!' && *(p+1) == '!')
        p += 2;
      if(*p == ';')
        while(*p == ';')
          p++;
      if(*p && *p != '!') {
        while(*p != ';' && *p)
          p++;
        if(*p == ';') {
          i++;
          while(*p == ';')
            p++;
        } else if(*p == '\0') {
          i++;
          break;
        }
      }
    }

    numtree = i;

    pathbuff = malloc(numtree * sizeof(char *));

    pa = texmfdocs;
    i = 0;

    while (*pa) {
      if (*pa == '!' && *(pa + 1) == '!')
        pa += 2;
      if (*pa == ';') {
        while (*pa == ';')
          pa++;
      }
      if(*pa && *pa != '!') {
        pathbuff[i] = malloc(strlen(pa) + 1);
        pc = pathbuff[i];
        while (*pa != ';' && *pa)
          *pc++ = *pa++;
        *pc = '\0';
        q = pathbuff[i] + strlen(pathbuff[i]) - 1;
        while(*q == '/') *q-- = '\0';
        if (*pa == ';') {
          while (*pa == ';')
            pa++;
          i++;
        } else if (*pa == '\0') {
          i++;
          break;
        }
      }
    }

    for(i=0; i < numtree; i++) {
      Viewer = -1;
      doc_search(pathbuff[i], dirsubstr);
      if(Listonly == 0 && Viewer != -1)
        break;
    }

    for(i = 0; i < numtree; i++) {
      if(pathbuff[i]) free(pathbuff[i]);
    }

    if(saveptr) free(saveptr);
    free(pathbuff);
    if(Foundname[0] == '\0') {
      fprintf(stderr, "Cannot find %s\n", infname);
      return 1;
    }

    if(Listonly) {
      return 0;
    } else {
      strcpy(fullname, Foundname);
      viewer = Viewer;
      goto done;
    }
  }

  p = k_find_file (fname, dirsubstr);
  if(p) {
    if(Listonly == 0) {
      strcpy (fullname, p);
      free(p);
      q = strrchr (fullname, '.');
      if (!q) {
        viewer = 0;
      } else if (!stricmp (q, ".dvi")) {
        viewer = 1;
      } else if (!stricmp (q, ".htm")) {
        viewer = 2;
      } else if (!stricmp (q, ".html")) {
        viewer = 2;
      } else if (!stricmp (q, ".pdf")) {
        viewer = 3;
      } else if (!stricmp (q, ".ps")) {
        viewer = 4;
      } else if (!stricmp (q, ".eps")) {
        viewer = 4;
      } else {
        viewer = 0;
      }
      goto done;
    } else {
      free(p);
    }
  }

  strcat (fname, ".txt");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 0;
      strcpy (fullname, p);
      free(p);
      goto done;
    } else {
      free (p);
    }
  }

  strcpy (fname, infname);
  strcat (fname, ".dvi");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 1;
      strcpy (fullname, p);
      free(p);
      goto done;
    } else {
      free (p);
    }
  }

  strcpy (fname, infname);
  strcat (fname, ".htm");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 2;
      strcpy (fullname, p);
      free(p);
      goto done;
    } else {
      free (p);
    }
  }

  strcpy (fname, infname);
  strcat (fname, ".html");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 2;
      strcpy (fullname, p);
      free(p);
      goto done;
    } else {
      free (p);
    }
  }

  strcpy (fname, infname);
  strcat (fname, ".pdf");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 3;
      strcpy (fullname, p);
      free(p);
      goto done;
    } else {
      free (p);
    }
  }

  strcpy (fname, infname);
  strcat (fname, ".ps");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 4;
      strcpy (fullname, p);
      free(p);
      goto done;
    } else {
      free (p);
    }
  }

  strcpy (fname, infname);
  strcat (fname, ".eps");
  p = k_find_file (fname, dirsubstr);
  if (p) {
    if(Listonly == 0) {
      viewer = 4;
      strcpy (fullname, p);
    }
    free(p);
  }

 done:
  if(Listonly == 0) {
    if (!fullname[0]) {
      fprintf (stderr, "I cannot find a document %s", infname);
      if (dirsubstr[0]) {
        fprintf (stderr, " in the path containing the string \"%s\"\n", dirsubstr);
      } else {
        fprintf (stderr, "\n");
      }
      return 1;
    }

    if (viewer == 0) {
      char *env = getenv("PAGER");
      if(env) {
        strcpy(cmd, env);
        strcat(cmd, " ");
      } else {
        strcpy (cmd, "more ");
      }
    } else if (viewer > 0 && viewer < 5) {
      strcpy (cmd, "start \"document viewer\" ");
    } else {
      fprintf (stderr, "There is no application to view %s\n", infname);
      return 1;
    }
    if (viewer == 1)
      strcat (cmd, DVIVIEW);
    if (viewer == 4)
      strcat (cmd, GSVIEW);
    if (viewer == 0) {
      for(p=fullname; *p; p++) {
        if(IS_KANJI(p)) {
          p++;
          continue;
        }
        if(*p == '/') *p = '\\';
      }
    }
    strcat (cmd, "\"");
    strcat (cmd, fullname);
    strcat (cmd, "\"");
    if(Verbose) {
      fprintf(stderr, "I now display a document by the command\n");
      fprintf(stderr, "%s\n", cmd);
    }
    return run (cmd);
  }
  return 0;
}
