/*========================================================*\
| This is zwepmtex.e  (toolbar actions file)  version 1.0a |
| Copyright (C)  Z. Wagner -- Ice Bear Soft,   31 Dec 1995 |
| ======================================================== |
| See zwtexsup.inf for Licence Agreement and documentation |
\*========================================================*/


/************************************************************************************

This file defines the actions of the TeX toolbar. It relies on the actions and buttons of
the tiny toolbar distributed with the Enhanced Editor 6.03.

The actions are defined by ZWEPMTEX_ACTIONLIST. Each line consists of the command to be
executed, a description of the action, and the name of the '.ex' file, each separated by
a common delimiter ('' is used here).

The action commands may be called with the following parameters:
   I  =  menu has been initialized
   E  =  menu end message (usually ignored)
   H  =  help
   S  =  toolbar item has been selected

Here is the list of commands and procedures with short descriptions.

Commands:
  zw_TeXme                    -  invokes TeXme.cmd on the current file, collects options
                                 for run_TeX_me (see below)
  zw_Save_All                 -  saves all files in the ring
  zw_View                     -  invokes preview by calling vp.cmd through dvi_command()
  zw_dvi_command              -  executes dvi-command as given in the parameter
  zw_Open_Log                 -  (re)opens the log-file, tries to restore the cursor
                                 position
  zw_Refresh                  -  erfreshes the current file, tries to restore the cursor
                                 position
  zw_Open_Main                -  opens the associated file specified in 'MAIN='
  zw_Open_TeX                 -  opens the TeX input file under cursor
  zw_TeX_subject              -  sets/view the current file's TeX subject


Procedures:
  get_subject(var subj)       -  gets the subject of the current file

  get_main_tex(drv, path)     -  gets 'MAIN=' value from the subject, if the value does
                                 not contain path, the arguments are prepended to the
                                 value

  split_file_name(var drive, var path, var base, var ext, filespec)
                              -  splits the file name into parts

  quit_log( logfile )         -  quits the log file of the specified name and saves the
                                 cursor position in the universal variable

  refresh_log( logfile )      -  refreshes the log file from disk and tries to restore the
                                 cursor position saved by quit_log

  run_TeX_me(command, opt)    -  executes zw_TeXme, opt may only be '-' if given
                                 in the tool parameter, command may contain:
                                    All      =  save all files
                                    CURrent  =  save current file

  save_the_ring()             -  saves all files in the ring

  dvi_command(pgm, cmdline)   -  cuts the extension from the current file, replaces all
                                 '*' chars in the cmdline with the file name without
                                 extension and then executes asynchronously 'pgm' with
                                 'cmdline'

  refresh_file( fn )          -  refreshes the specified file, tries to restore the cursor
                                 position

  open_it( fn )               -  (re)opens the file specified

  open_if_exists_w_x( fn, extns, drv, path )
                              -  opens the file if it exists, otherwise it tries the
                                 extensions specified, returns 1 on success
                                 If the file cannot be found and its path is not
                                 specified, the last two arguments are prepended to the
                                 file name

  open_TeX()                  -  opens the TeX input/style/class... file under cursor

  TeX_subject()               -  sets/views TeX subject

  MsgBox(caption, text, attr) -  uses WinMessageBox to display the 'caption' and 'text'
                                 with attributes set as MB_OK + MB_MOVEABLE + attr,
                                 returns WinMessageBox's result

  InfoBox(caption, text)      -  calls MsgBox with attr set to MB_INFORMATION,
                                 returns its result

  ErrorBox(caption, text)     -  beeps and calls MsgBox with attr set to MB_ERROR,
                                 returns its result

  DebugBox(text)              -  used for debugging

All commands and procedures start at bookmarks.

************************************************************************************/

/* Constants for EA handling */

const
  EAT_ASCII       = \253\255  -- FFFD
  EAT_MVMT        = \223\255  -- FFDF

/* For OS/2 command line */

  OS2cmd          = 'start /C /WIN'

/* TeX extensions */
  TeXextns        = '.tex .sty .cls .clo .dtx'

/* For TeX subjects */
  max_TeX_subj    =  265

/* Message strings */
  zw_TeXme_inf    =  "Runs TeXme,cmd, parameter '-' (if present) passed as the second option, All used to save all files, CURrent used to save the current file"
  zw_Save_All_inf =  'Save all files in the edit ring'
  zw_View_inf     =  "Runs preview by calling 'vp.cmd', all occurences of '*' in the parameter will be replaced with the file name"
  zw_dvi_com_inf  =  "Runs specified command on the dvi-file. The parameter must contain the program action and its argument where all occurences of '*' in the parameter will be replaced with the file name"
  zw_Open_Log_inf =  'Reopens the log-file'
  zw_Refresh_inf  =  'Refresh current file'
  zw_Open_Main_inf=  'Opens TeX main file (extension=.tex)'
  zw_Open_TeX_inf =  'Opens TeX input file under cursor'
  zw_TeX_subj_inf =  'Sets/views the TeX subject, deletion is not supported yet'

  zw_help         =  'start /C /PM view.exe zwtexsup.inf'
  Hmsg            =  0
  Hinf            =  1

include 'stdconst.e'

/* Initialization for log-refresh and TeX subjects */

DEFINIT
universal lp, TeX_subj_fld
lp = -1
TeX_subj_fld = 'FORMAT=plain'

/* The following command adds the action commands to the list */

DEFC zwepmtex_actionlist
universal ActionsList_FileID

 insertline 'zw_TeXme'zw_TeXme_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_Save_All'zw_Save_All_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_View'zw_View_Inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_dvi_command'zw_dvi_com_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_Open_Log'zw_Open_Log_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_Refresh'zw_Refresh_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_Open_Main'zw_Open_Main_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_Open_TeX'zw_Open_TeX_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID
 insertline 'zw_TeX_subject'zw_TeX_subj_inf'zwepmtex', ActionsList_FileID.last+1, ActionsList_FileID


/* Get file's subject, return it in subj. Returns 0 if subject not found or non-ASCII */

DEFPROC get_subject( var subj ) =
found = find_ea( '.SUBJECT', ea_seg, ea_ofs, ea_ptr1, ea_ptr2, ea_len, ea_entrylen, ea_valuelen )
if not found | ea_valuelen = 0 then found = 0      -- no subject
elseif peek( ea_seg, ea_ptr2, 2) = EAT_ASCII then  -- ASCII subject
  subj = peek( ea_seg, ea_ptr2+4, min(ltoa(peek(ea_seg, ea_ptr2+2, 2)\0\0, 10), MAXCOL))
else found = 0                                     -- non-ASCII subject
endif
return found

/* Checks the file's subject, returns the value of 'MAIN=' or nothing */

DEFPROC get_main_tex(drv, path) =
found = get_subject(subj)
if not found then return ''; endif
parse value subj with kw '=' val
if upcase(kw) = 'MAIN' then
  ret = val
  call split_file_name(drive, dir, base, ext, ret)
  if drive = '' & dir = '' then
    ret = drv || path || base || ext
  endif
else ret = ''
endif
return ret

/* Split file name into parts */

DEFPROC split_file_name(var drive, var path, var base, var ext, filespec) =
fn = strip(filespec)
if substr(fn, 2, 1) = ':' then
  drive = substr(fn, 1, 2)
  fn = substr(fn, 3)
else drive = ''
endif
dotpos = lastpos('.', fn)
bspos = lastpos('\', fn)
if dotpos <= bspos then dotpos = length(fn) + 1; endif
path = substr(fn, 1, bspos)
base = substr(fn, bspos+1, dotpos-bspos-1)
ext = substr(fn, dotpos)
return

/* Quit the log file, save the curcor position */

DEFPROC quit_log( logfile ) =
universal lp, log_pos
lp = -1
while is_in_ring( logfile, lgid ) do
  activatefile( lgid )
  .modify = 0
  lp = .line
  call psave_pos( log_pos )
  call quitfile()
endwhile

/* Refresh log file */

DEFPROC refresh_log( logfile ) =
universal lp, log_pos
quit_log( logfile )
qlp = lp;   lp = -1
if exist( logfile ) then
  'xcom edit /d' logfile
  getfileid lgid, logfile
  activatefile( lgid )
  if qlp > 0 & qlp <= .last then call prestore_pos( log_pos )
  elseif qlp > .last then bottom
  endif
else
  call ErrorBox('Refresh error', 'File "' || logfile || '" not found')
endif

/* This command invokes TeXme.cmd */

DEFC zw_TeXme
parse arg action o1 o2
if action = 'S' then    -- button initialized
  sayerror 0
  command = '';   opt = ''
  /* command = current -> save current file, command = all -> save all files in ring */
  call parse_TeX_opt(command, opt, o1)
  call parse_TeX_opt(command, opt, o2)
  call run_TeX_me(command, opt)
else handle_action(action, 'Runs TeXme.cmd', zw_TeXme_inf, Hmsg)
endif

/* Parse TeX options */

DEFPROC parse_TeX_opt(var command, var opt, o) =
cc = upcase(o)
if o = '-' then opt = '-'
elseif o = '+' then opt = '+'
elseif abbrev('CURRENT', cc, 3) | abbrev('ALL', cc, 1) then command = o
endif

/* Executes TeXme.cmd */

DEFPROC run_TeX_me(command, opt) =
filespec = .filename
call split_file_name( drive, dir, base, ext, filespec )
if upcase(ext) = '.LOG' then logfile = filespec
else
  mainfile = get_main_tex(drive, dir)
  if mainfile <> '' then
    call split_file_name( drive, dir, base, ext, mainfile )
  endif
  logfile = drive || dir || base || '.log'
endif

/* Now logfile contains the name of the log-file to be open later. We close it. */
quit_log( logfile )

/* Check the command argument, case insensitive */
sv = upcase( command )
if filespec <> logfile then
  if abbrev( 'CURRENT', sv, 3 ) | abbrev( 'ALL', sv, 1 ) then
    getfileid fid, filespec
    activatefile( fid )
    if .modify then call savefile(.filename); endif
  endif
endif
if abbrev( 'ALL', sv, 1 ) then
  call save_the_ring()
endif

/* Execute the command */

OS2cmd 'TeXme' filespec opt

/* Opening the log-file impossible because OS2cmd is asynchronous

'xcom edit /d' logfile
getfileid lgid, logfile
activatefile( lgid )
if lp > 0 & lp < .last then call prestore_pos( log_pos ); endif

*/


/* This procedure saves all files in the ring */

DEFPROC save_the_ring() =
n = filesinring()
getfileid fid
for i = 1 to n
  if .modify then call savefile(.filename); endif
  nextfile
endfor
activatefile( fid )

/* This is the command front end */

DEFC zw_Save_All
parse arg action rest
if action = 'S' then
  sayerror 0
  call save_the_ring()
else call handle_action(action, 'Save all files', zw_Save_All_inf, Hmsg)
endif

/* This command runs preview */

DEFC zw_View
parse arg action cmdline
if action = 'S' then
  sayerror 0
  call dvi_command( 'vp.cmd', cmdline)
else call handle_action(action, 'Preview', zw_View_inf, Hmsg)
endif

/* This command executes any command on the DVI-file */

DEFC zw_dvi_command
parse arg action pgm cmdline
if action = 'S' then
  sayerror 0
  if pgm = '' then
    call ErrorBox('DVI Command', 'No program action specified')
  else call dvi_command( pgm, cmdline)
  endif
else call handle_action(action, 'Execute dvi command', zw_dvi_com_inf, Hmsg)
endif

/* This procedure executes the DVI command */

DEFPROC dvi_command(pgm, cmdline) =
/* First we get the main file name without extension */
fn = strip(.filename)
split_file_name( drive, dir, base, ext, fn)
if upcase(ext) <> '.LOG' then
  mainfile = get_main_tex(drive, dir)
  if mainfile <> '' then
    call split_file_name( drive, dir, base, ext, mainfile )
  endif
endif

/* chdir will change the current directory */

if drive <> '' | dir <> '' then
  if drive = '' then
    drive = 'echo'  -- drive must contain a valid command, 'echo' is harmless
  endif
  if dir = '' then path = '.'
  else
    path = dir
    bs = lastpos('\', dir)
    if bs > 1 & bs = length(dir) then path = substr(dir, 1, bs-1);  endif
  endif
  chdir = 'TeXchdir.cmd' drive path
  dvifile = base
else
  chdir = ''
  dvifile = drive || dir || base
endif

/* Each * in the cmdline will be replaced with dvifile */

j = pos( '*', cmdline )
while j do
  cmdline = substr( cmdline, 1, j-1) || dvifile || substr( cmdline ,j+1)
  j = pos( '*', cmdline )
endwhile

/* Execute the command asynchromously */

OS2cmd chdir pgm cmdline


/* This command opens the log-file */

DEFC zw_Open_Log
parse arg action rest
if action = 'S' then
  sayerror 0
  /* Get the main file name */
  fn = strip( .filename )
  call split_file_name( drive, dir, base, ext, fn )
  if upcase(ext) = '.LOG' then logfile = fn
  else
    mainfile = get_main_tex(drive, dir)
    if mainfile <> '' then
      call split_file_name( drive, dir, base, ext, mainfile )
    endif
    logfile = drive || dir || base || '.log'
  endif
  call refresh_log( logfile )
else call handle_action(action, 'Open log-file', zw_Open_Log_inf, Hmsg)
endif

/* Refresh file */

DEFPROC refresh_file( fn ) =
ch = AskYesNo( 'About to refresh ' || fn)
if ch <> 'Y' then return; endif
lp = -1
latefid = -1
while is_in_ring( fn, fid ) do
  activatefile( fid )
  lp = .line
  call psave_pos( fn_pos )
  .modify = 0
  if filesinring(0) = 1 then
    latefid = fid;  leave
  else call quitfile()
  endif
endwhile
'xcom edit /d' fn
if lp > 0 & lp <= .last then call prestore_pos( fn_pos )
elseif lp > .last then bottom
endif
if latefid >= 0 then
  activatefile( latefid )
  .modify = 0
  call quitfile()
endif

DEFC zw_Refresh
parse arg action rest
if action = 'S' then
  sayerror 0
  call refresh_file( .filename )
else call handle_action(action, 'Refresh', zw_Refresh_inf, Hmsg)
endif

/* Open the current view of a file or load it */

DEFPROC open_it( fn ) =
if is_in_ring( fn, fid ) then activatefile( fid )
else 'xcom edit' fn
endif

/* Open the main.tex */

DEFC zw_Open_Main
parse arg action rest
if action = 'S' then
  sayerror 0
  fn = strip(.filename)
  call split_file_name( drive, dir, base, ext, fn )
  mainfile = get_main_TeX(drive, dir)
  if mainfile <> '' then
    call split_file_name( drive, dir, base, ext, mainfile )
    if ext = '' then
      fn = drive || dir || base || '.tex'
    else
      fn = mainfile
    endif
  else fn = drive || dir || base || '.tex'
  endif
  call open_it( fn )
else call handle_action(action, 'Open main TeX file', zw_Open_Main_inf, Hmsg)
endif

/* If the file exists, it is open, otherwise the list of extensions is checked */

DEFPROC open_if_exists_w_x( fn, extns, drv, path ) =
found = 1
fnx = fn
if exist( fnx ) then call open_it( fnx )
else
  call split_file_name(drive, dir, base, ext, fnx)
  found = 0;    pathused = 0
  if drive <> '' | dir <> '' then pathused = 1;   endif
  loop
    for i = 1 to words( extns )
      f = fnx || word( extns, i)
      if exist( f ) then
        found = 1;   call open_it( f )
        leave
      endif
    endfor
    if found | pathused then leave;  endif
    if not pathused then
      pathused = 1
      fnx = drv || path || base || ext
      if exist( fnx ) then
        call open_it( fnx )
        found = 1
      endif
    endif
  endloop
  if not found then call ErrorBox('File Open Error', 'File "' || fn || '" not found');  endif
endif
return found

/* Open TeX input file the name of which is under cursor */

DEFC zw_Open_TeX
parse arg action rest
if action = 'S' then
  sayerror 0
  call open_TeX()
else call handle_action(action, 'Open TeX file under cursor', zw_Open_TeX_inf, Hmsg)
endif

/* This is the proc */

DEFPROC open_TeX() =
call split_file_name(drv, dir, base, ext, .filename)
getline curline, .line
col = .col
ch = substr( curline, col, 1)
wrong = " "\t"()[]{},;?*<>""'"
while col > 0 & pos( ch, wrong ) = 0 do
  col = col - 1
  ch = substr( curline, col, 1)
endwhile
stc = col + 1
col = .col
ch = substr( curline, col, 1)
len = length( curline )
while col <= len & pos( ch, wrong ) = 0 do
  col = col + 1
  ch = substr( curline, col, 1)
endwhile
enc = col - 1
if stc > enc then call ErrorBox('Error', 'Invalid file name under cursor')
else
  fn = substr( curline, stc, enc - stc + 1 )
  ps = pos('/', fn)
  while ps do
    fn = substr( fn, 1, ps-1) || '\' || substr( fn, ps+1)
    ps = pos('/', fn)
  endwhile
  call open_if_exists_w_x(fn, TeXextns, drv, dir)
endif

/* Sets or displays TeX subject of the current file */

DEFPROC TeX_subject() =
universal TeX_subj_fld
answer = listbox( 'TeX subject view/set', TeX_subj_fld,
  '/~Set/~Cancel/~Other/~View', 0, 0, 10, 40)
if answer = '' then return;  endif
if answer = 4 then /* view the subject */
  answer = get_subject(subj)
  if answer then
    current_subject = 'The subject is:' subj
  else
    current_subject = 'The file has no subject'
  endif
  call InfoBox('TeX subject', current_subject)
else
  if answer = 3 then
    default = 'MAIN='
    if length( TeX_subj_fld ) > 1 then
      parse value TeX_subj_fld with '' default '' rest
    endif
    answer = entrybox('Set TeX subject', '/~Set/~Cancel', default, 40, 80)
  endif
  answer = strip( answer )
  pa = pos( '=', answer )
  if pa then
    parse value answer with kw '=' val
    if val = '' then pa = 0
    else
      val = upcase( kw )
      if kw <> 'MAIN' & kw <> 'FORMAT' then pa = 0;  endif
    endif
  endif
  if pa = 0 then
    if not AskYesNo('About to set unsupported subject: ' || answer) then
      answer = ''
    endif
  endif
  if answer <> '' then
    'subject' answer
    call savefile(.filename)
    pa = pos( \1 || answer || \1, TeX_subj_fld )
    if pa > 1 then
      TeX_subj_fld = \1 || answer || substr( TeX_subj_fld, 1, pa) ||
          substr( TeX_subj_fld, pa + length( answer ) + 1)
    elseif pa = 0 then
      TeX_subj_fld = \1 || answer || TeX_subj_fld
      if length( TeX_subj_fld) > max_TeX_subj then
        CutAt = lastpos( \1, TeX_subj_fld, max_TeX_subj )
        if CutAt > 1 then TeX_subj_fld = substr( TeX_subj_fld, 1, CutAt );  endif
      endif
    endif
  endif
endif

/* Here is the front end for the above procedure */

DEFC zw_TeX_subject
parse arg action rest
if action = 'S' then
  sayerror 0
  call TeX_subject()
else call handle_action(action, 'Set/View TeX subject', zw_TeX_subj_inf, Hmsg)
endif

/* Message box */

DEFPROC MsgBox( caption, text, attr) =
return WinMessageBox( caption, text, MB_OK + MB_MOVEABLE + attr)

/* Information box */

DEFPROC InfoBox( caption, text) =
return MsgBox( caption, text, MB_INFORMATION )

/* ErrorBox */

DEFPROC ErrorBox( caption, text) =
Beep(512, 150)
return MsgBox( caption, text, MB_ERROR)

/* Debug box */

DEFPROC DebugBox( text ) =
call MsgBox( 'Debugging', text, MB_NOICON)

/* Action handling */

DEFPROC handle_action( action, Itext, Htext, Htype ) =
if action = 'I' then
  display -8
  sayerror Itext
  display 8
elseif action = 'H' then
  if Htype = Hmsg then call InfoBox('Help window', Htext)
  elseif Htype = Hinf then zw_help Htext
  else ErrorBox('Help Error', 'Invalid help panel for: ' || Itext)
  endif
elseif action = 'E' then
  -- Do nothing
else ErrorBox("Error", 'This action is not yet implemented')
endif

/* Similar to already_in_ring which ceased to work... */

DEFPROC is_in_ring( fname, var fid ) =
getfileid fid, fname
return fid <> ''



EA_comment 'This file supports TeX actions on the EPM 6.03 toolbar'
