/* SIMPLE - Simple Is a Macro Processing Language Element */
/* Copyright (c) 1998 David A. Madore (david.madore@ens.fr) */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file contains the definitions of the builtin functions.
 */

#include <stdio.h>
#include <stdlib.h>

#include "simple.h"
#include "token.h"
#include "stack.h"

int
read_integer(token_list *tl)
{
  char *tmp;
  int i;
  tmp=(char *)token_list_to_string(tl);
  if (sscanf(tmp,"%i",&i)<1)
    fatal(0,"This is not an integer: %s",tmp);
  FREE(tmp);
  return i;
}

void
write_integer(int i,token_list *tl)
{
  char *tmp;
  int j;
  tmp=malloc(30);
#if defined(HAVE_SNPRINTF)
  if (snprintf(tmp,30,"%d",i)==30)
    fatal(1,"Ooops! not enough characters...");
#else
  sprintf(tmp,"%d",i);
#endif
  for (j=0;tmp[j];j++)
    append_token(tl,tmp[j]);
  FREE(tmp);
}

void
bi_identity(token_list *expansion,int argc,token_list *argv)
{
  if (argc>=2)
    copy_list(expansion,argv+1);
}

void
bi_void(token_list *expansion,int argc,token_list *argv)
{
}

void
bi_output(token_list *expansion,int argc,token_list *argv)
{
  int i;
  if (argc<2) return;
  for (i=0;i<argv[1].size;i++)
    output_token(argv[1].l[i]);
}

void
bi_define(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  command todef;
  if (argc!=3)
    fatal(0,"<def> expects two arguments.");
  name=(char *)token_list_to_string(argv+1);
  todef=identify_command(name);
  FREE(name);
  if (todef<0)
    fatal(0,"Trying to redefine a builtin command.");
  define_user(argv+2,todef);
}

void
bi_include(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  if (argc!=2)
    fatal(0,"<include> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  push_input_file(name);
  FREE(name);
}

void
bi_it(token_list *expansion,int argc,token_list *argv)
{
  call_macro(expansion,argc-1,argv+1);
  /* Perhaps we should do this nonrecursively - but then, one is
   * simply _not_ supposed to call <|||||||||||||id|Hello!>... */
}

void
bi_quote(token_list *expansion,int argc,token_list *argv)
{
  int i;
  if (argc!=2)
    fatal(0,"<quote> expects one argument.");
  ensure_length(expansion,argv[1].size*2);
  expansion->size=argv[1].size*2;
  for (i=0;i<argv[1].size;i++) {
    expansion->l[2*i]=QUOTE_NEXT;
    expansion->l[2*i+1]=argv[1].l[i];
  }
}

void
bi_dquote(token_list *expansion,int argc,token_list *argv)
{
  int i;
  if (argc!=2)
    fatal(0,"<dquote> expects one argument.");
  ensure_length(expansion,argv[1].size*4);
  expansion->size=argv[1].size*4;
  for (i=0;i<argv[1].size;i++) {
    expansion->l[4*i]=QUOTE_NEXT;
    expansion->l[4*i+1]=QUOTE_NEXT;
    expansion->l[4*i+2]=QUOTE_NEXT;
    expansion->l[4*i+3]=argv[1].l[i];
  }
}

void
bi_inputform(token_list *expansion,int argc,token_list *argv)
     /* Warning!  Use of this function is deprecated. */
{
  token tk;
  int i;
  if (argc!=2)
    fatal(0,"<inputform> expects one argument.");
  for (i=0;i<argv[1].size;i++) {
    tk=argv[1].l[i];
    if (tk<FIRST_SPECIAL) {
      if ((tk=='`')||(tk=='"')||(tk=='%')||(tk=='<')
	  ||(tk=='>')||(tk=='|')||(tk=='[')||(tk==']')
	  ||(tk=='#')||(tk=='@'))
	append_token(expansion,'`');
      append_token(expansion,tk);
    } else if (tk==BEGIN_COMMAND) append_token(expansion,'<');
    else if (tk==END_COMMAND) append_token(expansion,'>');
    else if (tk==NEXT_PARAM) append_token(expansion,'|');
    else if (tk==OPEN_QUOTE) append_token(expansion,'[');
    else if (tk==CLOSE_QUOTE) append_token(expansion,']');
    else if (tk==QUOTE_NEXT) append_token(expansion,'#');
    else if (tk==AT_SIGN) append_token(expansion,'@');
  }
}

void
bi_if(token_list *expansion,int argc,token_list *argv)
{
  int i;
  if (argc%3==0)
    fatal(0,"<if> can't deal with %d parameters.",argc-1);
  for (i=0;3*i+3<argc;i++) {
    if (compare_list(argv+(3*i+1),argv+(3*i+2))) {
      copy_list(expansion,argv+(3*i+3));
      return;
    }
  }
  if (3*i+1<argc) {
    copy_list(expansion,argv+(3*i+1));
  } /* Otherwise expansion is empty */
}

void
bi_defof(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  command cmd;
  if (argc!=2)
    fatal(0,"<defof> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  cmd=identify_command(name);
  FREE(name);
  if (cmd<0)
    fatal(0,"Trying to expand the definition of a builtin.");
  def_of_user(expansion,cmd);
}

void
bi_qdefof(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  command cmd;
  token_list def;
  int i;
  if (argc!=2)
    fatal(0,"<qdefof> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  cmd=identify_command(name);
  FREE(name);
  if (cmd<0)
    fatal(0,"Trying to expand the definition of a builtin.");
  make_empty_list(&def);
  def_of_user(&def,cmd);
  ensure_length(expansion,def.size*2);
  expansion->size=def.size*2;
  for (i=0;i<def.size;i++) {
    expansion->l[2*i]=QUOTE_NEXT;
    expansion->l[2*i+1]=def.l[i];
  }
  free_list(&def);
}

void
bi_quit(token_list *expansion,int argc,token_list *argv)
{
  output_token(EOF); /* The proper way to quit. */
}

void
bi_error(token_list *expansion,int argc,token_list *argv)
{
  char *errmsg;
  if (argc>=2) {
    errmsg=(char *)token_list_to_string(argv+1);
    fatal(0,"%s",errmsg);
    FREE(errmsg); /* This doesn't get executed, of course. */
  } else
    fatal(0,"User error.");
}

void
bi_head(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<head> expects one argument.");
  if (argv[1].size==0)
    fatal(0,"<head> of an empty list is invalid.");
  append_token(expansion,argv[1].l[0]);
}

void
bi_tail(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<tail> expects one argument.");
  if (argv[1].size==0)
    fatal(0,"<tail> of an empty list is invalid.");
  copy_list(expansion,argv+1);
  remove_first(expansion);
}

void
bi_ahead(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<ahead> expects one argument.");
  if (argv[1].size==0)
    fatal(0,"<ahead> of an empty list is invalid.");
  append_token(expansion,argv[1].l[argv[1].size-1]);
}

void
bi_atail(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<atail> expects one argument.");
  if (argv[1].size==0)
    fatal(0,"<atail> of an empty list is invalid.");
  copy_list(expansion,argv+1);
  remove_last(expansion);
}

void
bi_len(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<len> expects one argument.");
  write_integer(argv[1].size,expansion);
}

void
bi_push(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  token_stack *ts;
  int i;
  if (argc<=2)
    fatal(0,"<push> expects at least one argument.");
  name=(char *)token_list_to_string(argv+1);
  ts=identify_stack(name);
  FREE(name);
  for (i=2;i<argc;i++)
    push_on_stack(ts,argv+i);
}

void
bi_last(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  token_stack *ts;
  if (argc!=2)
    fatal(0,"<last> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  ts=identify_stack(name);
  FREE(name);
  last_from_stack(ts,expansion);
}

void
bi_pop(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  token_stack *ts;
  if (argc!=2)
    fatal(0,"<pop> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  ts=identify_stack(name);
  FREE(name);
  pop_from_stack(ts);
}

void
bi_poplast(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  token_stack *ts;
  if (argc!=2)
    fatal(0,"<poplast> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  ts=identify_stack(name);
  FREE(name);
  last_from_stack(ts,expansion);
  pop_from_stack(ts);
}

void
bi_depth(token_list *expansion,int argc,token_list *argv)
{
  char *name;
  token_stack *ts;
  if (argc!=2)
    fatal(0,"<depth> expects one argument.");
  name=(char *)token_list_to_string(argv+1);
  ts=identify_stack(name);
  FREE(name);
  write_integer(stack_depth(ts),expansion);
}

void
bi_plus(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  j=0;
  for (i=1;i<argc;i++)
    j+=read_integer(argv+i);
  write_integer(j,expansion);
}

void
bi_minus(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<-> expects two arguments.");
  write_integer(read_integer(argv+1)-read_integer(argv+2),expansion);
}

void
bi_times(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  j=1;
  for (i=1;i<argc;i++)
    j*=read_integer(argv+i);
  write_integer(j,expansion);
}

void
bi_div(token_list *expansion,int argc,token_list *argv)
{
  int d;
  if (argc!=3)
    fatal(0,"<div> expects two arguments.");
  d=read_integer(argv+2);
  if (d==0)
    fatal(0,"Division by zero.");
  write_integer(read_integer(argv+1)/d,expansion);
}

void
bi_mod(token_list *expansion,int argc,token_list *argv)
{
  int d;
  if (argc!=3)
    fatal(0,"<mod> expects two arguments.");
  d=read_integer(argv+2);
  if (d==0)
    fatal(0,"Division by zero.");
  write_integer(read_integer(argv+1)%d,expansion);
}

void
bi_eq(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<eq> expects two arguments.");
  write_integer(read_integer(argv+1)==read_integer(argv+2),expansion);
}

void
bi_neq(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<neq> expects two arguments.");
  write_integer(read_integer(argv+1)!=read_integer(argv+2),expansion);
}

void
bi_gt(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<gt> expects two arguments.");
  write_integer(read_integer(argv+1)>read_integer(argv+2),expansion);
}

void
bi_ge(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<ge> expects two arguments.");
  write_integer(read_integer(argv+1)>=read_integer(argv+2),expansion);
}

void
bi_lt(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<lt> expects two arguments.");
  write_integer(read_integer(argv+1)<read_integer(argv+2),expansion);
}

void
bi_le(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=3)
    fatal(0,"<le> expects two arguments.");
  write_integer(read_integer(argv+1)<=read_integer(argv+2),expansion);
}

void
bi_and(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  j=1;
  for (i=1;i<argc;i++)
    j=j&&read_integer(argv+i);
  write_integer(j,expansion);
}

void
bi_or(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  j=0;
  for (i=1;i<argc;i++)
    j=j||read_integer(argv+i);
  write_integer(j,expansion);
}

void
bi_not(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<not> expects one arguments.");
  write_integer(!read_integer(argv+1),expansion);
}

void
bi_band(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  j=-1;
  for (i=1;i<argc;i++)
    j=j&read_integer(argv+i);
  write_integer(j,expansion);
}

void
bi_bor(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  j=0;
  for (i=1;i<argc;i++)
    j=j|read_integer(argv+i);
  write_integer(j,expansion);
}

void
bi_bnot(token_list *expansion,int argc,token_list *argv)
{
  if (argc!=2)
    fatal(0,"<bnot> expects one arguments.");
  write_integer(~read_integer(argv+1),expansion);
}

void
bi_cartype_ordinary(token_list *expansion,int argc,token_list *argv)
{
  unsigned char *tmp,*tmp2;
  if (argc!=2)
    fatal(0,"<cartype_ordinary> expects one argument.");
  tmp=token_list_to_string(argv+1);
  tmp2=tmp;
  while (*tmp) {
    syntax_cartype_ordinary(*tmp);
    tmp++;
  }
  FREE(tmp2);
}

void
bi_cartype_escape_next(token_list *expansion,int argc,token_list *argv)
{
  unsigned char *tmp,*tmp2;
  if (argc!=2)
    fatal(0,"<cartype_escape_next> expects one argument.");
  tmp=token_list_to_string(argv+1);
  tmp2=tmp;
  while (*tmp) {
    syntax_cartype_escape_next(*tmp);
    tmp++;
  }
  FREE(tmp2);
}

void
bi_cartype_escape_string(token_list *expansion,int argc,token_list *argv)
{
  unsigned char *tmp,*tmp2;
  if (argc!=2)
    fatal(0,"<cartype_escape_string> expects one argument.");
  tmp=token_list_to_string(argv+1);
  tmp2=tmp;
  while (*tmp) {
    syntax_cartype_escape_string(*tmp);
    tmp++;
  }
  FREE(tmp2);
}

void
bi_cartype_comment(token_list *expansion,int argc,token_list *argv)
{
  unsigned char *tmp,*tmp2;
  if (argc!=2)
    fatal(0,"<cartype_comment> expects one argument.");
  tmp=token_list_to_string(argv+1);
  tmp2=tmp;
  while (*tmp) {
    syntax_cartype_comment(*tmp);
    tmp++;
  }
  FREE(tmp2);
}

void
bi_cartype_active(token_list *expansion,int argc,token_list *argv)
{
  unsigned char *tmp,*tmp2;
  if (argc!=3)
    fatal(0,"<cartype_active> expects two argument.");
  tmp=token_list_to_string(argv+1);
  tmp2=tmp;
  while (*tmp) {
    syntax_cartype_active(*tmp,argv+2);
    tmp++;
  }
  FREE(tmp2);
}

void
bi_translate(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  if (argc!=3)
    fatal(0,"<translate> expects two arguments.");
  if (argv[1].size%2)
    fatal(0,"The first argument to <translate> should be of even length.");
  copy_list(expansion,argv+2);
  for (i=0;i<expansion->size;i++) {
    for (j=0;j<argv[1].size;j+=2) {
      if (expansion->l[i]==argv[1].l[j]) {
	expansion->l[i]=argv[1].l[j+1];
	break;
      }
    }
  }
}

void
bi_find(token_list *expansion,int argc,token_list *argv)
{
  int i,j;
  char found,missed;
  if (argc!=3)
    fatal(0,"<find> expects two arguments.");
  found=0;
  for (i=0;i<=argv[1].size-argv[2].size;i++) {
    missed=0;
    for (j=0;j<argv[2].size;j++)
      if (argv[1].l[i+j]!=argv[2].l[j]) {
	missed=1;
	break;
      }
    if (!missed) {
      found=1;
      break;
    }
  }
  if (found) {
    write_integer(i,expansion);
  } /* Otherwise expansion is empty */
}

void
bi_substr(token_list *expansion,int argc,token_list *argv)
{
  int pos,len,i;
  if (argc!=4)
    fatal(0,"<substr> expects three arguments.");
  pos=read_integer(argv+2);
  len=read_integer(argv+3);
  if (pos>=argv[1].size) return;
  if (pos+len>argv[1].size) len=argv[1].size-pos;
  ensure_length(expansion,len);
  expansion->size=len;
  for (i=0;i<len;i++)
    expansion->l[i]=argv[1].l[pos+i];
}

static struct {
  command who;
  void (* what)(token_list *,int,token_list *);
} builtin_table[] = {
  {BI_IDENTITY, bi_identity},
  {BI_VOID, bi_void},
  {BI_OUTPUT, bi_output},
  {BI_DEFINE, bi_define},
  {BI_INCLUDE, bi_include},
  {BI_IT, bi_it},
  {BI_QUOTE, bi_quote},
  {BI_DQUOTE, bi_dquote},
  {BI_INPUTFORM, bi_inputform},
  {BI_IF, bi_if},
  {BI_DEFOF, bi_defof},
  {BI_QDEFOF, bi_qdefof},
  {BI_QUIT, bi_quit},
  {BI_ERROR, bi_error},
  {BI_HEAD, bi_head},
  {BI_TAIL, bi_tail},
  {BI_AHEAD, bi_ahead},
  {BI_ATAIL, bi_atail},
  {BI_LEN, bi_len},
  {BI_PUSH, bi_push},
  {BI_LAST, bi_last},
  {BI_POP, bi_pop},
  {BI_POPLAST, bi_poplast},
  {BI_DEPTH, bi_depth},
  {BI_PLUS, bi_plus},
  {BI_MINUS, bi_minus},
  {BI_TIMES, bi_times},
  {BI_DIV, bi_div},
  {BI_MOD, bi_mod},
  {BI_EQ, bi_eq},
  {BI_NEQ, bi_neq},
  {BI_GT, bi_gt},
  {BI_GE, bi_ge},
  {BI_LT, bi_lt},
  {BI_LE, bi_le},
  {BI_AND, bi_and},
  {BI_OR, bi_or},
  {BI_NOT, bi_not},
  {BI_BAND, bi_band},
  {BI_BOR, bi_bor},
  {BI_BNOT, bi_bnot},
  {BI_CARTYPE_ORDINARY, bi_cartype_ordinary},
  {BI_CARTYPE_ESCAPE_NEXT, bi_cartype_escape_next},
  {BI_CARTYPE_ESCAPE_STRING, bi_cartype_escape_string},
  {BI_CARTYPE_COMMENT, bi_cartype_comment},
  {BI_CARTYPE_ACTIVE, bi_cartype_active},
  {BI_TRANSLATE, bi_translate},
  {BI_FIND, bi_find},
  {BI_SUBSTR, bi_substr},
  {0, NULL}
};

void
call_builtin(token_list *expansion,command cmd,int argc,token_list *argv)
     /* Expand builtin token tk looking up its definition, and put the
      * result in expansion, the number of arguments being argc and
      * their values given by argv. */
{
  int i;
  for (i=0;builtin_table[i].who;i++) {
    if (cmd==builtin_table[i].who) {
      builtin_table[i].what(expansion,argc,argv);
      return;
    }
  }
  fatal(1,"Unresolved builtin!");
}
