//========================================================================
//
// GPGMESignatureHandler.cc
//
// This file is licensed under the GPLv2 or later
//
// Copyright 2023 g10 Code GmbH, Author: Sune Stolborg Vuorela <sune@vuorela.dk>
//========================================================================
#include "GPGMESignatureHandler.h"
#include "DistinguishedNameParser.h"
#include <gpgme.h>
#include <gpgme++/key.h>
#include <gpgme++/gpgmepp_version.h>
#include <gpgme++/signingresult.h>
#include <gpgme++/engineinfo.h>

#if (GPGMEPP_VERSION_MINOR < 19)
#    error "Too old gpgme"
#endif

// TODO remove commented debug stuff

// TODO remove
#include <iostream>

bool GpgSignatureBackend::hasSufficientVersion()
{
    // gpg 2.4.0 does not support padded signatures.
    // Most gpg signatures are padded. This is fixed for 2.4.1
    // gpg 2.4.0 does not support generating signatures
    // with definite lengths. This will hopefully be fixed soon
    // No need to proceed if we have too old versions.
    auto context = GpgME::Context::create(GpgME::CMS);
    return context->engineInfo().engineVersion() > GpgME::EngineInfo::Version("2.4.0");
}

/// GPGME helper methods

// gpgme++'s string-like functions returns char pointers that can be nullptr
// Creating std::string from nullptr is, depending on c++ standards versions
// either undefined behavior or illegal, so we need a helper.

static std::string fromCharPtr(const char *data)
{
    if (data) {
        return std::string { data };
    }
    return {};
}

static X509CertificateInfo::Validity getValidityFromSubkey(const GpgME::Subkey &key)
{
    X509CertificateInfo::Validity validity;
    validity.notBefore = key.creationTime();
    validity.notAfter = key.expirationTime();
    return validity;
}

static X509CertificateInfo::EntityInfo getEntityInfoFromKey(std::string_view dnString)
{
    auto dn = DN::parseString(dnString);
    X509CertificateInfo::EntityInfo info;
    info.commonName = DN::FindFirstValue(dn, "CN").value_or(std::string {});
    info.organization = DN::FindFirstValue(dn, "O").value_or(std::string {});
    info.email = DN::FindFirstValue(dn, "EMAIL").value_or(std::string {});
    info.distinguishedName = std::string { dnString };
    return info;
}

static std::unique_ptr<X509CertificateInfo> getCertificateInfoFromKey(const GpgME::Key &key)
{
    auto certificateInfo = std::make_unique<X509CertificateInfo>();
    certificateInfo->setIssuerInfo(getEntityInfoFromKey(fromCharPtr(key.issuerName())));
    certificateInfo->setSerialNumber(GooString { DN::detail::parseHexString(key.issuerSerial()).value_or("") });
    auto subjectInfo = getEntityInfoFromKey(fromCharPtr(key.userID(0).id()));
    if (subjectInfo.email.empty()) {
        subjectInfo.email = fromCharPtr(key.userID(1).email());
    }
    certificateInfo->setSubjectInfo(std::move(subjectInfo));
    certificateInfo->setValidity(getValidityFromSubkey(key.subkey(0)));
    certificateInfo->setNickName(GooString(fromCharPtr(key.primaryFingerprint())));
    X509CertificateInfo::PublicKeyInfo pkInfo;
    pkInfo.publicKeyStrength = key.subkey(0).length();
    switch (key.subkey(0).publicKeyAlgorithm()) {
    case GpgME::Subkey::AlgoDSA:
        pkInfo.publicKeyType = DSAKEY;
        break;
    case GpgME::Subkey::AlgoECC:
    case GpgME::Subkey::AlgoECDH:
    case GpgME::Subkey::AlgoECDSA:
    case GpgME::Subkey::AlgoEDDSA:
        pkInfo.publicKeyType = ECKEY;
        break;
    case GpgME::Subkey::AlgoRSA:
    case GpgME::Subkey::AlgoRSA_E:
    case GpgME::Subkey::AlgoRSA_S:
        pkInfo.publicKeyType = RSAKEY;
        break;
    case GpgME::Subkey::AlgoELG:
    case GpgME::Subkey::AlgoELG_E:
    case GpgME::Subkey::AlgoMax:
    case GpgME::Subkey::AlgoUnknown:
        pkInfo.publicKeyType = OTHERKEY;
    }
    auto ctx = GpgME::Context::create(GpgME::CMS);
    GpgME::Data pubkeydata;
    ctx->exportPublicKeys(key.primaryFingerprint(), pubkeydata);
    certificateInfo->setCertificateDER(GooString(pubkeydata.toString()));

    certificateInfo->setPublicKeyInfo(std::move(pkInfo));

    certificateInfo->setKeyUsageExtensions([&key]() {
        // this function is kind of a hack. GPGSM collapses multiple
        // into one bit, so trying to match it back can never be good
        int kue = 0;
        if (key.canReallySign()) {
            kue |= KU_NON_REPUDIATION;
            kue |= KU_DIGITAL_SIGNATURE;
        }
        if (key.canEncrypt()) {
            kue |= KU_KEY_ENCIPHERMENT;
            kue |= KU_DATA_ENCIPHERMENT;
        }
        if (key.canCertify()) {
            kue |= KU_KEY_CERT_SIGN;
        }
        return kue;
    }());

    //    std::cout << "CERTINFO" << dump(*certificateInfo) << std::endl;

    return certificateInfo;
}

/// implementation of header file

GpgSignatureBackend::GpgSignatureBackend()
{
    GpgME::initializeLibrary();
}

std::unique_ptr<CryptoSign::SigningInterface> GpgSignatureBackend::createSigningHandler(const std::string &certID, HashAlgorithm digestAlgTag)
{
    return std::make_unique<GpgSignatureCreation>(certID, digestAlgTag);
}

std::unique_ptr<CryptoSign::VerificationInterface> GpgSignatureBackend::createVerificationHandler(std::vector<unsigned char> &&pkcs7)
{
    return std::make_unique<GpgSignatureVerification>(std::move(pkcs7));
}

std::vector<std::unique_ptr<X509CertificateInfo>> GpgSignatureBackend::getAvailableSigningCertificates()
{
    std::vector<std::unique_ptr<X509CertificateInfo>> certificates;
    auto context = GpgME::Context::create(GpgME::CMS);
    auto err = context->startKeyListing(static_cast<const char *>(nullptr), true /*secretOnly*/);
    while (!err) {
        auto key = context->nextKey(err);
        if (!key.isNull() && !err) {
            if (key.isBad()) {
                continue;
            }
            if (!key.canReallySign()) {
                continue;
            }
            certificates.push_back(getCertificateInfoFromKey(key));
        } else {
            break;
        }
    }
    return certificates;
}

GpgSignatureCreation::GpgSignatureCreation(const std::string &certId, HashAlgorithm digestAlgTag) : gpgContext { GpgME::Context::create(GpgME::CMS) }
{
    GpgME::Error error;
    auto signingKey = gpgContext->key(certId.c_str(), error, true);
    if (!error) {
        gpgContext->addSigningKey(signingKey);
        key = signingKey;
    }
}

void GpgSignatureCreation::addData(unsigned char *dataBlock, int dataLen)
{
    gpgData.write(dataBlock, dataLen);
}
std::optional<GooString> GpgSignatureCreation::signDetached(const std::string &password)
{
    if (!key) {
        return {};
    }
    gpgData.rewind();
    GpgME::Data signatureData;
    auto signingResult = gpgContext->sign(gpgData, signatureData, GpgME::SignatureMode::Detached);

    auto signatureString = signatureData.toString();

    return GooString(std::move(signatureString));
}

std::unique_ptr<X509CertificateInfo> GpgSignatureCreation::getCertificateInfo() const
{
    if (!key) {
        return nullptr;
    }
    return getCertificateInfoFromKey(*key);
}

GpgSignatureVerification::GpgSignatureVerification(const std::vector<unsigned char> &p7data) : gpgContext { GpgME::Context::create(GpgME::CMS) }, signatureData(reinterpret_cast<const char *>(p7data.data()), p7data.size())
{
    signatureData.setEncoding(GpgME::Data::BinaryEncoding);
    // Get the first bits of data out. TODO: figure out if we can delay getting these databits until after verification.
    GpgME::Data out {};
    GpgME::Data opaqueSign { reinterpret_cast<const char *>(p7data.data()), p7data.size() };
    opaqueSign.setEncoding(GpgME::Data::BinaryEncoding);
    gpgResult = gpgContext->verifyDetachedSignature(opaqueSign, out);
}

void GpgSignatureVerification::addData(unsigned char *dataBlock, int dataLen)
{
    gpgData.write(dataBlock, dataLen);
}

std::unique_ptr<X509CertificateInfo> GpgSignatureVerification::getCertificateInfo() const
{
    if (!gpgResult || gpgResult->error()) {
        return nullptr;
    }
    auto gpgInfo = getCertificateInfoFromKey(gpgResult->signature(0).key(true, true));
    // std::cout << "GPG" << dump(*gpgInfo) << std::endl;
    return gpgInfo;
}

HashAlgorithm GpgSignatureVerification::getHashAlgorithm() const
{
    if (gpgResult) {
        switch (gpgResult->signature(0).hashAlgorithm()) {
        case GPGME_MD_MD5:
            return HashAlgorithm::Md5;
        case GPGME_MD_SHA1:
            return HashAlgorithm::Sha1;
        case GPGME_MD_MD2:
            return HashAlgorithm::Md2;
        case GPGME_MD_SHA256:
            return HashAlgorithm::Sha256;
        case GPGME_MD_SHA384:
            return HashAlgorithm::Sha384;
        case GPGME_MD_SHA512:
            return HashAlgorithm::Sha512;
        case GPGME_MD_SHA224:
            return HashAlgorithm::Sha224;
        case GPGME_MD_NONE:
        case GPGME_MD_RMD160:
        case GPGME_MD_TIGER:
        case GPGME_MD_HAVAL:
        case GPGME_MD_MD4:
        case GPGME_MD_CRC32:
        case GPGME_MD_CRC32_RFC1510:
        case GPGME_MD_CRC24_RFC2440:
        default:
            return HashAlgorithm::Unknown;
        }
    }
    return HashAlgorithm::Unknown;
}

std::string GpgSignatureVerification::getSignerName() const
{
    if (!gpgResult || gpgResult->error()) {
        return {};
    }

    auto dn = DN::parseString(gpgResult->signature(0).key(true, true).userID(0).id());
    return DN::FindFirstValue(dn, "CN").value_or("");
}

std::string GpgSignatureVerification::getSignerSubjectDN() const
{
    if (!gpgResult || gpgResult->error()) {
        return {};
    }
    return gpgResult->signature(0).key(true, true).userID(0).id();
}

std::chrono::system_clock::time_point GpgSignatureVerification::getSigningTime() const
{
    if (!gpgResult || gpgResult->error()) {
        return {};
    }
    return std::chrono::system_clock::from_time_t(gpgResult->signature(0).creationTime());
}

CertificateValidationStatus GpgSignatureVerification::validateCertificate(std::chrono::system_clock::time_point validation_time, bool ocspRevocationCheck, bool useAIACertFetch)
{
    if (!gpgResult) {
        return CERTIFICATE_NOT_VERIFIED;
    }
    if (gpgResult->error()) {
        return CERTIFICATE_GENERIC_ERROR;
    }
    gpgContext->setOffline(!ocspRevocationCheck);
    auto key = gpgResult->signature(0).key(true, true);
    if (key.isExpired()) {
        return CERTIFICATE_EXPIRED;
    }
    if (key.isRevoked()) {
        return CERTIFICATE_REVOKED;
    }
    if (key.isBad()) {
        return CERTIFICATE_NOT_VERIFIED;
    }
    return CERTIFICATE_TRUSTED;
}

SignatureValidationStatus GpgSignatureVerification::validateSignature()
{
    gpgData.rewind();
    auto result = gpgContext->verifyDetachedSignature(signatureData, gpgData);
    gpgResult = result;

    std::cout << "GpgResult" << result << std::endl;
    for (size_t i = 0; i < result.numSignatures(); i++) {
        std::cout << "GpgKey" << i << " " << result.signature(i).key(true, true) << std::endl;
    }
    std::cout << "DN: '" << result.signature(0).key(true, true).userID(0).id() << "'" << std::endl;

    if (result.error()) {
        return SIGNATURE_DECODING_ERROR;
    }
    auto summary = result.signature(0).summary();

    // std::cout << "SUMMARY" << summary << " " << static_cast<int>(summary) << " " <<  result.signature(0).validity() << std::endl;
    using Summary = GpgME::Signature::Summary;
    if (summary & Summary::Red) {
        return SIGNATURE_INVALID;
    }
    if (summary & Summary::Green || summary & Summary::Valid) {
        return SIGNATURE_VALID;
    }
    return SIGNATURE_GENERIC_ERROR;
}
