/* pggoutput.c - output object
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggdebug.h>
#include <pggoutput.h>


/*
 * Automatically type casts
 */
#define out		((PggOutputPtr)(_out))
#define old_out		((PggOutputPtr)(_old_out))


PggOutput pgg_output_new(PggErrenv errenv)
{
    PggOutputPtr	new_out;
    
    PGG_CHECK_SKIP_ARG(NULL);
    
    if (!( new_out = _malloc(PggOutput) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    memset(new_out, 0, _size(PggOutput));
    
    new_out->magic      = PggOutputMAGIC;
    new_out->refcounter = 1;
    new_out->fd         = -1;
    
    return _hd(PggOutput, new_out);
}


void pgg_output_addref(PggOutput _out, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutput, out);
    out->refcounter++;
}


PggOutput pgg_output_clone(PggOutput _old_out, PggErrenv errenv)
{
    PggOutputPtr	new_out;
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT_ARG(PggOutput, old_out, NULL);
    
    pgg_errenv_reset(local_errenv);
    
    if (!( new_out = _malloc(PggOutput) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    new_out->magic      = PggOutputMAGIC;
    new_out->refcounter = 1;
    new_out->state      = old_out->state;
    new_out->fd         = old_out->fd;
    
    switch (new_out->state) {
        case 1: /* filename */
            if (!( new_out->data.filename = strdup(old_out->data.filename) )) {
                free(new_out);
                PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
            }
            break;
            
        case 2: /* buffer */
            new_out->data.buf = pgg_buffer_clone(old_out->data.buf, local_errenv);
            
            if (pgg_errenv_is_set(local_errenv)) {
                pgg_errenv_copy(errenv, local_errenv);
                free(new_out);
                return NULL;
            }
            break;
        
        default:
            break;
   }
   
   return _hd(PggOutput, new_out);
}


static void cleanup(PggOutputPtr outptr, PggErrenv errenv)
{
    switch(outptr->state) {
        case 1: /* filename */
            if (outptr->fd != -1) {
                close(outptr->fd);
                outptr->fd = -1;
            }
            free(outptr->data.filename);
            outptr->data.filename = NULL;
            break;
        
        case 2: /* buffer */
            pgg_buffer_release(outptr->data.buf, errenv);
            break;
    }
    
    outptr->state = 0;
}


void pgg_output_release(PggOutput _out, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutput, out);
    
    if (!--out->refcounter) {
        cleanup(out, errenv);
        free(out);
    }
}


void pgg_output_set_filename(PggOutput _out, const char *filename, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    char *		tmp;
    
    PGG_STD_ASSERT(PggOutput, out);
    PGG_ASSERT(filename, ARGUMENT, NULLPTR);
    
    pgg_errenv_reset(local_errenv);
    
    if (strcmp(filename, "-")==0)		/* filename '-' not allowed */
        PGG_RETURN_ERR(ARGUMENT, VALUE);
    
    if (!( tmp = strdup(filename) ))
        PGG_RETURN_ERR(RESOURCE, MEMORY);
    
    cleanup(out, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        pgg_errenv_copy(errenv, local_errenv);
        free(tmp);
        return;
    }
    
    out->data.filename = tmp;
    out->state = 1;
}


void pgg_output_set_buffer(PggOutput _out, PggBuffer buf, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggOutput, out);
    PGG_ASSERT(buf, ARGUMENT, NULLPTR);
    
    pgg_errenv_reset(local_errenv);
    
    pgg_buffer_addref(buf, local_errenv);
    
    if (!pgg_errenv_is_set(local_errenv))
        cleanup(out, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        out->data.buf = buf;
        out->state = 2;
    }
}


void pgg_output_set_fd(PggOutput _out, int fd, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggOutput, out);
    PGG_ASSERT(fd >= 0, ARGUMENT, VALUE);
    
    pgg_errenv_reset(local_errenv);
    cleanup(out, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    
    out->state = 3;
    out->fd    = fd;
}


void pgg_output_open(PggOutput _out, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutput, out);
    
    switch (out->state) {
        case 1: /* filename */
            if ( (out->fd = creat(out->data.filename, 0666)) == -1 )
                PGG_RETURN_ERR(UNKNOWN, NONE);
            break;
        
        case 2: /* buffer */
        case 3: /* FD */
            break;
        
        default:
            PGG_RETURN_ERR(INTERNAL, NONE);
    }
}


long pgg_output_write(PggOutput _out, void *data, long amount, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    long		len;
    
    PGG_STD_ASSERT_ARG(PggOutput, out, -1);
    PGG_ASSERT_ARG(data, ARGUMENT, NULLPTR, -1);
    PGG_ASSERT_ARG(amount >= 0, ARGUMENT, VALUE, -1);
    
    pgg_errenv_reset(local_errenv);
    
    switch (out->state) {
        case 1: /* filename */
            if (out->fd == -1)
                PGG_RETURN_ERR_ARG(REQUEST, STATE, -1);
            /* fall down */
            
        case 3: /* FD */
            if ((len = write(out->fd, data, amount)) < 0)
                PGG_RETURN_ERR_ARG(UNKNOWN, NONE, -1);
            return len;
        
        case 2: /* buffer */
            pgg_buffer_append(out->data.buf, data, amount, local_errenv);
            
            if (pgg_errenv_is_set(local_errenv)) {
                pgg_errenv_copy(errenv, local_errenv);
                return -1;
            }
            
            return amount;
        
        default:
            PGG_RETURN_ERR_ARG(INTERNAL, NONE, -1);
    }
}


void pgg_output_close(PggOutput _out, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutput, out);
    
    switch (out->state) {
        case 1: /* filename */
            if (out->fd == -1)
                PGG_RETURN_ERR(REQUEST, STATE);
            close(out->fd);
            out->fd = -1;
            return;
            
        case 2: /* buffer */
        case 3: /* FD */
            return;
        
        default:
            PGG_RETURN_ERR(INTERNAL, NONE);
    }
}






