/* pggoutopts.c - options for generating output
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggdebug.h>
#include <pggoutopts.h>


/*
 * Automatically type casts
 */
#define old_oo		((PggOutoptsPtr)(_old_oo))
#define oo		((PggOutoptsPtr)(_oo))


PggOutopts pgg_outopts_new(PggErrenv errenv)
{
    PggOutoptsPtr	new_oo;
    
    PGG_CHECK_SKIP_ARG(NULL);
    
    if (!( new_oo = _malloc(PggOutopts) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    new_oo->magic      = PggOutoptsMAGIC;
    new_oo->refcounter = 1;
    new_oo->comment    = (char *)PGG_OUTOPTS_COMMENT_DEFAULT;
    new_oo->version    = 1;
    new_oo->armor      = 0;
    
    return _hd(PggOutopts, new_oo);
}


PggOutopts pgg_outopts_clone(PggOutopts _old_oo, PggErrenv errenv)
{
    PggOutoptsPtr	new_oo;
    
    PGG_STD_ASSERT_ARG(PggOutopts, old_oo, NULL);
    
    if (!( new_oo = _malloc(PggOutopts) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    if (old_oo->comment != PGG_OUTOPTS_COMMENT_NONE && old_oo->comment != PGG_OUTOPTS_COMMENT_DEFAULT) {
        if (!( new_oo->comment = strdup(old_oo->comment) )) {
            free(new_oo);
            PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
        }
    } else
        new_oo->comment = old_oo->comment;
    
    new_oo->magic      = PggOutoptsMAGIC;
    new_oo->refcounter = 1;
    new_oo->version    = old_oo->version;
    new_oo->armor      = old_oo->armor;
    
    return _hd(PggOutopts, new_oo);
}


void pgg_outopts_addref(PggOutopts _oo, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutopts, oo);
    oo->refcounter++;
}


void pgg_outopts_release(PggOutopts _oo, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutopts, oo);
    
    if (!(--oo->refcounter)) {
        if (oo->comment != PGG_OUTOPTS_COMMENT_NONE && oo->comment != PGG_OUTOPTS_COMMENT_DEFAULT) {
            free(oo->comment);
        }
        free(oo);
    }
}


void pgg_outopts_set_comment(PggOutopts _oo, const char *text, PggErrenv errenv)
{
    char *		tmp;
    
    PGG_STD_ASSERT(PggOutopts, oo);
    
    if (text != PGG_OUTOPTS_COMMENT_NONE && text != PGG_OUTOPTS_COMMENT_DEFAULT) {
        if (!( tmp = strdup(text) ))
            PGG_RETURN_ERR(RESOURCE, MEMORY);
    }
    else
        tmp = (char *)text;	/* Discard const because this is either _COMMENT_NONE or _COMMENT_DEFAULT */
    
    if (oo->comment != PGG_OUTOPTS_COMMENT_NONE && oo->comment != PGG_OUTOPTS_COMMENT_DEFAULT)
        free(oo->comment);
    
    oo->comment = tmp;
}


void pgg_outopts_set_armor(PggOutopts _oo, int yesno, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutopts, oo);
    oo->armor = yesno ? 1 : 0;
}


void pgg_outopts_set_version(PggOutopts _oo, int yesno, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggOutopts, oo);
    oo->version = yesno ? 1 : 0;
}


const char * pgg_outopts_get_comment(PggOutopts _oo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggOutopts, oo, NULL);
    return oo->comment;
}


int pgg_outopts_get_version(PggOutopts _oo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggOutopts, oo, -1);
    return oo->version;
}


int pgg_outopts_get_armor(PggOutopts _oo, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggOutopts, oo, -1);
    return oo->armor;
}


void pgg_outopts_setup_exe(PggOutopts _oo, PggExe exe, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggOutopts, oo);
    PGG_ASSERT(exe, ARGUMENT, NULLPTR);
    
    pgg_errenv_reset(local_errenv);
    
    if (oo->comment == PGG_OUTOPTS_COMMENT_NONE) {
        pgg_exe_add_arg(exe, "--comment", local_errenv);
        pgg_exe_add_arg(exe, "", local_errenv);
    }
    else if (oo->comment == PGG_OUTOPTS_COMMENT_DEFAULT) {
        pgg_exe_add_arg(exe, "--default-comment", local_errenv);
    } 
    else {
        pgg_exe_add_arg(exe, "--comment", local_errenv);
        pgg_exe_add_arg(exe, oo->comment, local_errenv);
    }
    
    if (!oo->version)
        pgg_exe_add_arg(exe, "--no-version", local_errenv);
    else
        pgg_exe_add_arg(exe, "--emit-version", local_errenv);
    
    if (oo->armor)
        pgg_exe_add_arg(exe, "--armor", local_errenv);
    else
        pgg_exe_add_arg(exe, "--no-armor", local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
}


