/* pggencrypt.c - Public encryption functions
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggdebug.h>
#include <pggencrypt.h>
#include <pggstate.h>


#define encrypt		((PggEncryptPtr)(_encrypt))
#define old_encrypt	((PggEncryptPtr)(_old_encrypt))


PggEncrypt pgg_encrypt_new(PggErrenv errenv)
{
    PggEncryptPtr	new_encrypt;
    
    if (!( new_encrypt = _malloc(PggEncrypt) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    memset(new_encrypt, 0, _size(PggEncrypt));
    
    new_encrypt->magic      = PggEncryptMAGIC;
    new_encrypt->refcounter = 1;
    
    if (pgg_strvector_init(&(new_encrypt->recipients))) {
        free(new_encrypt);
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    }
    
    return _hd(PggEncrypt, new_encrypt);
}


PggEncrypt pgg_encrypt_clone(PggEncrypt _encrypt, PggErrenv errenv)
{
    PggEncryptPtr	new_encrypt;
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    
    pgg_errenv_reset(local_errenv);
    
    if (!( new_encrypt = _malloc(PggEncrypt) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    memset(new_encrypt, 0, _size(PggEncrypt));
    
    new_encrypt->magic      = PggEncryptMAGIC;
    new_encrypt->refcounter = 1;
    new_encrypt->sign       = encrypt->sign;
    
    if (encrypt->signer && !(new_encrypt->signer = strdup(encrypt->signer))) {
        free(new_encrypt);
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    }
    
    if (pgg_strvector_cloneto(&(encrypt->recipients), &(new_encrypt->recipients))) {
        if (new_encrypt->signer)
            free(new_encrypt->signer);
        free(new_encrypt);
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    }
    
    if (encrypt->algo)
        new_encrypt->algo = pgg_algo_clone(encrypt->algo, local_errenv);
    
    if (encrypt->stdio)
        new_encrypt->stdio = pgg_stdio_clone(encrypt->stdio, local_errenv);
    
    if (encrypt->passcache)
        new_encrypt->passcache = pgg_passcache_clone(encrypt->passcache, local_errenv);
    
    if (encrypt->config)
        new_encrypt->config = pgg_config_clone(encrypt->config, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        if (new_encrypt->config)
            pgg_config_release(new_encrypt->config, NULL);
        
        if (new_encrypt->passcache)
            pgg_passcache_release(new_encrypt->passcache, NULL);
        
        if (new_encrypt->stdio)
            pgg_stdio_release(new_encrypt->stdio, NULL);
        
        if (new_encrypt->algo)
            pgg_algo_release(new_encrypt->algo, NULL);
        
        pgg_strvector_release(&(new_encrypt->recipients));
        
        if (new_encrypt->signer);
            free(new_encrypt->signer);
        
        free(new_encrypt);
        
        pgg_errenv_copy(errenv, local_errenv);
        
        return NULL;
    }
    
    return _hd(PggEncrypt, new_encrypt);
}


void pgg_encrypt_addref(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    encrypt->refcounter++;
}


void pgg_encrypt_release(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    
    if (!--encrypt->refcounter) {
        if (encrypt->config)
            pgg_config_release(encrypt->config, NULL);
        
        if (encrypt->passcache)
            pgg_passcache_release(encrypt->passcache, NULL);
        
        if (encrypt->stdio);
            pgg_stdio_release(encrypt->stdio, NULL);
        
        if (encrypt->algo)
            pgg_algo_release(encrypt->algo, NULL);
        
        pgg_strvector_release(&(encrypt->recipients));
        
        if (encrypt->signer)
            free(encrypt->signer);
        
        free(encrypt);
    }
}


void pgg_encrypt_set_sign(PggEncrypt _encrypt, int yesno, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    encrypt->sign = yesno ? 1 : 0;
}


int pgg_encrypt_get_sign(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, -1);
    return encrypt->sign;
}


void pgg_encrypt_set_signer(PggEncrypt _encrypt, const char *signer, PggErrenv errenv)
{
    char *		tmp;
    
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    
    if (signer && !(tmp = strdup(signer)))
        PGG_RETURN_ERR(RESOURCE, MEMORY);
    
    if (encrypt->signer)
        free(encrypt->signer);
    
    encrypt->signer = tmp;
}


const char * pgg_encrypt_get_signer(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    PGG_ASSERT_ARG(encrypt->signer, REQUEST, NOTSET, NULL);
    return encrypt->signer;
}


void pgg_encrypt_set_noencryptto(PggEncrypt _encrypt, int yesno, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    encrypt->noencryptto = yesno ? 1 : 0;
}


int pgg_encrypt_get_noencryptto(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, -1);
    return encrypt->noencryptto;
}


void pgg_encrypt_add_recipient(PggEncrypt _encrypt, const char *recipient, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    PGG_ASSERT(recipient, ARGUMENT, NULLPTR);
    
    if (pgg_strvector_append(&(encrypt->recipients), recipient))
        PGG_RETURN_ERR(RESOURCE, MEMORY);
}


int pgg_encrypt_get_recipient_count(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, -1);
    return encrypt->recipients.used;
}


const char * pgg_encrypt_get_recipient(PggEncrypt _encrypt, int index, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    PGG_ASSERT_ARG(index >= 0, ARGUMENT, TOSMALL, NULL);
    PGG_ASSERT_ARG(index < encrypt->recipients.used, ARGUMENT, TOLARGE, NULL);
    return encrypt->recipients.vector[index];
}


void pgg_encrypt_rem_recipient(PggEncrypt _encrypt, int index, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    PGG_ASSERT(index >= 0, ARGUMENT, TOSMALL);
    PGG_ASSERT(index < encrypt->recipients.used, ARGUMENT, TOLARGE);
    pgg_strvector_remove(&(encrypt->recipients), index);
}


void pgg_encrypt_set_algo(PggEncrypt _encrypt, PggAlgo algo, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    
    pgg_errenv_reset(local_errenv);
    
    if (algo)
        pgg_algo_addref(algo, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (encrypt->algo)
            pgg_algo_release(encrypt->algo, NULL);
        encrypt->algo = algo;
    }
}


PggAlgo pgg_encrypt_get_algo(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    PGG_ASSERT_ARG(encrypt->algo, REQUEST, NOTSET, NULL);
    pgg_algo_addref(encrypt->algo, NULL);
    return encrypt->algo;
}


void pgg_encrypt_set_stdio(PggEncrypt _encrypt, PggStdio stdio, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    
    pgg_errenv_reset(local_errenv);
    
    if (stdio)
        pgg_stdio_addref(stdio, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (encrypt->stdio)
            pgg_stdio_release(encrypt->stdio, NULL);
        encrypt->stdio = stdio;
    }
}


PggStdio pgg_encrypt_get_stdio(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    PGG_ASSERT_ARG(encrypt->stdio, REQUEST, NOTSET, NULL);
    pgg_stdio_addref(encrypt->stdio, NULL);
    return encrypt->stdio;
}


void pgg_encrypt_set_passcache(PggEncrypt _encrypt, PggPasscache passcache, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    
    pgg_errenv_reset(local_errenv);
    
    if (passcache)
        pgg_passcache_addref(passcache, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (encrypt->passcache)
            pgg_passcache_release(encrypt->passcache, NULL);
        encrypt->passcache = passcache;
    }
}


PggPasscache pgg_encrypt_get_passcache(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    PGG_ASSERT_ARG(encrypt->passcache, REQUEST, NOTSET, NULL);
    pgg_passcache_addref(encrypt->passcache, NULL);
    return encrypt->passcache;
}


void pgg_encrypt_set_config(PggEncrypt _encrypt, PggConfig config, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    
    pgg_errenv_reset(local_errenv);
    
    if (config)
        pgg_config_addref(config, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (encrypt->config)
            pgg_config_release(encrypt->config, NULL);
        encrypt->config = config;
    }
}


PggConfig pgg_encrypt_get_config(PggEncrypt _encrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggEncrypt, encrypt, NULL);
    PGG_ASSERT_ARG(encrypt->config, REQUEST, NOTSET, NULL);
    pgg_config_addref(encrypt->config, NULL);
    return encrypt->config;
}


void pgg_encrypt_execute(PggEncrypt _encrypt, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    PggExe		exe;
    int			i;
    int			return_code;
    char		keyid_buffer[32];
    const char *	passphrase;
    STM_DECLS;
    
    PGG_STD_ASSERT(PggEncrypt, encrypt);
    PGG_ASSERT(encrypt->stdio, REQUEST, STATE);			/* No input/output */
    PGG_ASSERT(encrypt->recipients.used, REQUEST, STATE);	/* No recipients */
    
    pgg_errenv_reset(local_errenv);
    
    exe = pgg_exe_new(local_errenv);
    
    if (encrypt->config)
        pgg_config_setup_exe(encrypt->config, exe, local_errenv);
    else {
        PggConfig config = pgg_config_new(local_errenv);
        pgg_config_setup_exe(config, exe, local_errenv);
        pgg_config_release(config, NULL);
    }
    
    if (encrypt->algo)
        pgg_algo_setup_exe(encrypt->algo, exe, local_errenv);
    
    if (encrypt->noencryptto)
        pgg_exe_add_arg(exe, "--no-encrypt-to", local_errenv);
    
    if (encrypt->sign && encrypt->signer) {
        pgg_exe_add_arg(exe, "--local-user", local_errenv);
        pgg_exe_add_arg(exe, encrypt->signer, local_errenv);
    }
    
    for (i=0; i<encrypt->recipients.used; ++i) {
        pgg_exe_add_arg(exe, "--recipient", local_errenv);
        pgg_exe_add_arg(exe, encrypt->recipients.vector[i], local_errenv);
    }
    
    if (encrypt->sign)
        pgg_exe_add_arg(exe, "--sign", local_errenv);
    
    pgg_exe_add_arg(exe, "--encrypt", local_errenv);
    
    pgg_stdio_setup_exe(encrypt->stdio, exe, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("initializing PggExe failed"));
        pgg_exe_release(exe, NULL);
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    else
        PGG_DEBUG(("setup PggExe ok"));
    
    pgg_exe_request_shm(exe, local_errenv);
    pgg_exe_request_status(exe, local_errenv);
    pgg_exe_start(exe, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("starting GnuPG failed"));
        pgg_exe_release(exe, NULL);
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    else
        PGG_DEBUG(("GnuPG started"));
    
    STM_BEGIN
    
        IF_EVENT_FINISHED(0) {
            PGG_DEBUG(("GnuPG terminated. Hopefully all gone right"));
            STM_LEAVE();
        }
        IF_STATUS(0, GOOD_PASSPHRASE) {
            PGG_DEBUG(("used key for signing has no passphrase"));
            SWITCH_TO_STATE(3);
        }
        IF_STATUS(0, NEED_PASSPHRASE) {		/* Sign with a key which has a passphrase */
            PGG_DEBUG(("passphrase for keyid %s requested", status_arg));
            if (strlen(status_arg) != 16) {
                PGG_DEBUG(("unexpected keyid length!"));
                pgg_exe_release(exe, NULL);
                PGG_RETURN_ERR(GNUPG, UNEXPECTED);
            }
            if (!encrypt->passcache) {
                PGG_DEBUG(("no passcache set"));
                pgg_exe_release(exe, NULL);
                PGG_RETURN_ERR(CRYPT, PASSPHRASE);	/* FIXME: what about a new suberr code ? */
            }
            strcpy(keyid_buffer, status_arg);
            SWITCH_TO_STATE(1);
        }
        IF_STATUS_ARG(0, SHM_GET_BOOL, "openfile.overwrite.okay") {
            PGG_DEBUG(("GnuPG asks to overwrite a file"));
            pgg_exe_reply_bool(exe, 1, local_errenv);
            if (pgg_errenv_is_set(local_errenv)) {
                PGG_DEBUG(("reply 'yes' failed"));
                pgg_exe_release(exe, NULL);
                pgg_errenv_copy(errenv, local_errenv);
                return;
            }
        }
        
        
        IF_STATUS_ARG(1, SHM_GET_HIDDEN, "passphrase.enter") {
            if (!( passphrase = pgg_passcache_get_passphrase(encrypt->passcache, keyid_buffer, local_errenv) )) {
                PGG_DEBUG(("no passphrase available for keyid %s", keyid_buffer));
                pgg_exe_release(exe, NULL);
                PGG_RETURN_ERR(CRYPT, PASSPHRASE);	/* FIXME: what about a new suberr code ? */
            }
            
            PGG_DEBUG(("reply passphrase for keyid %s", keyid_buffer));
            pgg_exe_reply_str(exe, passphrase, local_errenv);
            
            if (pgg_errenv_is_set(local_errenv)) {
                PGG_DEBUG(("error during passphrase reply"));
                pgg_exe_release(exe, NULL);
                pgg_errenv_copy(errenv, local_errenv);
                return;
            }
            
            SWITCH_TO_STATE(2);
        }
        
        
        IF_STATUS(2, GOOD_PASSPHRASE) {
            PGG_DEBUG(("passphrase was ok"));
            SWITCH_TO_STATE(3);
        }
        IF_STATUS(2, BAD_PASSPHRASE) {
            PGG_DEBUG(("wrong passphrase"));
            pgg_passcache_rem_passphrase(encrypt->passcache, keyid_buffer, NULL);
            pgg_exe_release(exe, NULL);
            PGG_RETURN_ERR(CRYPT, PASSPHRASE);
        }
        
        
        IF_STATUS_ARG(3, SHM_GET_BOOL, "openfile.overwrite.okay") {
            PGG_DEBUG(("GnuPG asks to overwrite a file"));
            pgg_exe_reply_bool(exe, 1, local_errenv);
            if (pgg_errenv_is_set(local_errenv)) {
                PGG_DEBUG(("reply 'yes' failed"));
                pgg_exe_release(exe, NULL);
                pgg_errenv_copy(errenv, local_errenv);
                return;
            }
        }
        IF_EVENT_FINISHED(3) {
            STM_LEAVE();
        }
        
        
        if (pgg_errenv_is_set(local_errenv)) {
            PGG_DEBUG(("unhandled error in state: %d", state));
            pgg_exe_release(exe, NULL);
            PGG_RETURN_ERR(INTERNAL, NONE);
        }
    
    STM_END
    
    return_code = pgg_exe_get_return_code(exe, local_errenv);
    
    pgg_exe_release(exe, NULL);
    
    if (return_code != 0) {
        PGG_DEBUG(("unexpected return code: %d", return_code));
        PGG_RETURN_ERR(GNUPG, RETURNCODE);
    }
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("an unhandled error occured"));
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    
    PGG_DEBUG(("yep, all gone well"));
}


