/* pggdecrypt.c - Public decryption functions
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggdebug.h>
#include <pggdecrypt.h>
#include <pggstate.h>


#define decrypt		((PggDecryptPtr)(_decrypt))


PggDecrypt pgg_decrypt_new(PggErrenv errenv)
{
    PggDecryptPtr	new_decrypt;
    
    if (!( new_decrypt = _malloc(PggDecrypt) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    memset(new_decrypt, 0, _size(PggDecrypt));
    
    new_decrypt->magic      = PggDecryptMAGIC;
    new_decrypt->refcounter = 1;
    
    return _hd(PggDecrypt, new_decrypt);
}


PggDecrypt pgg_decrypt_clone(PggDecrypt _decrypt, PggErrenv errenv)
{
    PggDecryptPtr	new_decrypt;
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT_ARG(PggDecrypt, decrypt, NULL);
    
    pgg_errenv_reset(local_errenv);
    
    if (!( new_decrypt = _malloc(PggDecrypt) ))
        PGG_RETURN_ERR_ARG(RESOURCE, MEMORY, NULL);
    
    memset(new_decrypt, 0, _size(PggDecrypt));
    
    new_decrypt->magic      = PggDecryptMAGIC;
    new_decrypt->refcounter = 1;
    
    if (decrypt->siginfo)
        new_decrypt->siginfo = pgg_siginfo_clone(decrypt->siginfo, local_errenv);
    
    if (decrypt->stdio)
        new_decrypt->stdio = pgg_stdio_clone(decrypt->stdio, local_errenv);
    
    if (decrypt->passcache)
        new_decrypt->passcache = pgg_passcache_clone(decrypt->passcache, local_errenv);
    
    if (decrypt->config)
        new_decrypt->config = pgg_config_clone(decrypt->config, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        if (new_decrypt->config)
            pgg_config_release(new_decrypt->config, NULL);
        
        if (new_decrypt->passcache)
            pgg_passcache_release(new_decrypt->passcache, NULL);
        
        if (new_decrypt->stdio)
            pgg_stdio_release(new_decrypt->stdio, NULL);
        
        if (new_decrypt->siginfo)
            pgg_siginfo_release(new_decrypt->siginfo, NULL);
        
        free(new_decrypt);
        
        pgg_errenv_copy(errenv, local_errenv);
        return NULL;
    }
    
    return _hd(PggDecrypt, new_decrypt);
}


void pgg_decrypt_addref(PggDecrypt _decrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggDecrypt, decrypt);
    decrypt->refcounter++;
}


void pgg_decrypt_release(PggDecrypt _decrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT(PggDecrypt, decrypt);
    
    if (!--decrypt->refcounter) {
        if (decrypt->config)
            pgg_config_release(decrypt->config, NULL);
        
        if (decrypt->passcache)
            pgg_passcache_release(decrypt->passcache, NULL);
        
        if (decrypt->stdio);
            pgg_stdio_release(decrypt->stdio, NULL);
        
        if (decrypt->siginfo)
            pgg_siginfo_release(decrypt->siginfo, NULL);
        
        free(decrypt);
    }
}


void pgg_decrypt_set_stdio(PggDecrypt _decrypt, PggStdio stdio, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggDecrypt, decrypt);
    
    pgg_errenv_reset(local_errenv);
    
    if (stdio)
        pgg_stdio_addref(stdio, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (decrypt->stdio)
            pgg_stdio_release(decrypt->stdio, NULL);
        
        decrypt->stdio = stdio;
    }
}


PggStdio pgg_decrypt_get_stdio(PggDecrypt _decrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggDecrypt, decrypt, NULL);
    PGG_ASSERT_ARG(decrypt->stdio, REQUEST, NOTSET, NULL);
    return decrypt->stdio;
}


void pgg_decrypt_set_siginfo(PggDecrypt _decrypt, PggSiginfo siginfo, PggErrenv errenv)
{
    PggErrenv           local_errenv;

    PGG_STD_ASSERT(PggDecrypt, decrypt);

    pgg_errenv_reset(local_errenv);

    if (siginfo)
        pgg_siginfo_addref(siginfo, local_errenv);

    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (decrypt->siginfo)
            pgg_siginfo_release(decrypt->siginfo, NULL);

        decrypt->siginfo = siginfo;
    }
}


PggSiginfo pgg_decrypt_get_siginfo(PggDecrypt _decrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggDecrypt, decrypt, NULL);
    PGG_ASSERT_ARG(decrypt->siginfo, REQUEST, NOTSET, NULL);
    return decrypt->siginfo;
}


void pgg_decrypt_set_passcache(PggDecrypt _decrypt, PggPasscache passcache, PggErrenv errenv)
{
    PggErrenv           local_errenv;

    PGG_STD_ASSERT(PggDecrypt, decrypt);

    pgg_errenv_reset(local_errenv);

    if (passcache)
        pgg_passcache_addref(passcache, local_errenv);

    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (decrypt->passcache)
            pgg_passcache_release(decrypt->passcache, NULL);

        decrypt->passcache = passcache;
    }
}


PggPasscache pgg_decrypt_get_passcache(PggDecrypt _decrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggDecrypt, decrypt, NULL);
    PGG_ASSERT_ARG(decrypt->passcache, REQUEST, NOTSET, NULL);
    return decrypt->passcache;
}


void pgg_decrypt_set_config(PggDecrypt _decrypt, PggConfig config, PggErrenv errenv)
{
    PggErrenv           local_errenv;

    PGG_STD_ASSERT(PggDecrypt, decrypt);

    pgg_errenv_reset(local_errenv);

    if (config)
        pgg_config_addref(config, local_errenv);

    if (pgg_errenv_is_set(local_errenv))
        pgg_errenv_copy(errenv, local_errenv);
    else {
        if (decrypt->config)
            pgg_config_release(decrypt->config, NULL);

        decrypt->config = config;
    }
}


PggConfig pgg_decrypt_get_config(PggDecrypt _decrypt, PggErrenv errenv)
{
    PGG_STD_ASSERT_ARG(PggDecrypt, decrypt, NULL);
    PGG_ASSERT_ARG(decrypt->config, REQUEST, NOTSET, NULL);
    return decrypt->config;
}


void pgg_decrypt_execute(PggDecrypt _decrypt, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    PggExe		exe;
    const char *	passphrase;
    char		keyid_buffer[64];
    int			return_code;
    STM_DECLS;
    
    PGG_STD_ASSERT(PggDecrypt, decrypt);
    PGG_ASSERT(decrypt->stdio, REQUEST, STATE);			/* No input/output */
    
    pgg_errenv_reset(local_errenv);
    
    exe = pgg_exe_new(local_errenv);
    
    if (decrypt->config)
        pgg_config_setup_exe(decrypt->config, exe, local_errenv);
    else {
        PggConfig config = pgg_config_new(local_errenv);
        pgg_config_setup_exe(config, exe, local_errenv);
        pgg_config_release(config, NULL);
    }
    
    pgg_stdio_setup_exe(decrypt->stdio, exe, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("PggExe init failed"));
        pgg_exe_release(exe, NULL);
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    
    pgg_exe_request_shm(exe, local_errenv);
    pgg_exe_request_status(exe, local_errenv);
    pgg_exe_start(exe, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("starting GnuPG failed"));
        pgg_exe_release(exe, NULL);
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    else
        PGG_DEBUG(("GnuPG started..."));
    
    STM_BEGIN
        
        /*
         * State 0 is the main state. Other states are called like subroutines
         * from here.
         */
        IF_STATUS(0, ENC_TO) {
            PGG_DEBUG(("public encrypted packet for keyid 0x%s found.", status_arg));
        }
        IF_STATUS(0, SIG_ID) {
            PGG_DEBUG(("A SIG_ID event found. Switched to signature checking state"));
            SWITCH_TO_STATE(10);
        }
        IF_STATUS(0, GOOD_PASSPHRASE) {
            PGG_DEBUG(("a passphrase was ok or a key has no passphrase"));
            SWITCH_TO_STATE(5);
        }
        IF_STATUS(0, BAD_PASSPHRASE) {
            PGG_DEBUG(("wrong passphrase"));
            pgg_passcache_rem_passphrase(decrypt->passcache, keyid_buffer, NULL);
            pgg_exe_release(exe, NULL);
            PGG_RETURN_ERR(CRYPT, PASSPHRASE);
        }
        IF_STATUS(0, DECRYPTION_FAILED) {
            PGG_DEBUG(("decryption failed. looks like no secret key available"));
            pgg_exe_release(exe, NULL);
            PGG_RETURN_ERR(CRYPT, NOSECKEY);
        }
        IF_STATUS(0, NEED_PASSPHRASE) {			/* encrypted to a passphrase protected key */
            PGG_DEBUG(("passphrase for keyid %s requested", status_arg));
            if (strlen(status_arg) < 33 || status_arg[16] != ' ') {
                PGG_DEBUG(("unexpected keyid format"));
                pgg_exe_release(exe, NULL);
                PGG_RETURN_ERR(GNUPG, UNEXPECTED);
            }
            if (!decrypt->passcache) {
                PGG_DEBUG(("no passcache set"));
                pgg_exe_release(exe, NULL);
                PGG_RETURN_ERR(CRYPT, PASSPHRASE);
            }
            strncpy(keyid_buffer, status_arg + 17, 16);
            keyid_buffer[16] = 0;
            SWITCH_TO_STATE(1);
        }
        IF_STATUS(0, NEED_PASSPHRASE_SYM) {		/* A conventionally encrypted message */
            PGG_DEBUG(("conventionally encrypted message found (currently not supported)"));
            PGG_RETURN_ERR(GNUPG, UNEXPECTED);
        }
        IF_EVENT_FINISHED(0) {
            PGG_DEBUG(("GnuPG terminated in state 0"));
            STM_LEAVE();
        }
        
        
        /*
         * State 1 is a temporary reminder for submitting a passphrase for
         * a keyid. After the passphrase is replied, switch back to state 0.
         */
        IF_STATUS_ARG(1, SHM_GET_HIDDEN, "passphrase.enter") {
            if (!( passphrase = pgg_passcache_get_passphrase(decrypt->passcache, keyid_buffer, local_errenv) )) {
                PGG_DEBUG(("no passphrase available for keyid %s", keyid_buffer));
                pgg_exe_release(exe, NULL);
                PGG_RETURN_ERR(CRYPT, PASSPHRASE);
            }
            
            PGG_DEBUG(("reply passphrase for keyid %s", keyid_buffer));
            pgg_exe_reply_str(exe, passphrase, local_errenv);
            
            if (pgg_errenv_is_set(local_errenv)) {
                PGG_DEBUG(("passphrase reply failed"));
                pgg_exe_release(exe, NULL);
                pgg_errenv_copy(errenv, local_errenv);
                return;
            }
            
            SWITCH_TO_STATE(0);
        }
        
        
        /*
         * In state 5 their is at least one secret key available to decrypt the message.
         * It is possible that a valid passphrase was submitted or the used key has no
         * passphrase.
         * We only wait for signatures events or for finishing GnuPG.
         */
        IF_STATUS(5, ENC_TO) {
            PGG_DEBUG(("another public encrypted packet for keyid 0x%s found.", status_arg));
        }
        IF_STATUS(5, SIG_ID) {
            PGG_DEBUG(("message was encrypted and a SIG_ID event found."));
            SWITCH_TO_STATE(10);
        }
        IF_STATUS(5, DECRYPTION_OKAY) {
            PGG_DEBUG(("GnuPG says decryption is ok"));
        }
        IF_STATUS(5, DECRYPTION_FAILED) {
            PGG_DEBUG(("GnuPG say decryption failed. But a valid secret key was used!?!?"));
        }
        IF_STATUS_ARG(5, SHM_GET_BOOL, "openfile.overwrite.okay") {
            PGG_DEBUG(("GnuPG asks to overwrite a file"));
            pgg_exe_reply_bool(exe, 1, local_errenv);
            if (pgg_errenv_is_set(local_errenv)) {
                PGG_DEBUG(("reply 'yes' failed"));
                pgg_exe_release(exe, NULL);
                pgg_errenv_copy(errenv, local_errenv);
                return;
            }
        }
        IF_EVENT_FINISHED(5) {
            PGG_DEBUG(("GnuPG finished after decrypting a public encrypted message"));
            STM_LEAVE();
        }
        
        
        /*
         * State 10 is the signature checking state.
         */
        IF_STATUS(10, SIG_ID) {
            PGG_DEBUG(("ignored SIG_ID"));
        }
        IF_STATUS(10, GOODSIG) {
            PGG_DEBUG(("ignored GOODSIG"));
        }
        IF_STATUS(10, ERRSIG) {
            PGG_DEBUG(("ignored ERRSIG"));
        }
        IF_STATUS(10, BADSIG) {
            PGG_DEBUG(("ignored BADSIG"));
        }
        IF_STATUS(10, VALIDSIG) {
            PGG_DEBUG(("ignored VALIDSIG"));
        }
        IF_STATUS(10, TRUST_UNDEFINED) {
            PGG_DEBUG(("ignored TRUST_UNDEFINED"));
        }
        IF_STATUS(10, TRUST_NEVER) {
            PGG_DEBUG(("ignored TRUST_NEVER"));
        }
        IF_STATUS(10, TRUST_MARGINAL) {
            PGG_DEBUG(("ignored TRUST_MARGINAL"));
        }
        IF_STATUS(10, TRUST_FULLY) {
            PGG_DEBUG(("ignored TRUST_FULLY"));
        }
        IF_STATUS(10, TRUST_ULTIMATE) {
            PGG_DEBUG(("ignored TRUST_ULTIMATE"));
        }
        IF_STATUS(10, SIGEXPIRED) {
            PGG_DEBUG(("ignored SIGEXPIRED"));
        }
        IF_STATUS(10, DECRYPTION_OKAY) {
            PGG_DEBUG(("GnuPG says, decryption is ok"));
        }
        IF_STATUS(10, DECRYPTION_FAILED) {
            PGG_DEBUG(("GnuPG says, decryption failed. But their were at least one signature!?!?"));
        }
        IF_EVENT_FINISHED(10) {
            PGG_DEBUG(("GnuPG finished during signature verification"));
            STM_LEAVE();
        }
        
        
        /*
         * Error handler for uncatched errors
         */
        if (pgg_errenv_is_set(local_errenv)) {
            PGG_DEBUG(("unhandled error in state: %d", state));
            pgg_exe_release(exe, NULL);
            PGG_RETURN_ERR(INTERNAL, NONE);
        }
        
    STM_END
    
    return_code = pgg_exe_get_return_code(exe, local_errenv);
    pgg_exe_release(exe, NULL);
    
    if (return_code != 0) {
        PGG_DEBUG(("unexpected return code: %d", return_code));
        PGG_RETURN_ERR(GNUPG, RETURNCODE);
    }
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("an unhandled error occured"));
        pgg_errenv_copy(errenv, local_errenv);
        return;
    }
    
    PGG_DEBUG(("It looks like that decryption was ok"));
}


