/* pgg_pubkey_process.c - Process PggColons event and fillup a PggPubkey object
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggdebug.h>
#include <pggpubkey.h>


#define pubkey			((PggPubkeyPtr)(_pubkey))


static void add_uid(PggPubkeyPtr pkey, const char * name, PggErrenv errenv)
{
    PggPubkeyUID *	uid_node;
    
    if (!( uid_node = (PggPubkeyUID *)malloc(sizeof(PggPubkeyUID) + strlen(name))))
        PGG_RETURN_ERR(RESOURCE, MEMORY);
    
    pgg_list_init(&(uid_node->sigs));
    
    strcpy(uid_node->name, name);
    
    pgg_list_append(&(pkey->uids), (PggListNode *)uid_node);
}


#define EXPECT_COLON(c)		if (colon != (c)) PGG_RETURN_ERR(ARGUMENT, VALUE)
#define EXPECT_TEXT(t)		if (strcmp(data, (t))) PGG_RETURN_ERR(ARGUMENT, VALUE)


void pgg_pubkey_process(PggPubkey _pubkey, int colon, const char *data, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggPubkey, pubkey);
    PGG_ASSERT(data, ARGUMENT, NULLPTR);
    PGG_ASSERT(pubkey->process_state >= 10 && pubkey->process_state < 999, REQUEST, STATE);
    
    pgg_errenv_reset(local_errenv);
    
    switch (pubkey->process_state) {
        
        /*
         * State 10 to 18 reads the first line of a GnuPG --with-colons listing.
         * After the user id is read, the state maschine switch to state 20 to
         * skip optional remaining columns and then jump to the appropriate handler.
         */
        case 0x10:				/* In State 10, the first "pub" colon is expected */
            EXPECT_COLON(0);
            EXPECT_TEXT("pub");
            pubkey->process_state = 0x11;
            break;
        
        case 0x11: 				/* Validity of primary key */
            EXPECT_COLON(1);
            pubkey->process_state = 0x12;
            break;
        
        case 0x12:				/* Size of primary key */
            EXPECT_COLON(2);
            pubkey->process_state = 0x13;
            break;
        
        case 0x13:				/* Key algorithm identifier */
            EXPECT_COLON(3);
            pubkey->process_state = 0x14;
            break;
        
        case 0x14:				/* Long key ID of primary key */
            EXPECT_COLON(4);
            pubkey->process_state = 0x15;
            break;
        
        case 0x15:				/* Key creation date-time of primary key */
            EXPECT_COLON(5);
            pubkey->process_state = 0x16;
            break;
        
        case 0x16:				/* Expiration date-time of primary key */
            EXPECT_COLON(6);
            pubkey->process_state = 0x17;
            break;
        
        case 0x17:				/* LID (not used by PGG) */
            EXPECT_COLON(7);
            pubkey->process_state = 0x18;
            break;
        
        case 0x18:				/* Ownertrust of primary key */
            EXPECT_COLON(8);
            pubkey->process_state = 0x19;
            break;
        
        case 0x19:				/* Main uid of primary key */
            EXPECT_COLON(9);
            add_uid(pubkey, data, local_errenv);
            if (pgg_errenv_is_set(local_errenv)) {
                pgg_errenv_copy(errenv, local_errenv);
                return;
            }
            pubkey->process_state = 0x20;
            break;
        
        
        /*
         * 
         */
        case 0x20:				/* State 20: Wait for colum 0 and then jump to the different handler */
            if (colon == 0) {
                if (!strcmp(data, "sig"))
                    pubkey->process_state = 0x31;
                else if (!strcmp(data, "uid"))
                    pubkey->process_state = 0x41;
                else {
                    PGG_DEBUG(("Unknown key listing line identifier: %s", data));
                }
            }
            break;
        
        
        /*
         * States 30 to 3A parses all types of signatures
         */
        case 0x31:				/* Empty fields */
        case 0x32:
        case 0x33:
        case 0x36:
        case 0x37:
        case 0x38:
            EXPECT_COLON(pubkey->process_state & 0x0f);
            EXPECT_TEXT("");
            pubkey->process_state++;
            break;
        
        case 0x34:
            EXPECT_COLON(4);			/* Key ID of used key to make the signature */
            pubkey->process_state = 0x35;
            break;
        
        case 0x35:				/* Creation date-time of signature */
            EXPECT_COLON(5);
            pubkey->process_state = 0x36;
            break;
        
        case 0x39:				/* UID of signer (not used by PGG) */
            EXPECT_COLON(9);
            pubkey->process_state = 0x3A;
            break;
        
        case 0x3A:				/* Class of signature */
            EXPECT_COLON(10);
            pubkey->process_state = 0x20;
            break;
        
        
        /*
         * 0x40 to 0x49: Additional UIDs
         */
        case 0x41:				/* Validity of this UID */
            EXPECT_COLON(1);
            pubkey->process_state = 0x42;
            break;
        
        case 0x42:				/* Empty Fields */
        case 0x43:
        case 0x44:
        case 0x45:
        case 0x46:
        case 0x47:
        case 0x48:
            EXPECT_COLON(pubkey->process_state & 0xf);
            EXPECT_TEXT("");
            pubkey->process_state++;
            break;
        
        case 0x49:				/* UID String */
           EXPECT_COLON(9);
           add_uid(pubkey, data, local_errenv);
           if (pgg_errenv_is_set(local_errenv)) {
               pgg_errenv_copy(errenv, local_errenv);
               return;
           }
           pubkey->process_state = 0x20;
           break;
       
    }
}












