/* pgg_algo_setup_exe.c - Sets up a PggExe object with the values of a PggAlgo object
 *      Copyright (C) 1999 Michael Roth <mroth@gnupg.org>
 *
 * This file is part of PGG (Privacy Guard Glue).
 *
 * PGG is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * PGG is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <includes.h>
#include <pgg.h>
#include <pggalgo.h>
#include <pggdebug.h>


#define algo		((PggAlgoPtr)(_algo))


void pgg_algo_setup_exe(PggAlgo _algo, PggExe exe, PggErrenv errenv)
{
    PggErrenv		local_errenv;
    
    PGG_STD_ASSERT(PggAlgo, algo);
    PGG_ASSERT(algo, ARGUMENT, NULLPTR);
    
    pgg_errenv_reset(local_errenv);
    
    pgg_exe_add_arg(exe, "--openpgp", local_errenv);
    
    switch (algo->packet) {
        case PGG_PACKET_V3SIGS:
            pgg_exe_add_arg(exe, "--force-v3-sigs", local_errenv);
            break;
        
        case PGG_PACKET_RFC1991:
            pgg_exe_add_arg(exe, "--rfc1991", local_errenv);
            break;
        
        default:
            break;
    }
    
    switch (algo->compress) {
        case PGG_COMPRESS_NONE:
            pgg_exe_add_arg(exe, "--compress-algo", local_errenv);
            pgg_exe_add_arg(exe, "0", local_errenv);
            break;
        
        case PGG_COMPRESS_PGPZLIB:
            pgg_exe_add_arg(exe, "--compress-algo", local_errenv);
            pgg_exe_add_arg(exe, "1", local_errenv);
            break;
        
        case PGG_COMPRESS_RFCZLIB:
            pgg_exe_add_arg(exe, "--compress-algo", local_errenv);
            pgg_exe_add_arg(exe, "2", local_errenv);
            break;
        
        default:
            break;
    }
    
    if (algo->cipher > PGG_CIPHER_USEPREFS) {
        pgg_exe_add_arg(exe, "--cipher-algo", local_errenv);
        pgg_algo_build_cipher(algo->cipher, exe, local_errenv);
    }
    
    if (algo->digest > PGG_DIGEST_USEPREFS) {
        pgg_exe_add_arg(exe, "--digest-algo", local_errenv);
        pgg_algo_build_digest(algo->digest, exe, local_errenv);
    }
    
    pgg_exe_add_arg(exe, "--s2k-cipher-algo", local_errenv);
    pgg_algo_build_cipher(algo->s2k_cipher, exe, local_errenv);
    
    pgg_exe_add_arg(exe, "--s2k-digest-algo", local_errenv);
    pgg_algo_build_digest(algo->s2k_digest, exe, local_errenv);
    
    if (pgg_errenv_is_set(local_errenv)) {
        PGG_DEBUG(("problems on setting up exe"));
        pgg_errenv_copy(errenv, local_errenv);
    }
}


