/* encrypt.c - Demonstrate how to encrypt a file with public keys
 *      Copyright (c) 1999 by Michael Roth <mroth@gnupg.org>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
 * MICHAEL ROTH BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
 * IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 *
 * Except as contained in this notice, the name of Michael Roth shall not be
 * used in advertising or otherwise to promote the sale, use or other dealings
 * in this Software without prior written authorization from Michael Roth.
 */


#include <stdio.h>
#include <string.h>
#include <pgg.h>


int main(int argc, char *argv[])
{
    PggEncrypt		encrypt;
    PggStdio		stdio;
    PggInput		input;
    PggErrenv		errenv;
    int			i;
    
    if (argc <= 2) {
        fprintf(stderr, "Usage: encrypt FILE USERID1 [ USERID2 [ USERID3 [...]]]\n");
        return 1;
    }
    
    
    /*
     * As usual, a PggErrenv object carries errors and we must reset
     * it before use.
     */
    pgg_errenv_reset(errenv);
    
    
    /*
     * A PggStdio object carries the source and destination of data.
     * In this example we would like to produce a .gpg file, so we
     * don't create a PggOutput object later.
     */
    stdio = pgg_stdio_new(errenv);
    
    
    /*
     * To supply the input filename we must create a PggInput object
     * and set the filename. Then we attach this input object to the
     * PggStdio object.
     */
    input = pgg_input_new(errenv);
    pgg_input_set_filename(input, argv[1], errenv);
    pgg_stdio_set_input(stdio, input, errenv);
    
    
    /*
     * A PggEncrypt object is the workhorse for public key encryption.
     * We set the PggStdio object to define the source and the destination
     * of data. Because the PggStdio object hasn't a PggOutput object set
     * and the PggInput object is a filename, the output is written to
     * a .gpg file.
     */
    encrypt = pgg_encrypt_new(errenv);
    pgg_encrypt_set_stdio(encrypt, stdio, errenv);
    
    
    /*
     * Here we set the recipients of the file to encrypt. We add simply each
     * provided user-id to the PggEncrypt object. GnuPG will use its normal
     * algorithm to search the appropriate keys.
     */
    for (i=2; i<argc; ++i)
        pgg_encrypt_add_recipient(encrypt, argv[i], errenv);
    
    
    /*
     * This call will do all the magic. It could take a few seconds because
     * public key encryption is a heavy calculation process. When the call
     * returns and their aren't any errors, the result was written to a file
     * with a .gpg extension.
     */
    pgg_encrypt_execute(encrypt, errenv);
    
    
    /*
     * A simple error check. We don't print only a minimal message in the
     * case of an error.
     */
    if (pgg_errenv_is_set(errenv)) {
        fprintf(stderr, "An error occured\n");
        return 1;
    }
    
    
    /*
     * When the PggXxx object are no longer used, they should release to
     * avoid memory leaks.
     */
    pgg_stdio_release(stdio, NULL);
    pgg_input_release(input, NULL);
    pgg_encrypt_release(encrypt, NULL);
    
    return 0;
}



