package gnu.crypto;

// ----------------------------------------------------------------------------
// $Id: $
//
// Copyright (C) 2003 Free Software Foundation, Inc.
//
// This file is part of GNU Crypto.
//
// GNU Crypto is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// GNU Crypto is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to the
//
//    Free Software Foundation Inc.,
//    59 Temple Place - Suite 330,
//    Boston, MA 02111-1307
//    USA
//
// Linking this library statically or dynamically with other modules is
// making a combined work based on this library.  Thus, the terms and
// conditions of the GNU General Public License cover the whole
// combination.
//
// As a special exception, the copyright holders of this library give
// you permission to link this library with independent modules to
// produce an executable, regardless of the license terms of these
// independent modules, and to copy and distribute the resulting
// executable under terms of your choice, provided that you also meet,
// for each linked independent module, the terms and conditions of the
// license of that module.  An independent module is a module which is
// not derived from or based on this library.  If you modify this
// library, you may extend this exception to your version of the
// library, but you are not obligated to do so.  If you do not wish to
// do so, delete this exception statement from your version.
// ----------------------------------------------------------------------------

import java.util.HashMap;

/**
 * <p>A global object containing build-specific properties that affect the
 * behaviour of the generated binaries from this library.</p>
 *
 * @version $Revision: $
 */
public class Properties {

   // Constants and variables
   // -------------------------------------------------------------------------

   public static final String REPRODUCIBLE_PRNG = "gnu.crypto.with.reproducible.prng";
   public static final String CHECK_WEAK_KEYS = "gnu.crypto.with.check.for.weak.keys";

   private static final HashMap props = new HashMap();
   private static Properties singleton = null;

   private boolean reproducible = false;
   private boolean checkForWeakKeys = false;

   // Constructors.
   // -----------------------------------------------------------------------

   /** Trivial constructor to enforce Singleton pattern. */
   private Properties() {
      super();

      init();
   }

   // Class methods.
   // -----------------------------------------------------------------------

   public static synchronized final String getProperty(String key) {
      if (key == null) {
         return null;
      }
      return (String) props.get(key.trim());
   }

   public static synchronized final void setProperty(String key, String value) {
      if (key == null || value == null) {
         return;
      }
      key = key.trim();
      if (key.equals(REPRODUCIBLE_PRNG)) {
         setReproducible(Boolean.valueOf(value.trim()).booleanValue());
      } else if (key.equals(CHECK_WEAK_KEYS)) {
         setCheckForWeakKeys(Boolean.valueOf(value.trim()).booleanValue());
      } else {
         props.put(key, value);
      }
   }

   public static synchronized final boolean isReproducible() {
      return instance().reproducible;
   }

   public static synchronized final boolean checkForWeakKeys() {
      return instance().checkForWeakKeys;
   }

   public static synchronized final void setReproducible(boolean value) {
      instance().reproducible = value;
      props.put(REPRODUCIBLE_PRNG, new Boolean(value).toString());
   }

   public static synchronized final void setCheckForWeakKeys(boolean value) {
      instance().checkForWeakKeys = value;
      props.put(CHECK_WEAK_KEYS, new Boolean(value).toString());
   }

   private static synchronized final Properties instance() {
      if (singleton == null) {
         singleton = new Properties();
      }
      return singleton;
   }

   // Instance methods
   // -------------------------------------------------------------------------

   private void init() {
      // default values
      props.put(REPRODUCIBLE_PRNG, new Boolean(reproducible).toString());
      props.put(CHECK_WEAK_KEYS, new Boolean(checkForWeakKeys).toString());

      // TODO: allow override by, may be, reading a properties file

      // re-sync the 'known' properties
      reproducible =
            Boolean.valueOf((String) props.get(REPRODUCIBLE_PRNG)).booleanValue();
      checkForWeakKeys =
            Boolean.valueOf((String) props.get(CHECK_WEAK_KEYS)).booleanValue();
   }
}
