/*
 * Algebraic manipulator expression display routines.
 *
 * Copyright (c) 1996 George Gesslein II.
 */

#include "am.h"
#include "externs.h"

int	cur_color = -1;

/* ANSI color code array */
int	carray[] = {
	32, 33, 31, 35, 34, 36, 37
};

/* HTML color array */
char	*html_carray[] = {
	"green",
	"yellow",
	"orange",
	"red",
	"purple",
	"blue",
	"cyan"
};

reset_attr()
{
	cur_color = -1;
	if (html_flag) {
		printf("</font>");
	} else if (color_flag) {
		printf("\033[0m");
	}
}

set_color(color)
int	color;
{
	if (gfp != stdout)
		return;
	if (cur_color == color)
		return;
	if (html_flag) {
		if (color_flag)
			printf("</font><font color=\"%s\">", html_carray[color%ARR_CNT(html_carray)]);
	} else if (color_flag) {
		printf("\033[%dm", carray[color%ARR_CNT(carray)]);
	}
	cur_color = color;
}

default_color()
{
	set_color(0);
}

/*
 * Display the equation stored in equation space "n" in single line format.
 */
int
list1_sub(n)
int	n;
{
	if (n_lhs[n] <= 0)
		return false;
	if (!export_flag) {
		fprintf(gfp, _("#%d: "), n + 1);
	}
	list_proc(&lhs[n][0], n_lhs[n]);
	fprintf(gfp, _(" = "));
	list_proc(&rhs[n][0], n_rhs[n]);
	if (high_prec) {
#if	CYGWIN
		fprintf(gfp, "\r\n");
#else
		fprintf(gfp, "\n");
#endif
	} else {
		fprintf(gfp, "\n\n");
	}
	return true;
}

/*
 * Display the equation stored in equation space "n".
 */
int
list_sub(n)
int	n;
{
	if (n_lhs[n] <= 0)
		return false;
	if (groupall) {
		group_sub(n);
		flist_sub(n);
	} else {
		list1_sub(n);
	}
	return true;
}

list_tdebug(level)
int	level;
{
	if (debug_level >= level) {
		list_debug(level, &tlhs[0], n_tlhs, &trhs[0], n_trhs);
	}
}

list_debug(level, p1, n1, p2, n2)
int		level;
token_type	*p1;
int		n1;
token_type	*p2;
int		n2;
{
	if (debug_level >= level) {
		printf(_("debug %d: "), level);
		list_proc(p1, n1);
		printf(_(" = "));
		list_proc(p2, n2);
		printf("\n");
	}
}

side_debug(level, p1, n1)
int		level;
token_type	*p1;
int		n1;
{
	if (debug_level >= level) {
		printf(_("debug %d: "), level);
		list_proc(p1, n1);
		printf("\n");
	}
}

/*
 * Output a variable name in ASCII if "out_flag" is true.
 * Make it C language compatible if "c_flag" is true.
 * Return length of variable (number of ASCII characters).
 *
 * If "c_flag == 2", assume Java.
 */
int
list_var(v, out_flag, c_flag)
long	v;
int	out_flag;
int	c_flag;
{
	int		i, j;
	long		l1;

	var_str[0] = '\0';
	switch ((int) (v & VAR_MASK)) {
	case V_NULL:
		break;
	case SIGN:
		strcat(var_str, "sign");
		break;
	case IMAGINARY:
		if (export_flag)
			strcat(var_str, "%I");
		else
			strcat(var_str, "i#");
		break;
	case V_E:
		switch (c_flag) {
		case 0:
			strcat(var_str, "e#");
			break;
		case 1:
			strcat(var_str, "E");
			break;
		case 2:
			strcat(var_str, "Math.E");
			break;
		}
		break;
	case V_PI:
		switch (c_flag) {
		case 0:
			strcat(var_str, "pi");
			break;
		case 1:
			strcat(var_str, "PI");
			break;
		case 2:
			strcat(var_str, "Math.PI");
			break;
		}
		break;
	case MATCH_ANY:
		strcat(var_str, "all");
		break;
	case SPECIAL:
		strcat(var_str, "answer");
		break;
	case V_TEMP:
		strcat(var_str, "temp");
		break;
	case V_INTEGER:
		strcat(var_str, "integer");
		break;
	default:
		j = (v & VAR_MASK) - VAR_OFFSET;
		if (j < 0 || var_names[j] == NULL) {
			strcat(var_str, "bad_variable");
		} else {
			strcat(var_str, var_names[j]);
		}
		break;
	}
	j = (v >> VAR_SHIFT) & SUBSCRIPT_MASK;
	if (j) {
		j--;
		sprintf(&var_str[strlen(var_str)], "%d", j);
	}
	l1 = v;
	i = 0;
	while ((l1 -= PRIME_INCREMENT) >= 0) {
		if (!c_flag)
			strcat(var_str, "'");
		i++;
	}
	if (i && c_flag) {
		if (i == 1) {
			strcat(var_str, "_prime");
		} else {
			sprintf(&var_str[strlen(var_str)], "_%dprimes", i);
		}
	}
	if (v & PERCENT_CHANGE) {
		strcat(var_str, "_percent_change");
	}
	if (out_flag) {
		fprintf(gfp, "%s", var_str);
	}
	return(strlen(var_str));
}

/*
 * Display an expression in single line format.
 */
list_proc(equation, n)
token_type	*equation;
int		n;
{
	int	i,j,k,l;
	int	min1;
	int	cur_level;
	char	*cp;

	cur_level = min1 = min_level(equation, n);
	for (i = 0; i < n; i++) {
		j = cur_level - equation[i].level;
		k = abs(j);
		for (l = 1; l <= k; l++) {
			if (j > 0) {
				cur_level--;
				fprintf(gfp, ")");
				set_color(cur_level-min1);
			} else {
				cur_level++;
				set_color(cur_level-min1);
				fprintf(gfp, "(");
			}
		}
		switch (equation[i].kind) {
		case CONSTANT:
			if (high_prec) {
				fprintf(gfp, "%.20lg", equation[i].token.constant);
			} else {
				fprintf(gfp, "%.14lg", equation[i].token.constant);
			}
			break;
		case VARIABLE:
			list_var(equation[i].token.variable, true, export_flag);
			break;
		case OPERATOR:
			cp = _("(unknown operator)");
			switch (equation[i].token.operatr) {
			case PLUS:
				cp = " + ";
				break;
			case MINUS:
				cp = " - ";
				break;
			case TIMES:
				cp = "*";
				break;
			case DIVIDE:
				cp = "/";
				break;
			case MODULUS:
				cp = " % ";
				break;
			case POWER:
				cp = "^";
				break;
			case FACTORIAL:
				cp = "!";
				i++;
				break;
			}
			fprintf(gfp, "%s", cp);
			break;
		}
	}
	j = cur_level - min1;
	for (; j > 0;) {
		fprintf(gfp, ")");
		j--;
		set_color(j);
	}
}

/*
 * Return true if expression is a valid integer expression for
 * list_c_code().
 */
int
int_expr(equation, n)
token_type	*equation;
int		n;
{
	int	i;

	for (i = 0; i < n; i++) {
		switch (equation[i].kind) {
		case CONSTANT:
			if (fmod(equation[i].token.constant, 1.0)) {
				return false;
			}
			break;
		case OPERATOR:
			switch (equation[i].token.operatr) {
			case POWER:
				if (equation[i+1].level == equation[i].level
				    && equation[i-1].level == equation[i].level
				    && equation[i+1].kind == CONSTANT
				    && equation[i+1].token.constant == 2.0) {
					continue;
				}
				return false;
			}
			break;
		}
	}
	return true;
}

/*
 * Display C or Java code for an expression.
 */
list_c_code(equation, n, java_flag, int_flag)
token_type	*equation;
int		n;
int		java_flag;
int		int_flag;
{
	int	i,j,k,l;
	int	min1;
	int	cur_level;
	char	*cp;
	int	i1;
	char	buf[50];

	min1 = min_level(equation, n);
	if (n > 1)
		min1--;
	cur_level = min1;
	for (i = 0; i < n; i++) {
		j = cur_level - equation[i].level;
		k = abs(j);
		for (l = 1; l <= k; l++) {
			if (j > 0) {
				cur_level--;
				fprintf(gfp, ")");
			} else {
				cur_level++;
				for (i1 = i + 1; i1 < n && equation[i1].level >= cur_level; i1 += 2) {
					if (equation[i1].level == cur_level) {
						switch (equation[i1].token.operatr) {
						case POWER:
							if (equation[i1-1].level == cur_level
							    && equation[i1+1].level == cur_level
							    && equation[i1+1].kind == CONSTANT
							    && equation[i1+1].token.constant == 2.0) {
								equation[i1].token.operatr = TIMES;
								equation[i1+1].kind = equation[i1-1].kind;
								equation[i1+1].token = equation[i1-1].token;
							} else {
								if (java_flag) {
									fprintf(gfp, "Math.pow");
								} else {
									fprintf(gfp, "pow");
								}
							}
							break;
						case FACTORIAL:
							if (java_flag) {
								fprintf(gfp, "MathSupport.fact");
							} else {
								fprintf(gfp, "fact");
							}
							break;
						}
						break;
					}
				}
				fprintf(gfp, "(");
			}
		}
		switch (equation[i].kind) {
		case CONSTANT:
			if (int_flag) {
				fprintf(gfp, "%.0f", equation[i].token.constant);
			} else {
				sprintf(buf, "%#.14lg", equation[i].token.constant);
				j = strlen(buf) - 1;
				for (; j >= 0; j--) {
					if (buf[j] == '0')
						continue;
					if (buf[j] == '.')
						buf[j+2] = '\0';
					else
						break;
				}
				fprintf(gfp, "%s", buf);
			}
			break;
		case VARIABLE:
			list_var(equation[i].token.variable, true, (java_flag ? 2 : 1));
			break;
		case OPERATOR:
			cp = _("(unknown operator)");
			switch (equation[i].token.operatr) {
			case PLUS:
				cp = " + ";
				break;
			case MINUS:
				cp = " - ";
				break;
			case TIMES:
				cp = " * ";
				break;
			case DIVIDE:
				cp = " / ";
				break;
			case MODULUS:
				cp = " % ";
				break;
			case POWER:
				cp = ", ";
				break;
			case FACTORIAL:
				cp = "";
				i++;
				break;
			}
			fprintf(gfp, "%s", cp);
			break;
		}
	}
	j = cur_level - min1;
	for (; j > 0;) {
		fprintf(gfp, ")");
		j--;
	}
}

int	cur_line;
int	cur_pos;

/*
 * Display the equation at equation number "n" in fraction format.
 */
int
flist_sub(n)
int	n;
{
	int	i;
	int	sind;
	char	buf[30];
	int	len;
	int	len2, len3;
	int	pos;
	int	high, low;
	int	max_line;
	int	min_line;
	int	max2_line, min2_line;

	if (n_lhs[n] <= 0)
		return false;
	len = sprintf(buf, "#%d: ", n + 1);
	cur_line = 0;
	cur_pos = 0;
	sind = n_rhs[n];
	len += flist(&lhs[n][0], n_lhs[n], false, 0, &max_line, &min_line);
	len += 3;
make_smaller:
	len2 = flist(&rhs[n][0], sind, false, 0, &high, &low);
	if (screen_columns && (len + len2) >= screen_columns && sind > 0) {
		for (sind--; sind > 0; sind--) {
			if (rhs[n][sind].level == 1 && rhs[n][sind].kind == OPERATOR) {
				switch (rhs[n][sind].token.operatr) {
				case PLUS:
				case MINUS:
				case TIMES:
				case MODULUS:
				case POWER:
					goto make_smaller;
				}
			}
		}
		goto make_smaller;
	}
	if (high > max_line)
		max_line = high;
	if (low < min_line)
		min_line = low;
	len3 = flist(&rhs[n][sind], n_rhs[n] - sind, false, 0, &max2_line, &min2_line);
	if (screen_columns && max(len + len2, len3) >= screen_columns) {
		list1_sub(n);
		return true;
	}
	fprintf(gfp, "\n");
	for (i = max_line; i >= min_line; i--) {
		cur_line = i;
		cur_pos = 0;
		if (i == 0) {
			fprintf(gfp, "%s", buf);
			cur_pos = strlen(buf);
		}
		pos = strlen(buf);
		pos += flist(&lhs[n][0], n_lhs[n], true, pos, &high, &low);
		if (i == 0) {
			fprintf(gfp, "%s", " = ");
			cur_pos += 3;
		}
		pos += 3;
		flist(&rhs[n][0], sind, true, pos, &high, &low);
		fprintf(gfp, "\n");
	}
	if (sind < n_rhs[n]) {
		fprintf(gfp, "\n");
		for (i = max2_line; i >= min2_line; i--) {
			cur_line = i;
			cur_pos = 0;
			flist(&rhs[n][sind], n_rhs[n] - sind, true, 0, &high, &low);
			fprintf(gfp, "\n");
		}
	}
	fprintf(gfp, "\n");
	return true;
}

int
flist(equation, n, out_flag, pos, highp, lowp)
token_type	*equation;
int		n;
int		out_flag;
int		pos;
int		*highp, *lowp;
{
	int	i;

	i = flist_recurse(equation, n, out_flag, 0, pos, 1, highp, lowp);
	if (out_flag) {
		set_color(0);
	}
	return i;
}

int
flist_recurse(equation, n, out_flag, line, pos, cur_level, highp, lowp)
token_type	*equation;
int		n;
int		out_flag;
int		line;
int		pos;
int		cur_level;
int		*highp, *lowp;
{
	int	i,j,k,l;
	int	l1, l2;
	int	ii;
	int	stop_at;
	int	div_loc;
	int	len_div;
	int	level;
	int	start_level;
	int	oflag;
	int	len, len1, len2;
	int	high, low;
	char	buf[50];

	len = 0;
	start_level = cur_level;
	*highp = line;
	*lowp = line;
	if (n <= 0) {
		return 0;
	}
	oflag = (out_flag && line == cur_line);
	if (oflag) {
		set_color(cur_level-1);
		for (; cur_pos < pos; cur_pos++) {
			fprintf(gfp, " ");
		}
	}
	ii = 0;
check_again:
	stop_at = n;
	div_loc = -1;
	for (i = ii; i < n; i++) {
		if (equation[i].kind == OPERATOR && equation[i].token.operatr == DIVIDE) {
			level = equation[i].level;
			for (j = i - 2; j > 0; j -= 2) {
				if (equation[j].level < level)
					break;
			}
			j++;
			if (div_loc < 0) {
				div_loc = i;
				stop_at = j;
			} else {
				if (j < stop_at) {
					div_loc = i;
					stop_at = j;
				} else if (j == stop_at) {
					if (level < equation[div_loc].level)
						div_loc = i;
				}
			}
		}
	}
	for (i = ii; i < n; i++) {
		if (i == stop_at) {
			j = cur_level - equation[div_loc].level;
			k = abs(j) - 1;
		} else {
			j = cur_level - equation[i].level;
			k = abs(j);
		}
		for (l = 1; l <= k; l++) {
			if (j > 0) {
				cur_level--;
				len++;
				if (oflag) {
					fprintf(gfp, ")");
					set_color(cur_level-1);
				}
			} else {
				cur_level++;
				len++;
				if (oflag) {
					set_color(cur_level-1);
					fprintf(gfp, "(");
				}
			}
		}
		if (i == stop_at) {
			level = equation[div_loc].level;
			len1 = flist_recurse(&equation[stop_at], div_loc - stop_at, false, line + 1, pos + len, level, &high, &low);
			l1 = (2 * (line + 1)) - low;
			for (j = div_loc + 2; j < n; j += 2) {
				if (equation[j].level <= level)
					break;
			}
			len2 = flist_recurse(&equation[div_loc+1], j - (div_loc + 1), false, line - 1, pos + len, level, &high, &low);
			l2 = (2 * (line - 1)) - high;
			ii = j;
			len_div = max(len1, len2);
			j = 0;
			if (len1 < len_div) {
				j = (len_div - len1) / 2;
			}
			flist_recurse(&equation[stop_at], div_loc - stop_at, out_flag, l1, pos + len + j, level, &high, &low);
			if (high > *highp)
				*highp = high;
			if (low < *lowp)
				*lowp = low;
			if (oflag) {
				set_color(level-1);
				for (j = 0; j < len_div; j++) {
					fprintf(gfp, "-");
				}
				set_color(cur_level-1);
			}
			j = 0;
			if (len2 < len_div) {
				j = (len_div - len2) / 2;
			}
			flist_recurse(&equation[div_loc+1], ii - (div_loc + 1), out_flag, l2, pos + len + j, level, &high, &low);
			if (high > *highp)
				*highp = high;
			if (low < *lowp)
				*lowp = low;
			len += len_div;
			goto check_again;
		}
		switch (equation[i].kind) {
		case CONSTANT:
			len += sprintf(buf, "%.14lg", equation[i].token.constant);
			if (oflag)
				fprintf(gfp, "%s", buf);
			break;
		case VARIABLE:
			len += list_var(equation[i].token.variable, oflag, false);
			break;
		case OPERATOR:
			switch (equation[i].token.operatr) {
			case PLUS:
				strcpy(buf, " + ");
				break;
			case MINUS:
				strcpy(buf, " - ");
				break;
			case TIMES:
				strcpy(buf, "*");
				break;
			case MODULUS:
				strcpy(buf, " % ");
				break;
			case POWER:
				strcpy(buf, "^");
				break;
			case FACTORIAL:
				strcpy(buf, "!");
				i++;
				break;
			default:
				strcpy(buf, _("(unknown operator)"));
				break;
			}
			if (oflag)
				fprintf(gfp, "%s", buf);
			len += strlen(buf);
			break;
		}
	}
	j = cur_level - start_level;
	for (; j > 0;) {
		cur_level--;
		len++;
		j--;
		if (oflag) {
			fprintf(gfp, ")");
			set_color(cur_level-1);
		}
	}
	if (oflag)
		cur_pos += len;
	return len;
}
