/*
 * parse_opt_cfg.cpp
 *
 * Copyright (C) 2003 Ed Cogburn <ecogburn@xtn.net>
 *
 * This file is part of the program "d: The Directory Lister".
 *
 * "d: The Directory Lister" is free software; you can redistribute it and/or modify it under the terms of the GNU General Public
 * License as published by the Free Software Foundation; either version 2 of the License, or (at your option) any later version.
 *
 * "d: The Directory Lister" is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with "d: The Directory Lister" in a file named "COPYING".
 * If not, visit their website at http://www.gnu.org for a copy, or request a copy by writing to:
 *     Free Software Foundation, Inc.
 *     59 Temple Place, Suite 330
 *     Boston, MA 02111-1307 USA
*/



#include "globals.hpp"
#include <cerrno>
#include <cstdlib>
#include <cstring>



namespace poc
{



	opt_base** opt_base::options = NULL;

	int opt_base::opts_idx = 0;
	int opt_base::opts_sz = 0;

	//-----------------------------------------------------------------------------
	void opt_base::register_address(opt_base* obptr)
	{
		if (opt_base::options == NULL) {
			opt_base::options = new opt_base*[5];
			opt_base::opts_sz = 5;
		}

		if (opt_base::opts_idx == opt_base::opts_sz) {
			opt_base::opts_sz += 5;
			opt_base** newopts = new opt_base*[opt_base::opts_sz];
			for (int i = 0; i < opt_base::opts_idx; i++)
				newopts[i] = opt_base::options[i];
			delete [] opt_base::options;
			opt_base::options = newopts;
		}

		opt_base::options[opt_base::opts_idx] = obptr;
		opt_base::opts_idx++;
	}



	//-----------------------------------------------------------------------------
	int parse_opt_cfg::read_int_arg()
	{
		char *buf;
		char *tail;
		long sint;
		bool maybe_an_int;
		static const char valid_chars[] = "0123456789+-xX";

		stringstream err;
		stringstream errsubstr;

		errsubstr << '"';
		if (currently_reading == ConfigFile)
			errsubstr << opt << "\" in \"" << cf.name.name_ext() << "\" at line " << lineno;
		else {
			errsubstr << '-';
			if (current_option_size == Long)
				errsubstr << '-';
			errsubstr << opt;
		}

		if (arg.size() == 0) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " has a zero length argument!";
			throw runtime_error(err.str());
		}

		maybe_an_int = true;
		for (unsigned i = 0; i < arg.size(); i++) {
			bool is_valid = false;
			for (unsigned j = 0; j < sizeof(valid_chars); j++)
				if (arg[i] == valid_chars[j]) {
					is_valid = true;
					break;
				}
			if (is_valid == false) {
				maybe_an_int = false;
				break;
			}
		}

		if (maybe_an_int == false) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " requires an integer argument!";
			throw runtime_error(err.str());
		}

		buf = new char [arg.size() + 1];
		strcpy(buf, arg.c_str());
		errno = 0;
		sint = strtol(buf, &tail, 0);
		delete [] buf;

		if (errno || tail == buf) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " has an unreadable integer argument!";
			throw runtime_error(err.str());
		}

		return sint;
	}



	//-----------------------------------------------------------------------------
	double parse_opt_cfg::read_double_arg()
	{
		char *buf;
		char *tail;
		double sdbl;
		bool maybe_a_float;
		static const char valid_chars[] = "0123456789+-.eE";

		stringstream err;
		stringstream errsubstr;

		errsubstr << '"';
		if (currently_reading == ConfigFile)
			errsubstr << opt << "\" in \"" << cf.name.name_ext() << "\" at line " << lineno;
		else {
			errsubstr << '-';
			if (current_option_size == Long)
				errsubstr << '-';
			errsubstr << opt;
		}

		if (arg.size() == 0) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " has a zero length argument!";
			throw runtime_error(err.str());
		}

		maybe_a_float = true;
		for (unsigned i = 0; i < arg.size(); i++) {
			bool is_valid = false;
			for (unsigned j = 0; j < sizeof(valid_chars); j++)
				if (arg[i] == valid_chars[j]) {
					is_valid = true;
					break;
				}
			if (is_valid == false) {
				maybe_a_float = false;
				break;
			}
		}

		if (maybe_a_float == false) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " requires a floating point argument!";
			throw runtime_error(err.str());
		}

		buf = new char [arg.size() + 1];
		strcpy(buf, arg.c_str());
		errno = 0;
		sdbl = strtod(buf, &tail);
		delete [] buf;

		if (errno || tail == buf) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " has an unreadable floating point argument!";
			throw runtime_error(err.str());
		}

		return sdbl;
	}



	//-----------------------------------------------------------------------------
	string parse_opt_cfg::read_string_arg()
	{
		string val;
		stringstream err;
		stringstream errsubstr;

		errsubstr << '"';
		if (currently_reading == ConfigFile)
			errsubstr << opt << "\" in \"" << cf.name.name_ext() << "\" at line " << lineno;
		else {
			errsubstr << '-';
			if (current_option_size == Long)
				errsubstr << '-';
			errsubstr << opt;
		}

		if (arg.size() == 0) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " has a zero length argument!";
			throw runtime_error(err.str());
		}

		val = arg.c_str();
		return val;
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::trim_string(string& s)
	{
		while (s[0] == ' ' || s[0] == '\t')
			s.erase(0, 1);

		while (s[s.size() - 1] == ' ' || s[s.size() - 1] == '\t')
			s.erase(s.size() - 1, 1);
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::read_csv_strings_arg(vector<string>& vect)
	{
		stringstream err;
		stringstream errsubstr;

		errsubstr << '"';
		if (currently_reading == ConfigFile)
			errsubstr << opt << "\" in \"" << cf.name.name_ext() << "\" at line " << lineno;
		else {
			errsubstr << '-';
			if (current_option_size == Long)
				errsubstr << '-';
			errsubstr << opt;
		}

		if (arg.size() == 0) {
			err << "parse_opt_cfg():  option " << errsubstr.str() << " has a zero length argument!";
			throw runtime_error(err.str());
		}

		string s;
		unsigned idx;
		while (1) {
			idx = arg.find(",", 0);
			if (idx == arg.npos) {
				idx = arg.find("#", 0);
				if (idx != arg.npos)
					arg.erase(idx);
				if (arg != "")
					trim_string(arg);
				if (arg != "")
					vect.push_back(arg);
				break;
			}
			s = arg.substr(0, idx);
			arg.erase(0, idx + 1);
			if (s != "")
				trim_string(s);
			if (s == "")
				break;
			vect.push_back(s);
		}
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::validate_on_opt_type(stringstream& err, opt_base& ob)
	{
		opt_bool* obptr;
		opt_int* oiptr;
		opt_o_int* ooiptr;
		opt_dbl* odptr;
		opt_o_dbl* oodptr;
		opt_str* osptr;
		opt_o_str* oosptr;
		opt_v_str* ovsptr;
		opt_v_csv_str* ovcsvptr;

		vector<string> csv;

		switch (ob.OptType) {
			case Bool:
				obptr = reinterpret_cast<opt_bool*> (&ob);

				if (current_option_size == Short) {
					if (arg.size() == 0 || (arg.size() == 1 && arg[0] == '+')) {
						obptr->Set = true;
						return;
					}
					if (arg.size() == 1 && arg[0] == '-') {
						obptr->Set = false;
						return;
					}
					err << " is a boolean that can only take a '+' or '-' as an argument!";
					throw runtime_error(err.str());
				} else {
					if (arg == "") {
						obptr->Set = true;
						return;
					}
					if (arg != "true" && arg != "false" && arg != "t" && arg != "f" && arg != "1" && arg != "0") {
						err << " is a boolean that can only take one of [true/false/t/f/1/0] as an argument!";
						throw runtime_error(err.str());
					}
					if (arg == "true" || arg == "t" || arg == "1")
						obptr->Set = true;
					else
						obptr->Set = false;
				}
				return;

			case Int:
				if (arg == "") {
					err << " requires an integer argument!";
					throw runtime_error(err.str());
				}
				oiptr = reinterpret_cast<opt_int*> (&ob);
				if (currently_reading == ConfigFile)
					oiptr->LineNbr = lineno;
				else
					oiptr->LineNbr = -1;
				oiptr->Value = read_int_arg();
				return;

			case OInt:
				ooiptr = reinterpret_cast<opt_o_int*> (&ob);
				if (currently_reading == ConfigFile)
					ooiptr->LineNbr = lineno;
				else
					ooiptr->LineNbr = -1;
				ooiptr->Set = true;
				if (arg != "")
					ooiptr->Value = read_int_arg();
				return;

			case Dbl:
				if (arg == "") {
					err << " requires a floating point argument!";
					throw runtime_error(err.str());
				}
				odptr = reinterpret_cast<opt_dbl*> (&ob);
				if (currently_reading == ConfigFile)
					odptr->LineNbr = lineno;
				else
					odptr->LineNbr = -1;
				odptr->Value = read_double_arg();
				return;

			case ODbl:
				oodptr = reinterpret_cast<opt_o_dbl*> (&ob);
				if (currently_reading == ConfigFile)
					oodptr->LineNbr = lineno;
				else
					oodptr->LineNbr = -1;
				oodptr->Set = true;
				if (arg != "")
					oodptr->Value = read_double_arg();
				return;

			case Str:
				if (arg == "") {
					err << " requires a string argument!";
					throw runtime_error(err.str());
				}
				osptr = reinterpret_cast<opt_str*> (&ob);
				if (currently_reading == ConfigFile)
					osptr->LineNbr = lineno;
				else
					osptr->LineNbr = -1;
				osptr->Value = read_string_arg();
				return;

			case OStr:
				oosptr = reinterpret_cast<opt_o_str*> (&ob);
				if (currently_reading == ConfigFile)
					oosptr->LineNbr = lineno;
				else
					oosptr->LineNbr = -1;
				oosptr->Set = true;
				if (arg != "")
					oosptr->Value = read_string_arg();
				return;

			case VStr:
				if (arg == "") {
					err << " requires a string argument!";
					throw runtime_error(err.str());
				}
				ovsptr = reinterpret_cast<opt_v_str*> (&ob);
				if (currently_reading == ConfigFile)
					ovsptr->LineNbr.push_back(lineno);
				else
					ovsptr->LineNbr.push_back(-1);
				ovsptr->Value.push_back(read_string_arg());
				return;

			case VCSVStr:
				if (arg == "") {
					err << " requires a string argument!";
					throw runtime_error(err.str());
				}
				ovcsvptr = reinterpret_cast<opt_v_csv_str*> (&ob);
				if (currently_reading == ConfigFile)
					ovcsvptr->LineNbr.push_back(lineno);
				else
					ovcsvptr->LineNbr.push_back(-1);
				csv.clear();
				read_csv_strings_arg(csv);
				ovcsvptr->Value.push_back(csv);
				return;

			default:
				err << " has an invalid arg_type!";
				throw runtime_error(err.str());
		}
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::parse_long_option(string& src)
	{
		stringstream err;
		err << "parse_opt_cfg():  Invalid syntax in ";
		if (currently_reading == ConfigFile)
			err << "\"" << cf.name.name_ext() << "\" at line " << lineno << '!';
		else
			err << "\"--" << src << "\"!";

		current_option_size = Long;

		opt = "";
		arg = "";

		unsigned idx = src.find('=', 0);
		if (idx == src.npos) {
			unsigned idx2 = src.find(' ', idx);
			if (idx2 != src.npos) {
				err << "  Spaces in the expression!";
				throw runtime_error(err.str());
			} else {
				opt = src;
				for (int i = 0; opt_base::get_opt(i) != NULL; i++) {
					if (opt != opt_base::get_opt(i)->LongName)
						continue;

					validate_on_opt_type(err, *opt_base::get_opt(i));
					return;
				}
			}
		}

		if (idx == src.size() - 1) {
			err << "  An equals sign with nothing after it!";
			throw runtime_error(err.str());
		}

		unsigned idx2 = src.find('=', idx + 1);
		if (idx2 != src.npos) {
			err << "  More than one equals sign on the line!";
			throw runtime_error(err.str());
		}

		opt = src.substr(0, idx);
		arg = src.substr(idx + 1);

		while (opt[opt.size() - 1] == ' ' || opt[opt.size() - 1] == '\t')
			opt.erase(opt.size() - 1);

		while (arg[0] == ' ' || arg[0] == '\t')
			arg.erase(0, 1);

		for (int i = 0; opt_base::get_opt(i) != NULL; i++) {
			if (opt != opt_base::get_opt(i)->LongName)
				continue;

			validate_on_opt_type(err, *opt_base::get_opt(i));
			return;
		}

		err << "  Unrecognized option!";
		throw runtime_error(err.str());
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::parse_short_option(string& src)
	{
		stringstream err;
		err << "parse_opt_cfg():  Invalid syntax in \"-" << src << "\"!";

		current_option_size = Short;

		opt = src[0];

		arg = "";

		if (src.size() > 1) {
			arg = src.substr(1);

			if (arg[0] == '=') {
				if (arg.size() == 1) {
					err << "  An equals sign with nothing after it!";
					throw runtime_error(err.str());
				}
				arg.erase(0, 1);
			}

			unsigned idx = arg.find(' ', 0);
			if (idx != src.npos) {
				err << "  Spaces in the expression!";
				throw runtime_error(err.str());
			}
		}

		for (int i = 0; opt_base::get_opt(i) != NULL; i++) {
			if (opt[0] != opt_base::get_opt(i)->ShortName)
				continue;

			validate_on_opt_type(err, *opt_base::get_opt(i));
			return;
		}

		err << "  Unrecognized option!";
		throw runtime_error(err.str());
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::clear_vectors()
	{
		opt_v_str* ovsptr;
		opt_v_csv_str* ovcsvptr;

		if (opt_base::get_opt(0) == NULL)
			throw logic_error("parse_opt_cfg():  No options registered!");

		for (int i = 0; opt_base::get_opt(i) != NULL; i++) {
			switch (opt_base::get_opt(i)->OptType) {
				case VStr:
					ovsptr = reinterpret_cast<opt_v_str*> (opt_base::get_opt(i));
					ovsptr->LineNbr.clear();
					ovsptr->Value.clear();
					break;

				case VCSVStr:
					ovcsvptr = reinterpret_cast<opt_v_csv_str*> (opt_base::get_opt(i));
					ovcsvptr->LineNbr.clear();
					ovcsvptr->Value.clear();
					break;

				default:
					continue;
			}
		}
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::parse_cloptions(int argc, char** argv, vector<string>& non_opt_arg_array)
	{
		string src;

		if (argc < 2)
			return;

		currently_reading = CLOptions;

		clear_vectors();

		for (int i = 1; i < argc; i++) {
			src = argv[i];

			if (src.size() == 0)
				throw runtime_error("parse_opt_cfg():  Input is a zero length option!");

			if (src[0] != '-') {
				non_opt_arg_array.push_back(src);
				continue;
			}

			if (src.size() == 1)
				throw runtime_error("parse_opt_cfg():  Input is a dash with no option after it!");

			if (src[1] == '-') {
				if (src.size() == 2)
					throw runtime_error("parse_opt_cfg():  Input is dashes with no option after them!");
				else {
					src.erase(0, 2);
					parse_long_option(src);
				}
			} else {
				src.erase(0, 1);
				parse_short_option(src);
			}
		}
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::parse_configfile_read()
	{
		char buf[256];
		char* ptr = buf;
		string src;

		ifstream ifile(cf.name.full().c_str());

		while (ifile.good()) {
			ifile.getline(buf, 256, '\n');

			if (ifile.bad()) {
				stringstream ss;
				ss << "parse_opt_cfg():  Can't read \"" << cf.name.name_ext() << "\" at line " << lineno << '!';
				throw runtime_error(ss.str());
			}

			lineno++;

			while (*ptr == ' ' || *ptr == '\t')
				ptr++;

			if (*ptr == '#' || *ptr == '\0')
				continue;

			src = ptr;

			unsigned idx = src.find('#', 0);
			if (idx != src.npos)
				src.erase(idx);

			while (src[src.size() - 1] == ' ' || src[src.size() - 1] == '\t')
				src.erase(src.size() - 1);

			for (unsigned i = 0; i < src.size(); i++)
				if (src[i] >= 'A' && src[i] <= 'Z')
					src[i] += 32;

			parse_long_option(src);
		}
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::parse_configfile(char* file)
	{
		cf = file;

		currently_reading = ConfigFile;

		clear_vectors();

		lineno = 0;

		if (cf.exists)
			parse_configfile_read();
	}



	//-----------------------------------------------------------------------------
	void parse_opt_cfg::parse_configfile(string& file)
	{
		cf = file;

		currently_reading = ConfigFile;

		clear_vectors();

		lineno = 0;

		if (cf.exists)
			parse_configfile_read();
	}



} /* namespace poc */
