/***********************************************************************
 *
 os-interface.c

Defines the OS dependent function 
***********************************************************************/

#ifdef __CYGWIN__
/***********************************************************************
 *
 Windows interface code

***********************************************************************/
#include "resolver.h"

/* Conflict between Windows definitions and others */
#undef ERROR
#undef NOERROR
#undef DELETE

#include <windows.h>
#include <iphlpapi.h>

/***********************************************************************
 *
 get_registry_dnsaddress: returns dns server addresses from the registry
 
 kHey: Handle to registry key
 KeyValue: key value to read
 SockAddrPtr: arrays of struct in_addr, to be filled.
 MaxNum: maximum number of servers
 
 Returns the number of servers.
 
***********************************************************************/
static int get_registry_dns_address(HKEY hKey, LPCTSTR KeyValue,
				    struct sockaddr_in * SockAddrPtr, 
				    int MaxNum, int debug) 
{
  DWORD size;
  LONG res;
  LPBYTE list;
  
  DPRINTF(debug, "value %s\n", KeyValue);
  if ((res = RegQueryValueEx( hKey, KeyValue, NULL, NULL, NULL, &size ))
      != ERROR_SUCCESS) { 
    DPRINTF(debug, "RegQueryValueEx (size): error %ld (Windows)\n", res);
  }
  else if (size > 1) {
    if (!(list = (LPBYTE) alloca(size))) {
      DPRINTF(debug, "alloca: %s\n", strerror(errno));
    }
    else if ((res = RegQueryValueEx( hKey, KeyValue, NULL, NULL, list,
				     &size )) != ERROR_SUCCESS) {
      DPRINTF(debug, "RegQueryValueEx: error %ld (Windows)\n", res);
    }
    else {
      BYTE *ap, *srch;
      int numAddresses = 0;
      for (ap = list; ap < list + size && *ap; ap = srch) {
        /* The separation character can be 0, ' ', or ','. */
	for (srch = ap; *srch && (isdigit(*srch) || *srch == '.' ); srch++);
	*srch++ = 0;
	if (numAddresses < MaxNum) {
	  DPRINTF(debug, "server \"%s\"\n", ap);
	  SockAddrPtr[numAddresses].sin_addr.s_addr = inet_addr(ap);
	  if ( SockAddrPtr[numAddresses].sin_addr.s_addr != 0 ) 
	    numAddresses++;
	}
	else 
	  DPRINTF(debug, "no space for server \"%s\"\n", ap);
      }
      return numAddresses;
    }
  }
  return 0;
}

/***********************************************************************
 *
 get_registry_dns_addresses: Read the registry to get dns server addresses 
                   in Network Byte Order.
 
 SockAddrPtr: arrays of struct in_addr, to be filled.
 MaxNum: maximum number of servers
 
 Returned value: actual number of servers, 
 
 This code is only meant for Win95 and NT <= 4.0, but not DHCP on Win95
***********************************************************************/

static int get_registry_dns_addresses(struct sockaddr_in * SockAddrPtr, 
				      int MaxNum, int debug)
{
  HKEY hKey;     
  DWORD  res;
  int num;
  const char *keyName[] = {"SYSTEM\\CurrentControlSet\\Services\\Tcpip\\Parameters",
			   "System\\CurrentControlSet\\Services\\VxD\\MSTCP"};
  int is9x = !!(GetVersion() & 0x80000000);
  
  DPRINTF(debug, "key %s\n", keyName[is9x]);
  if ((res = RegOpenKeyEx( HKEY_LOCAL_MACHINE, keyName[is9x], 0, 
			   KEY_QUERY_VALUE | KEY_READ, &hKey)) != ERROR_SUCCESS) {
    DPRINTF(debug, "RegOpenKeyEx: error %ld (Windows)\n", res);
    return 0;
  }

  num = get_registry_dns_address(hKey, "NameServer", SockAddrPtr, MaxNum, debug);
  if (num == 0 && !is9x) 
    num = get_registry_dns_address(hKey, "DhcpNameServer", SockAddrPtr, MaxNum, debug);

  RegCloseKey(hKey);

  return num;
}

/***********************************************************************
 *
 get_dns_info: returns the domain name and dns server addresses in Network Byte Order
 
 DomNamePPtr: set to malloced Domain in which the local computer is 
              registered, if non empty.
 SockAddrPtr: arrays of struct in_addr, to be filled.
 MaxNum: maximum number of servers
 
 Returned value: actual number of servers, 
 
***********************************************************************/
int get_dns_info(char ** DomNamePPtr, struct sockaddr_in * SockAddrPtr, int MaxNum, int debug)
{
#if MAX_HOSTNAME_LEN > MAXHOSTNAMELEN
#define MAX_HOSTNAME_SIZE (MAX_HOSTNAME_LEN + 1)
#else
#define MAX_HOSTNAME_SIZE (MAXHOSTNAMELEN + 1)
#endif
  int res;
  char domain[MAX_HOSTNAME_SIZE] = "";

  if (!(res = getdomainname(domain, sizeof(domain)))) {
    domain[sizeof(domain) - 1] = 0;
    if (domain[0])
      *DomNamePPtr = make_string(domain, -1, debug);
  }
  DPRINTF(debug, "getdomainname \"%s\"\n", (res)?strerror(errno):domain);

  ULONG ulOutBufLen = 0;
  DWORD dwRetVal;
  IP_ADDR_STRING * pIPAddr;
  FIXED_INFO * pFixedInfo;
  HINSTANCE kerneldll;
  typedef DWORD WINAPI (*GNPType)(PFIXED_INFO, PULONG);
  GNPType PGetNetworkParams;
  int numAddresses = 0;

  if (!(kerneldll = LoadLibrary("IPHLPAPI.DLL"))) {
    DPRINTF(debug, "LoadLibrary: error %ld (Windows)\n", GetLastError());
    goto use_registry;
  }
  if (!(PGetNetworkParams = (GNPType) GetProcAddress(kerneldll, 
						     "GetNetworkParams"))) {
    DPRINTF(debug, "GetProcAddress: error %ld (Windows)\n", GetLastError());
    goto use_registry;
  }
  /* First call to get the buffer length we need */
  dwRetVal = PGetNetworkParams((FIXED_INFO *) 0, &ulOutBufLen);
  if (dwRetVal != ERROR_BUFFER_OVERFLOW) {
    DPRINTF(debug, "GetNetworkParams: error %ld (Windows)\n", dwRetVal);
    goto use_registry;
  }
  if ((pFixedInfo = (FIXED_INFO *) alloca(ulOutBufLen)) == 0) {
    DPRINTF(debug, "alloca: %s\n", strerror(errno));
    return 0;
  }
  if ((dwRetVal = PGetNetworkParams((FIXED_INFO *) pFixedInfo, & ulOutBufLen))) {
    DPRINTF(debug, "GetNetworkParams: error %ld (Windows)\n", dwRetVal);
    return 0;
  }

  /* Record up to MaxNum dns server addresses */
  for (pIPAddr = &(pFixedInfo->DnsServerList), numAddresses = 0;
       pIPAddr;
       pIPAddr = pIPAddr->Next) {
    if (numAddresses < MaxNum) {
      DPRINTF(debug, "server \"%s\"\n", pIPAddr->IpAddress.String);
      SockAddrPtr[numAddresses].sin_addr.s_addr = inet_addr(pIPAddr->IpAddress.String);
      if (SockAddrPtr[numAddresses].sin_addr.s_addr != 0) 
	numAddresses++;
    }
    else 
      DPRINTF(debug, "no space for server \"%s\"\n", pIPAddr->IpAddress.String);
  }
  return numAddresses;

 use_registry:
  return get_registry_dns_addresses(SockAddrPtr, MaxNum, debug);
}

#else
/***********************************************************************
 *
 Default interface code

***********************************************************************/

int get_dns_info(char ** DomNamePPtr, struct sockaddr_in * SockAddrPtr, int MaxNum, int debug)
{
    return 0;
}

#endif
